import 'package:flutter/material.dart';
import 'package:outline_material_icons/outline_material_icons.dart';
import 'package:spartathlon_app/AppStateContainer.dart';
import 'package:spartathlon_app/models/Cp.dart';
import 'package:spartathlon_app/providers/CpData.dart';
import 'package:spartathlon_app/util/DistanceUnit.dart';
import 'package:spartathlon_app/util/Utils.dart';
import 'package:spartathlon_app/SpartaLocalizations.dart';

/// Provides list of CPs
///
/// To that end, it parses the JSON in CpData.dart and builds Cp objects from it.

class CpProvider {
  /// Returns a list of 76 Cp objects, each fitted with the data of one Cp
  static List<Cp> getCps(BuildContext context) {
    return checkpoints
        .map((Map<String, Object> e) => Cp(
              id: e['id'],
              name: e['name'],
              fromStart: double.parse(e['fromStart'].toString()),
              toNext: double.parse(e['toNext'].toString()),
              toFinish: double.parse(e['toFinish'].toString()),
              openingTime: e['openingTime'].toString().substring(0, 5),
              closingTime: e['closingTime'].toString().substring(0, 5),
              liveTiming: (e['liveTiming'] == 'yes'),
              supporterAllowed: (e['supporterAllowed'] == 'yes'),
              firstAid: (e['firstAid'] == 'yes'),
              long: e['long'],
              lat: e['lat'],
              supplies: e['supplies'],
              description: e['description'],
            ))
        .toList();
  }

  /// Returns a widget that will be visible as a body for the expanded ExpansionTile
  static Widget getCpExpansionTileBody(Cp cp, BuildContext context) {
    DistanceUnit distanceUnit = AppStateContainer.of(context).data.distanceUnit;

    return Container(
      padding: EdgeInsets.only(left: 16.0, right: 16.0, top: 0.0, bottom: 16.0),
      child: Column(
        children: [
          Row(
            // Top row with distances
            mainAxisAlignment: MainAxisAlignment.spaceEvenly,
            children: [
              Row(
                children: [
                  Padding(
                    padding: EdgeInsets.only(right: 4.0),
                    child: Icon(Icons.account_balance),
                  ),
                  Text(getDistanceString(context, cp.fromStart, distanceUnit)),
                ],
              ),
              Row(
                children: [
                  Padding(
                    padding: EdgeInsets.only(right: 4.0),
                    child: Icon(Icons.redo),
                  ),
                  Text(getDistanceString(context, cp.toNext, distanceUnit)),
                ],
              ),
              Row(
                children: [
                  Padding(
                    padding: EdgeInsets.only(right: 4.0),
                    child: Icon(Icons.flag),
                  ),
                  Text(getDistanceString(context, cp.toFinish, distanceUnit)),
                ],
              ),
            ],
          ),
          Padding(
            padding: EdgeInsets.all(8.0),
            child: Row(
              mainAxisAlignment: MainAxisAlignment.spaceEvenly,
              children: [
                Row(
                  children: [
                    Padding(
                      padding: EdgeInsets.only(right: 4.0),
                      child: Icon(Icons.access_time),
                    ),
                    Text(cp.openingTime + '﹣' + cp.closingTime),
                  ],
                ),
              ],
            ),
          ),
          Row(
            children: [
              Expanded(
                child: Row(
                  mainAxisAlignment: MainAxisAlignment.start,
                  children: [
                    Padding(
                      padding: EdgeInsets.only(right: 16.0),
                      child: Icon(OMIcons.localDrink),
                    ),
                    Flexible(
                      child: Text(
                        cp.supplies,
                        maxLines: 100,
                        overflow: TextOverflow.ellipsis,
                        softWrap: true,
                      ),
                    ),
                  ],
                ),
              ),
            ],
          ),
          Padding(
            padding: EdgeInsets.only(top: 8.0),
            child: Row(
              mainAxisAlignment: MainAxisAlignment.end,
              children: [
                Padding(
                  padding: EdgeInsets.only(right: 16.0),
                  child: Icon(Icons.info_outline),
                ),
                Expanded(
                  child: Padding(
                    padding: EdgeInsets.only(right: 4.0),
                    child: Text(
                      cp.description,
                      maxLines: 25,
                      overflow: TextOverflow.ellipsis,
                      softWrap: true,
                    ),
                  ),
                ),
                RaisedButton.icon(
                  icon: Icon(Icons.directions),
                  label: Text(SpartaLocalizations.of(context).labelButtonRoute),
                  color: Theme.of(context).primaryColor,
                  textColor: Colors.white,
                  shape: RoundedRectangleBorder(
                    borderRadius: BorderRadius.circular(buttonCornerRadius),
                  ),
                  onPressed: () {
                    launchUrl('geo:' +
                        cp.lat.toString() +
                        ',' +
                        cp.long.toString() +
                        '?q=' +
                        cp.lat.toString() +
                        ',' +
                        cp.long.toString());
                  },
                ),
              ],
            ),
          ),
        ],
      ),
    );
  }

  /// Returns the "general" name of the place where the checkpoint is located
  /// using a lookup in a hardcoded map.
  static String getCpLocation(int cpId) {
    // Note: 13 characters max, anything else will be cut off until we dynamically
    // resize the FacAthleteCard
    Map<int, String> locationMap = {
      0: 'Athens',
      4: 'Aspropyrgos',
      11: 'Megara',
      22: 'Corinth',
      35: 'Ancient Nemea',
      43: 'Lyrkia',
      47: 'Mountain Base',
      52: 'Nestani',
      60: 'Alea Tegea',
      69: null, // End of steep ascent, hurts a lot
      75: 'Sparta',
    };

    String s = locationMap[cpId];
    if (s == null) {
      return '';
    } else {
      return ' (' + s + ')';
    }
  }
}
