import 'package:flutter/material.dart';
import 'package:shared_preferences/shared_preferences.dart';
import 'package:spartathlon_app/SpartaLocalizations.dart';

enum DistanceUnit { km, miles }

const String SETTING_DISTANCE = 'distanceUnit';

/// Returns a human-readable string representation of [unit]
String distanceUnitToString(BuildContext context, DistanceUnit unit) {
  switch (unit) {
    case DistanceUnit.km:
      return SpartaLocalizations.of(context).distanceUnitKm;
    case DistanceUnit.miles:
      return SpartaLocalizations.of(context).distanceUnitMiles;
    default:
      return 'Unknown';
  }
}

/// Get the DistanceUnit preference from disk
Future<DistanceUnit> getDistanceUnit() async {
  SharedPreferences prefs = await SharedPreferences.getInstance();
  int unitId = prefs.getInt(SETTING_DISTANCE);
  // Make sure setting exists
  while (unitId == null) {
    await setDistanceToPrefs(DistanceUnit.km);
    unitId = prefs.getInt(SETTING_DISTANCE);
  }
  return DistanceUnit.values[unitId];
}

/// Stores [unit] to the preferences on disk
Future<void> setDistanceToPrefs(DistanceUnit unit) async {
  SharedPreferences prefs = await SharedPreferences.getInstance();
  print('Setting distance unit to ' + unit.toString());
  await prefs.setInt(SETTING_DISTANCE, unit.index);
}

/// Returns a human-readable string from a [distanceInKm] and a [unit]
/// Values are rounded to one decimal place
String getDistanceString(
    BuildContext context, double distanceInKm, DistanceUnit unit) {
  double distanceInUnit;
  switch (unit) {
    case DistanceUnit.km:
      distanceInUnit = distanceInKm;
      break;
    case DistanceUnit.miles:
      distanceInUnit = 0.6213712 * distanceInKm;
      break;
    default:
      distanceInUnit = distanceInKm;
  }
  // Round to one decimal place
  return distanceInUnit.toStringAsFixed(1) +
      ' ' +
      distanceUnitToString(context, unit);
}
