import 'package:flutter/material.dart';
import 'package:flutter/foundation.dart';

import 'package:shared_preferences/shared_preferences.dart';

/// Defines the app's themes: light (default) and dark.
/// We might define a black (OLED) theme in the future as well

enum AppTheme { THEME_LIGHT, THEME_DARK }
const String SETTING_THEME = 'theme';

/// Returns the ThemeData object corresponding to the given enum AppTheme
ThemeData getThemeData(AppTheme theme) {
  switch (theme) {
    case AppTheme.THEME_DARK:
      return darkTheme;
    default:
      return lightTheme;
  }
}

/// Asynchronously returns the id of the theme currently stored in the shared prefs.
/// We assume that this is consistent with the one Flutter currently uses to draw the widgets (this should be made sure elsewhere!).
Future<AppTheme> getThemeFromPrefs() async {
  SharedPreferences prefs = await SharedPreferences.getInstance();
  int themeId = prefs.getInt(SETTING_THEME);
  // Make sure theme setting exists
  while (themeId == null) {
    await setThemeIdToPrefs(AppTheme.THEME_LIGHT);
    themeId = prefs.getInt(SETTING_THEME);
  }
  return AppTheme.values[themeId];
}

/// Stores new app theme in shared preferences to persist across restarts.
/// We use an int instead of a boolean since we might want to have a black/OLED theme with value 2 in the future.
///
setThemeIdToPrefs(AppTheme theme) async {
  SharedPreferences prefs = await SharedPreferences.getInstance();
  print('Setting app theme to ' +
      theme.toString() +
      ' with theme ID: ' +
      theme.index.toString());
  await prefs.setInt(SETTING_THEME, theme.index);
}

/// Custom colors
const Color spartaBlue = Color(0xFF005596);
const Color spartaBlueDark = Color(0xFF2962FF);
const Color spartaYellowDark = Color(0xFFFFC629);

/// Light theme
final ThemeData lightTheme = ThemeData(
  brightness: Brightness.light,
  primaryColor: spartaBlue,
  primaryColorLight: Colors.white,
  accentColor: spartaBlue,
  canvasColor: Colors.grey[50],
  cardColor: Colors.white,
  dialogBackgroundColor: Colors.white,
  scaffoldBackgroundColor: Color(0xFFF5F5F5),
  textTheme: Typography(platform: defaultTargetPlatform).black,
  textSelectionHandleColor: Colors.black54, // workaround for text themes
  toggleableActiveColor: spartaBlue,
  iconTheme: IconThemeData(color: spartaBlue),
  dividerColor: Color(0x1F000000),
);

/// Dark theme
final ThemeData darkTheme = ThemeData(
  brightness: Brightness.dark,
  primaryColor: spartaBlueDark,
  primaryColorLight: spartaBlueDark,
  accentColor: spartaBlueDark,
  canvasColor: Colors.grey[850],
  cardColor: Colors.grey[800],
  dialogBackgroundColor: Colors.grey[800],
  scaffoldBackgroundColor: Colors.grey[850],
  textTheme: Typography(platform: defaultTargetPlatform).white,
  textSelectionHandleColor: Colors.grey[300], // workaround for text themes
  toggleableActiveColor: spartaYellowDark,
  iconTheme: IconThemeData(color: spartaBlueDark),
  dividerColor: Color(0x1FFFFFFF),
);
