import 'package:flutter/material.dart';
import 'package:outline_material_icons/outline_material_icons.dart';
import 'package:spartathlon_app/AppStateContainer.dart';
import 'package:spartathlon_app/SpartaLocalizations.dart';
import 'package:spartathlon_app/models/Cp.dart';
import 'package:spartathlon_app/providers/CpProvider.dart';
import 'package:spartathlon_app/util/Utils.dart';

/// Course View
///
/// Lists all checkpoints (aka CPs in Spartathlon slang)

enum CpFilterOption { All, Supporters, Live }

class CourseView extends StatefulWidget {
  CourseView({Key key}) : super(key: key);

  @override
  _CourseViewState createState() => _CourseViewState();
}

class _CourseViewState extends State<CourseView> {
  List<Cp> cps;
  List<Cp> cpsToDisplay;

  @override
  Widget build(BuildContext context) {
    if (cps == null || cps.isEmpty) {
      cps = CpProvider.getCps(context);
      _filterCpList(CpFilterOption.All);
    }

    double checkpointsFilterBarHeight = 45.0;
    double courseListHeight = MediaQuery.of(context).size.height -
        checkpointsFilterBarHeight -
        kBottomNavBarHeight;
    // Account for yellow notice bar displayed when no network connection
    if (!AppStateContainer.of(context).data.hasConnection) {
      courseListHeight -= kNoNetworkBarHeight;
    }

    return Column(
      children: [
        Container(
          height: checkpointsFilterBarHeight,
          padding: EdgeInsets.all(10.0),
          child: Row(
            children: [
              Expanded(
                child: Text(
                  SpartaLocalizations.of(context).titleCheckpoints,
                  style: TextStyle(
                    fontSize: 16.0,
                    fontWeight: FontWeight.bold,
                    color: Theme.of(context).textSelectionHandleColor,
                  ),
                ),
              ),
              DropdownButtonHideUnderline(
                child: DropdownButton(
                  hint: Text(
                    SpartaLocalizations.of(context)
                        .labelButtonFilterCheckpoints,
                    style: TextStyle(
                        color: Theme.of(context).textSelectionHandleColor),
                  ),
                  items: [
                    CpFilterOption.All,
                    CpFilterOption.Supporters,
                    CpFilterOption.Live
                  ]
                      .map((CpFilterOption o) => DropdownMenuItem(
                            value: o,
                            child: Text(_cpFilterOptionToString(o)),
                          ))
                      .toList(),
                  onChanged: ((selection) => _filterCpList(selection)),
                ),
              ),
            ],
          ),
        ),
        Divider(height: 0.0),
        Divider(height: 0.0),
        Container(
          height: courseListHeight,
          child: ListView.builder(
            itemCount: cpsToDisplay.length,
            itemBuilder: ((BuildContext context, int position) {
              return ExpansionTile(
                key: ValueKey(cpsToDisplay[position].id),
                leading: Padding(
                    padding: EdgeInsets.only(top: 4.0),
                    child: Text('CP ' + cpsToDisplay[position].id.toString())),
                title: _getPanelTitle(cpsToDisplay[position]),
                initiallyExpanded: false,
                children: [
                  CpProvider.getCpExpansionTileBody(
                      cpsToDisplay[position], context)
                ],
              );
            }),
          ),
        ),
      ],
    );
  }

  Widget _getPanelTitle(Cp cp) {
    List<Widget> iconList = [];
    if (cp.firstAid) {
      iconList.add(Padding(
        padding: EdgeInsets.only(left: 4.0),
        child: Icon(
          OMIcons.localHospital,
          color: Theme.of(context).iconTheme.color,
        ),
      ));
    }
    if (cp.liveTiming) {
      iconList.add(Padding(
          padding: EdgeInsets.only(left: 4.0),
          child: Icon(
            OMIcons.insertChart,
            color: Theme.of(context).iconTheme.color,
          )));
    }
    if (cp.supporterAllowed) {
      iconList.add(Padding(
          padding: EdgeInsets.only(left: 4.0),
          child: Icon(
            OMIcons.group,
            color: Theme.of(context).iconTheme.color,
          )));
    }

    return Container(
      child: Row(
        children: [
          Expanded(
            child: Text(cp.name),
          ),
          Row(
            children: iconList,
          ),
        ],
      ),
    );
  }

  /// Filters through the list of all CPs and only adds those CPs to the display
  /// that satisfy a certain property
  void _filterCpList(CpFilterOption option) {
    setState(() {
      switch (option) {
        case CpFilterOption.Supporters:
          cpsToDisplay = List<Cp>();
          cps.forEach((Cp cp) {
            if (cp.supporterAllowed) {
              cpsToDisplay.add(cp);
            }
          });
          break;
        case CpFilterOption.Live:
          cpsToDisplay = List<Cp>();
          cps.forEach((Cp cp) {
            if (cp.liveTiming) {
              cpsToDisplay.add(cp);
            }
          });
          break;
        default:
          cpsToDisplay = cps;
          break;
      }
    });
  }

  String _cpFilterOptionToString(CpFilterOption option) {
    switch (option) {
      case CpFilterOption.Supporters:
        return SpartaLocalizations.of(context).caseCpFilterSupporters;
      case CpFilterOption.Live:
        return SpartaLocalizations.of(context).caseCpFilterLiveTiming;
      default:
        return SpartaLocalizations.of(context).caseCpFilterAll;
    }
  }
}
