import 'package:flutter/cupertino.dart';
import 'package:flutter/material.dart';
import 'package:outline_material_icons/outline_material_icons.dart';
import 'package:spartathlon_app/AppStateContainer.dart';
import 'package:spartathlon_app/SpartaLocalizations.dart';
import 'package:spartathlon_app/util/DistanceUnit.dart';
import 'package:spartathlon_app/util/Themes.dart';
import 'package:spartathlon_app/views/AboutView.dart';

/// Settings view

class SettingsView extends StatefulWidget {
  SettingsView({Key key}) : super(key: key);

  @override
  _SettingsViewState createState() => _SettingsViewState();
}

class _SettingsViewState extends State<SettingsView> {
  DistanceUnit _currentUnit;

  @override
  void initState() {
    super.initState();
  }

  @override
  Widget build(BuildContext context) {
    _currentUnit = AppStateContainer.of(context).data.distanceUnit;

    return Scaffold(
      appBar: AppBar(
        title: Text(SpartaLocalizations.of(context).appBarTitleSettings),
        backgroundColor: Theme.of(context).primaryColor,
      ),
      body: FutureBuilder<AppTheme>(
          future: getThemeFromPrefs(),
          builder: (BuildContext context, AsyncSnapshot<AppTheme> snapshot) {
            if (snapshot.hasData) {
              AppTheme currentTheme = snapshot.data;
              return Column(children: _getSettingsList(context, currentTheme));
            } else {
              return Container();
            }
          }),
    );
  }

  /// Store [newUnit] in the SharedPreferences and update state of widget
  void _setDistanceUnitToSharedPref(DistanceUnit newUnit) {
    AppStateContainer.of(context).setDistanceUnit(newUnit);
    setDistanceToPrefs(newUnit);
    setState(() {
      _currentUnit = newUnit;
    });
    Navigator.pop(context, true);
  }

  /// Returns a list of widgets that make up the settings screen
  List<Widget> _getSettingsList(BuildContext context, AppTheme currentTheme) {
    return [
      // THEME
      SwitchListTile(
        // set slider to true for dark theme
        value: currentTheme.index == 1,
        onChanged: (bool newVal) {
          AppTheme newTheme =
              (newVal) ? AppTheme.THEME_DARK : AppTheme.THEME_LIGHT;
          AppStateContainer.of(context).setTheme(newTheme);
        },
        title: Text(SpartaLocalizations.of(context).labelSettingDarkTheme),
        secondary: Icon(
          OMIcons.palette,
          color: Theme.of(context).iconTheme.color,
        ),
        activeColor: Theme.of(context).primaryColor,
      ),
      // DISTANCE UNIT
      ListTile(
        leading: Icon(
          Icons.redo,
          color: Theme.of(context).primaryColor,
        ),
        title: Text(SpartaLocalizations.of(context).labelSettingDistanceUnit),
        trailing: Padding(
          padding: EdgeInsets.only(right: 12.0),
          child: Text(capitalize(distanceUnitToString(context, _currentUnit))),
        ),
        onTap: () {
          showDialog(
              context: context,
              builder: (context) {
                return SimpleDialog(
                  title: Text(
                    SpartaLocalizations.of(context)
                        .titlePopupDistanceUnitChooser,
                  ),
                  shape: RoundedRectangleBorder(
                      borderRadius: BorderRadius.circular(8.0)),
                  children: [
                    RadioListTile<DistanceUnit>(
                      title: Text(
                        capitalize(
                            SpartaLocalizations.of(context).distanceUnitKm),
                      ),
                      value: DistanceUnit.km,
                      groupValue: _currentUnit,
                      onChanged: (DistanceUnit newUnit) =>
                          _setDistanceUnitToSharedPref(newUnit),
                    ),
                    RadioListTile<DistanceUnit>(
                      title: Text(
                        capitalize(
                            SpartaLocalizations.of(context).distanceUnitMiles),
                      ),
                      value: DistanceUnit.miles,
                      groupValue: _currentUnit,
                      onChanged: (DistanceUnit newUnit) =>
                          _setDistanceUnitToSharedPref(newUnit),
                    ),
                  ],
                );
              });
        },
      ),
      // CLEAR CACHE
      ListTile(
        leading: Padding(
          padding: EdgeInsets.only(top: 8.0),
          child: Icon(
            Icons.delete_outline,
            color: Theme.of(context).primaryColor,
          ),
        ),
        title: Text(SpartaLocalizations.of(context).labelSettingClearCache),
        subtitle: Text(SpartaLocalizations.of(context).hintSettingClearCache),
        onTap: () {
          AppStateContainer.of(context).evictCache();
          Scaffold.of(context).showSnackBar(
            SnackBar(
              content: Text(
                  SpartaLocalizations.of(context).snackbarTextCacheCleared),
            ),
          );
        },
      ),
      Divider(),
      // ABOUT
      ListTile(
        title: Text(SpartaLocalizations.of(context).appBarTitleAbout),
        leading: Icon(
          Icons.info_outline,
          color: Theme.of(context).primaryColor,
        ),
        onTap: () {
          Navigator.push(
              context, CupertinoPageRoute(builder: (context) => AboutView()));
        },
      ),
    ];
  }
}

String capitalize(String s) {
  if (s == null || s.isEmpty) {
    return '';
  }
  return s[0].toUpperCase() + s.substring(1);
}
