package com.androzic.util;

/*
 * Funambol is a mobile platform developed by Funambol, Inc.
 * Copyright (C) 2003 - 2007 Funambol, Inc.
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by
 * the Free Software Foundation with the addition of the following permission
 * added to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED
 * WORK IN WHICH THE COPYRIGHT IS OWNED BY FUNAMBOL, FUNAMBOL DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program; if not, see http://www.gnu.org/licenses or write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA.
 * 
 * You can contact Funambol, Inc. headquarters at 643 Bair Island Road, Suite
 * 305, Redwood City, CA 94063, USA, or at email address info@funambol.com.
 * 
 * The interactive user interfaces in modified source and object code versions
 * of this program must display Appropriate Legal Notices, as required under
 * Section 5 of the GNU Affero General Public License version 3.
 * 
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Powered by Funambol" logo. If the display of the logo is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Powered by Funambol".
 */

/*
 * Modified by Andrey Novikov for Androzic
 */

import java.util.Hashtable;

/**
 * Escapes/unescapes special chars according to XML specifications
 */
class Entities
{
	private static final String[][] BASIC_ARRAY = { { "quot", "34" }, // " - double-quote
			{ "amp", "38" }, // & - ampersand
			{ "lt", "60" }, // < - less-than
			{ "gt", "62" }, // > - greater-than
			{ "apos", "39" }, // XML apostrophe
			{ "comma", "44" }, // XML apostrophe
	};

	public static final Entities XML;

	static
	{
		XML = new Entities();
		XML.addEntities(BASIC_ARRAY);
	}

	static interface EntityMap
	{
		void add(String name, int value);

		String name(int value);

		int value(String name);
	}

	static class PrimitiveEntityMap implements EntityMap
	{
		private Hashtable<String, Integer> mapNameToValue = new Hashtable<String, Integer>();
		private Hashtable<Integer, String> mapValueToName = new Hashtable<Integer, String>();

		public void add(String name, int value)
		{
			mapNameToValue.put(name, value);
			mapValueToName.put(value, name);
		}

		public String name(int value)
		{
			return mapValueToName.get(value);
		}

		public int value(String name)
		{
			Integer value = mapNameToValue.get(name);
			if (value == null)
			{
				return -1;
			}
			return value.intValue();
		}
	}

	static class LookupEntityMap extends PrimitiveEntityMap
	{

		private String[] lookupTable;
		private int LOOKUP_TABLE_SIZE = 256;

		public String name(int value)
		{
			if (value < LOOKUP_TABLE_SIZE)
			{
				return lookupTable()[value];
			}
			return super.name(value);
		}

		private String[] lookupTable()
		{
			if (lookupTable == null)
			{
				createLookupTable();
			}
			return lookupTable;
		}

		private void createLookupTable()
		{
			lookupTable = new String[LOOKUP_TABLE_SIZE];
			for (int i = 0, l = LOOKUP_TABLE_SIZE; i < l; ++i)
			{
				lookupTable[i] = super.name(i);
			}
		}
	}

	EntityMap map = new Entities.LookupEntityMap();

	public void addEntities(String[][] entityArray)
	{
		for (int i = 0; i < entityArray.length; ++i)
		{
			addEntity(entityArray[i][0], Integer.parseInt(entityArray[i][1]));
		}
	}

	public void addEntity(String name, int value)
	{
		map.add(name, value);
	}

	public String entityName(int value)
	{
		return map.name(value);
	}

	public int entityValue(String name)
	{
		return map.value(name);
	}

	/**
	 * Escapes special characters in a <code>String</code>.
	 * 
	 * @param str
	 *            The <code>String</code> to escape.
	 * @return A escaped <code>String</code>.
	 */
	public String escape(String str)
	{
		char ch = ' ';

		String entityName = null;
		StringBuffer buf = null;

		int intValue = 0;

		buf = new StringBuffer(str.length() * 2);

		for (int i = 0, l = str.length(); i < l; ++i)
		{
			ch = str.charAt(i);
			entityName = this.entityName(ch);

			if (entityName == null)
			{
				if (ch < 0x20)
				{
					intValue = ch;
					buf.append("&#");
					buf.append(intValue);
					buf.append(';');
				}
				else
				{
					buf.append(ch);
				}
			}
			else
			{
				buf.append('&');
				buf.append(entityName);
				buf.append(';');
			}
		}

		return buf.toString();
	}

	/**
	 * Unescapes special characters in a <code>String</code>.
	 * 
	 * @param str
	 *            The <code>String</code> to escape.
	 * @return A un-escaped <code>String</code>.
	 */
	public String unescape(String str)
	{
		StringBuffer buf = null;
		String entityName = null;

		char ch = ' ';
		char charAt1 = ' ';

		int entityValue = 0;

		buf = new StringBuffer(str.length());

		for (int i = 0, l = str.length(); i < l; ++i)
		{
			ch = str.charAt(i);

			if (ch == '&')
			{
				int semi = str.indexOf(';', i + 1);

				if (semi == -1)
				{
					buf.append(ch);
					continue;
				}

				entityName = str.substring(i + 1, semi);

				if (entityName.charAt(0) == '#')
				{
					charAt1 = entityName.charAt(1);
					if (charAt1 == 'x' || charAt1 == 'X')
					{
						entityValue = Integer.valueOf(entityName.substring(2), 16).intValue();
					}
					else
					{
						entityValue = Integer.parseInt(entityName.substring(1));
					}
				}
				else
				{
					entityValue = this.entityValue(entityName);
				}
				if (entityValue == -1)
				{
					buf.append('&');
					buf.append(entityName);
					buf.append(';');
				}
				else
				{
					buf.append((char) (entityValue));
				}

				i = semi;
			}
			else
			{
				buf.append(ch);
			}
		}

		return buf.toString();
	}
}
