package com.cg.lrceditor;

import android.Manifest;
import android.annotation.TargetApi;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.pm.PackageManager;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.Environment;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.widget.EditText;
import android.widget.TextView;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.appcompat.app.AlertDialog;
import androidx.appcompat.app.AppCompatActivity;
import androidx.appcompat.view.ActionMode;
import androidx.appcompat.widget.SearchView;
import androidx.appcompat.widget.Toolbar;
import androidx.core.app.ActivityCompat;
import androidx.core.content.ContextCompat;
import androidx.documentfile.provider.DocumentFile;
import androidx.recyclerview.widget.DividerItemDecoration;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;
import androidx.recyclerview.widget.SimpleItemAnimator;
import androidx.swiperefreshlayout.widget.SwipeRefreshLayout;

import com.google.android.material.floatingactionbutton.FloatingActionButton;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Objects;

public class HomePage extends AppCompatActivity implements HomePageListAdapter.LyricFileSelectListener {

	private String readLocation;
	private Uri readUri;

	private TextView emptyTextview;
	private RecyclerView recyclerView;
	private HomePageListAdapter adapter;

	private ActionModeCallback actionModeCallback;
	private ActionMode actionMode;

	private SwipeRefreshLayout swipeRefreshLayout;

	private String currentTheme;

	private Toolbar toolbar;
	private MenuItem refreshItem;

	private boolean isDarkTheme = false;
	private boolean threadIsExecuting = false; // Variable to prevent multiple threading operations at once
	private boolean storagePermissionIsGranted = false;
	private boolean stopScanning = true;

	private SharedPreferences preferences;

	@Override
	protected void onCreate(Bundle savedInstanceState) {
		preferences = getSharedPreferences("LRC Editor Preferences", MODE_PRIVATE);
		String theme = preferences.getString(Constants.THEME_PREFERENCE, "light");
		currentTheme = theme;
		if (theme.equals("dark")) {
			isDarkTheme = true;
			setTheme(R.style.AppThemeDark);
		} else if (theme.equals("darker")) {
			isDarkTheme = true;
			setTheme(R.style.AppThemeDarker);
		}

		super.onCreate(savedInstanceState);
		setContentView(R.layout.activity_homepage);

		toolbar = findViewById(R.id.toolbar);
		if (isDarkTheme) {
			/* Dark toolbar popups for dark themes */
			toolbar.setPopupTheme(R.style.AppThemeDark_PopupOverlay);
		}
		setSupportActionBar(toolbar);

		emptyTextview = findViewById(R.id.empty_message_textview);

		if (isDarkTheme) {
			/* Switch to a light icon when using a dark theme */
			emptyTextview.setCompoundDrawablesRelativeWithIntrinsicBounds(null, getDrawable(R.drawable.ic_thats_a_miss_light), null, null);
		}

		recyclerView = findViewById(R.id.recyclerview);
		recyclerView.setLayoutManager(new LinearLayoutManager(this));

		DividerItemDecoration dividerItemDecoration = new DividerItemDecoration(recyclerView.getContext(),
				DividerItemDecoration.VERTICAL);
		recyclerView.addItemDecoration(dividerItemDecoration);
		try {
			((SimpleItemAnimator) recyclerView.getItemAnimator()).setSupportsChangeAnimations(false);
		} catch (NullPointerException e) {
			e.printStackTrace();
		}

		updateRecyclerviewAdapter();

		swipeRefreshLayout = findViewById(R.id.swiperefresh);
		swipeRefreshLayout.setOnRefreshListener(() -> new Thread(() -> {
			if (threadIsExecuting) {
				showToastOnUiThread(getString(R.string.another_operation_wait_message));
				return;
			}
			threadIsExecuting = true;
			runOnUiThread(() -> swipeRefreshLayout.setRefreshing(true));

			scanLyrics();

			runOnUiThread(() -> swipeRefreshLayout.setRefreshing(false));
			threadIsExecuting = false;
		}).start());

		FloatingActionButton fab = findViewById(R.id.fab);
		fab.setOnClickListener(view -> {
			Intent intent = new Intent(HomePage.this, CreateActivity.class);
			startActivity(intent);
		});

		readLocation = preferences.getString(Constants.READ_LOCATION_PREFERENCE, Constants.defaultLocation);

		actionModeCallback = new ActionModeCallback();

		prepareFileIO();
	}

	@Override
	protected void onResume() {
		super.onResume();

		String theme = preferences.getString(Constants.THEME_PREFERENCE, "light");
		if (!theme.equals(currentTheme)) {
			recreate();
		}

		String oldReadLocation = readLocation;
		readLocation = preferences.getString(Constants.READ_LOCATION_PREFERENCE, Constants.defaultLocation);
		String uriString = preferences.getString("readUri", null);
		if (uriString != null) {
			readUri = Uri.parse(uriString);
		}

		if (storagePermissionIsGranted && !oldReadLocation.equals(readLocation)) {
			new Thread(() -> {
				if (threadIsExecuting) {
					showToastOnUiThread(getString(R.string.another_operation_refresh_failed_message));
					return;
				}
				threadIsExecuting = true;
				runOnUiThread(() -> swipeRefreshLayout.setRefreshing(true));

				scanLyrics();

				runOnUiThread(() -> swipeRefreshLayout.setRefreshing(false));
				threadIsExecuting = false;
			}).start();
		}
	}

	/* Takes care of everything to scan lyrics and display it */
	synchronized private void scanLyrics() {
		runOnUiThread(() -> {
			if (actionMode != null) {
				actionMode.finish();
			}
			actionMode = null;

			toolbar.collapseActionView();
		});

		final File scanLocation = new File(readLocation);

		runOnUiThread(this::updateRecyclerviewAdapter);

		if (!scanLocation.exists()) {
			if (!scanLocation.mkdir()) {
				runOnUiThread(() -> showReadLocationResetDialog(scanLocation));
			}

			runOnUiThread(() -> {
				emptyTextview.setVisibility(View.VISIBLE);
				recyclerView.setVisibility(View.GONE);
			});

			return;
		}

		//showToastOnUiThread("Scanning for LRC files in the read location (and sub directories)");
		// This toast was too annoying...

		stopScanning = false;

		try {
			runOnUiThread(() -> {
				refreshItem.setIcon(getDrawable(R.drawable.ic_cancel_toolbar));

				emptyTextview.setVisibility(View.GONE);
				recyclerView.setVisibility(View.VISIBLE);
			});

			scanDirectory(scanLocation);

			runOnUiThread(() -> {
				if (adapter != null) {
					final int noOfItems = adapter.listData.size();
					if (noOfItems <= 0) {
						if (!stopScanning) {
							Toast.makeText(getApplicationContext(), getString(R.string.no_lrc_files_found_message), Toast.LENGTH_SHORT).show();
						} else {
							Toast.makeText(getApplicationContext(), getString(R.string.scan_cancelled_message), Toast.LENGTH_SHORT).show();
						}
						emptyTextview.setVisibility(View.VISIBLE);
						recyclerView.setVisibility(View.GONE);
					} else if (!stopScanning) {
						Toast.makeText(getApplicationContext(), getResources().getQuantityString(R.plurals.scanned_x_lrc_files_message, noOfItems, noOfItems), Toast.LENGTH_SHORT).show();
					} else {
						Toast.makeText(getApplicationContext(), getString(R.string.scan_cancelled_message) + "; " + getResources().getQuantityString(R.plurals.scanned_x_lrc_files_message, noOfItems, noOfItems), Toast.LENGTH_SHORT).show();
					}
				}
			});
		} catch (NullPointerException e) {
			e.printStackTrace();
			showToastOnUiThread(getString(R.string.failed_to_scan_lyrics_message));
			showToastOnUiThread(getString(R.string.send_a_bug_report_message));
		}

		runOnUiThread(() -> {
			stopScanning = true;
			refreshItem.setIcon(getDrawable(R.drawable.ic_refresh_toolbar));
		});
	}

	/* Scans all directories and sub directories for LRC files and updates the recyclerview adapter */
	private void scanDirectory(File dir) {
		if (adapter == null) {
			showToastOnUiThread(getString(R.string.failed_to_setup_list_message));
			return;
		}

		File[] fileList = dir.listFiles();
		Arrays.sort(fileList);

		ArrayList<File> dirList = new ArrayList<>();
		for (; ; ) {
			for (final File file : fileList) {
				if (stopScanning) {
					return;
				}

				if (file.isDirectory()) {
					dirList.add(file);
				} else if (file.getName().endsWith(".lrc")) {
					runOnUiThread(() -> {
						adapter.listData.add(new HomePageListItem(file, null, null));
						// TODO: Need to optimize this. Currently causes noticeable stuttering and skipped frames
						// Due to `notifyItemInserted` being called multiple times in a small time interval
						adapter.notifyItemInserted(adapter.listData.size() - 1);
					});
				}
			}

			if (!dirList.isEmpty()) {
				fileList = dirList.remove(dirList.size() - 1).listFiles();
				Arrays.sort(fileList);
			} else {
				break;
			}
		}
	}

	/* Creates/Clears the adapter of the recyclerview */
	private void updateRecyclerviewAdapter() {
		if (adapter == null) {
			adapter = new HomePageListAdapter(this, isDarkTheme);
			recyclerView.setAdapter(adapter);
			adapter.setClickListener(this);
		} else {
			adapter.clearExpandedItems();
			adapter.listData.clear();
			adapter.backupListData.clear();
			adapter.notifyDataSetChanged();
		}
	}

	private void showReadLocationResetDialog(File scanLocation) {
		Toast.makeText(getApplicationContext(), getString(R.string.permission_check_message), Toast.LENGTH_LONG).show();

		if (!readLocation.equals(Constants.defaultLocation)) {
			new AlertDialog.Builder(HomePage.this)
					.setMessage(getString(R.string.read_location_invalid_message, scanLocation.getAbsolutePath()))
					.setPositiveButton(getString(R.string.yes), (dialog, which) -> {
						SharedPreferences.Editor editor = preferences.edit();

						editor.putString(Constants.READ_LOCATION_PREFERENCE, Constants.defaultLocation);
						editor.apply();

						readLocation = Constants.defaultLocation;
						readUri = null;

						new Thread(() -> {
							if (threadIsExecuting) {
								showToastOnUiThread(getString(R.string.another_operation_refresh_failed_message));
								return;
							}
							threadIsExecuting = true;
							runOnUiThread(() -> swipeRefreshLayout.setRefreshing(true));

							scanLyrics();

							runOnUiThread(() -> swipeRefreshLayout.setRefreshing(false));
							threadIsExecuting = false;
						}).start();
					})
					.setNegativeButton(getString(R.string.no), (dialog, which) ->
							Toast.makeText(getApplicationContext(),
									getString(R.string.lrc_editor_may_not_work_as_expected_message),
									Toast.LENGTH_LONG).show())
					.setCancelable(false)
					.create()
					.show();
		} else {
			Toast.makeText(getApplicationContext(), getString(R.string.read_location_non_existent_message), Toast.LENGTH_LONG).show();
		}
	}

	private void prepareFileIO() {
		if (!isExternalStorageWritable()) {
			Toast.makeText(this, getString(R.string.storage_unavailable_message), Toast.LENGTH_LONG).show();
			finish();
			return;
		}

		if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M || grantPermission()) /* Marshmallow onwards require runtime permissions */
			storagePermissionIsGranted = true;
	}

	private boolean grantPermission() {
		if (ContextCompat.checkSelfPermission(this,
				Manifest.permission.WRITE_EXTERNAL_STORAGE)
				!= PackageManager.PERMISSION_GRANTED) {
			showPermissionDialog();
			return false;
		}

		return true;
	}

	private void showPermissionDialog() {
		AlertDialog.Builder dialog = new AlertDialog.Builder(this);
		dialog.setMessage(getString(R.string.storage_permission_prompt));
		dialog.setTitle(getString(R.string.need_permissions));
		dialog.setCancelable(false);
		dialog.setPositiveButton(getString(R.string.ok), (dialog1, which) -> ActivityCompat.requestPermissions(HomePage.this,
				new String[]{Manifest.permission.WRITE_EXTERNAL_STORAGE}, Constants.WRITE_EXTERNAL_REQUEST));
		dialog.show();

	}

	@TargetApi(Build.VERSION_CODES.M)
	@Override
	public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
		super.onRequestPermissionsResult(requestCode, permissions, grantResults);

		if (requestCode == Constants.WRITE_EXTERNAL_REQUEST) {
			for (int i = 0; i < permissions.length; i++) {
				String permission = permissions[i];
				int grantResult = grantResults[i];

				if (permission.equals(Manifest.permission.WRITE_EXTERNAL_STORAGE)) {
					if (grantResult == PackageManager.PERMISSION_GRANTED) {
						storagePermissionIsGranted = true;
						new Thread(() -> {
							if (threadIsExecuting) {
								showToastOnUiThread(getString(R.string.another_operation_refresh_failed_message));
								return;
							}
							threadIsExecuting = true;
							runOnUiThread(() -> swipeRefreshLayout.setRefreshing(true));

							scanLyrics();

							runOnUiThread(() -> swipeRefreshLayout.setRefreshing(false));
							threadIsExecuting = false;
						}).start();
						return;
					} else {
						Toast.makeText(this, getString(R.string.no_permission_granted_message), Toast.LENGTH_LONG).show();
						finish();
					}
				}
			}
		}
	}

	/* Checks if external storage is available for read and write */
	public boolean isExternalStorageWritable() {
		String state = Environment.getExternalStorageState();
		return Environment.MEDIA_MOUNTED.equals(state);
	}

	@Override
	public boolean onCreateOptionsMenu(final Menu menu) {
		// Inflate the menu; this adds items to the action bar if it is present.
		getMenuInflater().inflate(R.menu.menu_homepage_activity, menu);

		refreshItem = menu.findItem(R.id.action_refresh);

		final MenuItem searchItem = menu.findItem(R.id.action_search);
		try {
			final SearchView searchView = (SearchView) searchItem.getActionView();
			searchView.setOnQueryTextListener(new SearchView.OnQueryTextListener() {
				@Override
				public boolean onQueryTextSubmit(String s) {
					return false;
				}

				@Override
				public boolean onQueryTextChange(String s) {
					if (actionMode != null) { // Glitches and crashes might happen when the user uses the search bar
						// with selections while it is hidden behind the contextual menu bar
						adapter.clearSelections();
						actionMode.finish();
					}
					actionMode = null;

					adapter.getFilter().filter(s);
					return true;
				}
			});

			searchItem.setOnActionExpandListener(new MenuItem.OnActionExpandListener() {
				@Override
				public boolean onMenuItemActionExpand(MenuItem menuItem) {
					adapter.backupListData = adapter.listData; // Backup current data
					return true;
				}

				@Override
				public boolean onMenuItemActionCollapse(MenuItem menuItem) {
					adapter.listData = adapter.backupListData; // Restore backed up data
					recyclerView.swapAdapter(adapter, false);
					return true;
				}
			});
		} catch (NullPointerException e) {
			Toast.makeText(this, getString(R.string.failed_to_initialize_search_message), Toast.LENGTH_SHORT).show();
			searchItem.setVisible(false);
		}

		if (storagePermissionIsGranted) {
			new Thread(() -> {
				if (threadIsExecuting) {
					showToastOnUiThread(getString(R.string.another_operation_refresh_failed_message));
					return;
				}
				threadIsExecuting = true;
				runOnUiThread(() -> swipeRefreshLayout.setRefreshing(true));

				scanLyrics();

				runOnUiThread(() -> swipeRefreshLayout.setRefreshing(false));
				threadIsExecuting = false;
			}).start();
		}

		return super.onCreateOptionsMenu(menu);
	}

	@Override
	public boolean onOptionsItemSelected(MenuItem item) {
		// Handle action bar item clicks here. The action bar will
		// automatically handle clicks on the Home/Up button, so long
		// as you specify a parent activity in AndroidManifest.xml.
		Intent intent;
		int itemID = item.getItemId();
		if (itemID == R.id.action_refresh) {
			if (stopScanning) { // If this is true, scanning is not taking place; clicked on the refresh button
				new Thread(() -> {
					if (threadIsExecuting) {
						showToastOnUiThread(getString(R.string.another_operation_wait_message));
						return;
					}
					threadIsExecuting = true;
					runOnUiThread(() -> swipeRefreshLayout.setRefreshing(true));

					scanLyrics();

					runOnUiThread(() -> swipeRefreshLayout.setRefreshing(false));
					threadIsExecuting = false;
				}).start();
			} else { // Scan is taking place; clicked on the cancel scan button
				stopScanning = true;
			}
			return true;
		} else if (itemID == R.id.action_settings) {
			intent = new Intent(this, SettingsActivity.class);
			startActivity(intent);
			return true;
		} else if (itemID == R.id.action_about) {
			intent = new Intent(this, AboutActivity.class);
			startActivity(intent);
			return true;
		}
		return super.onOptionsItemSelected(item);
	}

	private void showToastOnUiThread(final String str) {
		this.runOnUiThread(() -> Toast.makeText(getApplicationContext(), str, Toast.LENGTH_SHORT).show());
	}

	@Override
	public void fileSelected(String fileLocation, final String fileName) {
		final LyricReader r = new LyricReader(fileLocation, fileName, this);
		new Thread(() -> {
			if (threadIsExecuting) {
				showToastOnUiThread(getString(R.string.another_operation_wait_message));
				return;
			}
			threadIsExecuting = true;
			runOnUiThread(() -> swipeRefreshLayout.setRefreshing(true));

			if (r.getErrorMsg() != null || !r.readLyrics()) {
				showToastOnUiThread(r.getErrorMsg());
			} else {
				runOnUiThread(() -> {
					Intent intent = new Intent(getApplicationContext(), EditorActivity.class);
					intent.putExtra(IntentSharedStrings.LYRIC_DATA, r.getLyricData());
					intent.putExtra(IntentSharedStrings.METADATA, r.getMetadata());
					intent.putExtra(IntentSharedStrings.LRC_FILE_NAME, fileName);
					intent.putExtra(IntentSharedStrings.LRC_FILE_PATH, fileLocation);

					startActivity(intent);
				});
			}

			runOnUiThread(() -> swipeRefreshLayout.setRefreshing(false));
			threadIsExecuting = false;
		}).start();
	}

	@Override
	public void onLyricItemSelected(int position) {
		if (actionMode == null) {
			actionMode = startSupportActionMode(actionModeCallback);
		}

		toggleSelection(position);
	}

	@Override
	public void onLyricItemClicked(int position) {
		if (actionMode == null)
			return;

		toggleSelection(position);
	}

	private void toggleSelection(int position) {
		adapter.toggleSelection(position);

		checkActionModeItems();
	}

	private void checkActionModeItems() {
		int count = adapter.getSelectionCount();

		if (count == 0) {
			if (actionMode != null) {
				actionMode.finish();
			}
			actionMode = null;
		} else {
			Menu menu = actionMode.getMenu();
			MenuItem itemRename = menu.findItem(R.id.action_rename_homepage);
			itemRename.setVisible(count < 2);

			actionMode.setTitle(String.valueOf(count));
			actionMode.invalidate();
		}
	}

	private void deleteLyricFiles() {
		if (threadIsExecuting) {
			Toast.makeText(this, getString(R.string.another_operation_wait_message), Toast.LENGTH_SHORT).show();
			return;
		}

		new AlertDialog.Builder(this)
				.setTitle(getString(R.string.confirmation))
				.setMessage(getString(R.string.delete_confirmation))
				.setPositiveButton(getString(R.string.yes), (dialog, which) -> {
					final ArrayList<HomePageListItem> itemsToDelete = new ArrayList<>();
					for (int i : adapter.getSelectedItemIndices()) {
						itemsToDelete.add(adapter.listData.get(i));
					}

					new Thread(() -> deleteAsync(itemsToDelete)).start();
				})
				.setNegativeButton(getString(R.string.no), null)
				.create()
				.show();

	}

	private void deleteAsync(ArrayList<HomePageListItem> itemsToDelete) {
		if (threadIsExecuting) {
			showToastOnUiThread(getString(R.string.another_operation_wait_message));
			return;
		}
		threadIsExecuting = true;
		runOnUiThread(() -> swipeRefreshLayout.setRefreshing(true));

		showToastOnUiThread(getString(R.string.deleting_message));

		boolean deleteFailure = false;

		while (!itemsToDelete.isEmpty()) {
			HomePageListItem currentItem = itemsToDelete.remove(itemsToDelete.size() - 1);
			File fileToDelete = currentItem.file;

			DocumentFile file = FileUtil.getDocumentFileFromPath(readUri, fileToDelete.getAbsolutePath(), this);

			if (file == null || !file.delete()) {
				deleteFailure = true;
			} else {
				runOnUiThread(() -> {
					if (toolbar.hasExpandedActionView()) {
						adapter.backupListData.remove(currentItem);
					}

					int index = adapter.listData.indexOf(currentItem);
					if (index != -1) {
						adapter.listData.remove(currentItem);
						adapter.notifyItemRemoved(index);
					}

					checkActionModeItems();
				});
			}
		}

		if (deleteFailure) {
			showToastOnUiThread(getString(R.string.delete_failed_message));
		} else {
			showToastOnUiThread(getString(R.string.delete_successful_message));
		}

		runOnUiThread(() -> swipeRefreshLayout.setRefreshing(false));

		threadIsExecuting = false;
	}

	private void promptRenameFile() {
		if (threadIsExecuting) {
			Toast.makeText(this, getString(R.string.another_operation_wait_message), Toast.LENGTH_SHORT).show();
			return;
		}

		LayoutInflater inflater = this.getLayoutInflater();
		View view = inflater.inflate(R.layout.dialog_edit, null);
		EditText editText = view.findViewById(R.id.dialog_edittext);
		TextView textView = view.findViewById(R.id.dialog_prompt);

		File fileToRename = adapter.listData.get(adapter.getSelectedItemIndices().get(0)).file;

		String fileName = fileToRename.getName();
		textView.setText(getString(R.string.new_file_name_prompt));
		editText.setText(fileName);

		editText.setHint(getString(R.string.new_file_name_hint));

		new AlertDialog.Builder(this)
				.setView(view)
				.setPositiveButton(getString(R.string.rename), (dialog, which) -> {
					HomePageListItem itemToRename = adapter.listData.get(adapter.getSelectedItemIndices().get(0));
					String newName = editText.getText().toString();

					if (itemToRename.file.getName().equals(newName)) {
						// User wants the new name to be the same as the old name for whatever reason
						showToastOnUiThread(getString(R.string.rename_successful_message));
						return;
					}

					new Thread(() -> renameAsync(fileToRename, itemToRename, newName)).start();
				}).setNegativeButton(getString(R.string.cancel), null)
				.create()
				.show();
	}

	private void renameAsync(File fileToRename, HomePageListItem itemToRename, String newName) {
		if (threadIsExecuting) {
			showToastOnUiThread(getString(R.string.another_operation_wait_message));
			return;
		}
		threadIsExecuting = true;
		runOnUiThread(() -> swipeRefreshLayout.setRefreshing(true));

		runOnUiThread(() -> {
			if (actionMode != null) {
				actionMode.finish();
			}
			actionMode = null;
		});

		String location = FileUtil.stripFileNameFromPath(fileToRename.getAbsolutePath());
		DocumentFile documentFile = FileUtil.getDocumentFileFromPath(readUri, fileToRename.getAbsolutePath(), this);

		try {
			// Apparently, file names are case-insensitively handled internally in Android (vfat)
			// So gotta do this abomination for checking the file name case sensitively
			// A bit buggy as files with a same name but with a different case will have a suffix appended by
			// the system if on the internal storage and downright fails if on an external storage (SD Card)
			if (Objects.requireNonNull(new File(location, newName).getParentFile().list(
					(file, fileName) -> fileName.equals(newName))).length > 0) {
				runOnUiThread(() -> new AlertDialog.Builder(this)
						.setTitle(getString(R.string.warning))
						.setMessage(getString(R.string.overwrite_prompt, newName, location))
						.setCancelable(false)
						.setPositiveButton(getString(R.string.yes), (dialog, which) -> new Thread(() -> {
							if (readUri != null) {
								// Have to manually delete the previous file in this case to prevent an number-suffixed file being created
								try {
									DocumentFile existingFile = documentFile.getParentFile().findFile(newName);
									if (existingFile != null || !existingFile.delete()) {
										throw new NullPointerException();
									}
								} catch (NullPointerException e) {
									showToastOnUiThread(getString(R.string.failed_to_overwrite_message));
								}
							}

							renameFile(documentFile, location, itemToRename, newName, true);

							swipeRefreshLayout.setRefreshing(false);
							threadIsExecuting = false;
						}).start())
						.setNegativeButton(getString(R.string.no), (dialog, which) -> {
							swipeRefreshLayout.setRefreshing(false);
							threadIsExecuting = false;
						})
						.show());
				return;
			}
		} catch (NullPointerException ignored) {
		}

		renameFile(documentFile, location, itemToRename, newName, false);
		runOnUiThread(() -> swipeRefreshLayout.setRefreshing(false));
		threadIsExecuting = false;
	}

	private void renameFile(DocumentFile documentFile, String location, HomePageListItem itemToRename, String newName, boolean overwrote) {
		showToastOnUiThread(getString(R.string.renaming_message));

		if (documentFile != null && documentFile.renameTo(newName)) {
			showToastOnUiThread(getString(R.string.rename_successful_message));
			runOnUiThread(() -> {
				int index = adapter.listData.indexOf(itemToRename);
				if (index != -1) {
					if (!overwrote) {
						adapter.listData.get(index).file = new File(location, newName);
						adapter.notifyItemChanged(index);
					} else {
						adapter.listData.remove(index);
						adapter.notifyItemRemoved(index);
					}
				}

				if (toolbar.hasExpandedActionView()) {
					int index2 = adapter.backupListData.indexOf(itemToRename);
					if (index != -1) {
						if (!overwrote) {
							adapter.backupListData.get(index2).file = adapter.listData.get(index).file;
						} else {
							adapter.backupListData.remove(index2);
						}
					} else if (!overwrote) {
						adapter.backupListData.get(index2).file = new File(location, newName);
					}
				}
			});
		} else {
			showToastOnUiThread(getString(R.string.rename_failed_message));
		}
	}

	private void selectAll() {
		adapter.selectAll();
		int count = adapter.getSelectionCount();

		if (count >= 2)
			actionMode.getMenu().findItem(R.id.action_rename_homepage).setVisible(false);

		actionMode.setTitle(String.valueOf(count));
		actionMode.invalidate();
	}

	private class ActionModeCallback implements ActionMode.Callback {
		@Override
		public boolean onCreateActionMode(ActionMode mode, Menu menu) {
			mode.getMenuInflater().inflate(R.menu.contextual_menu_homepage_activity, menu);
			return true;
		}

		@Override
		public boolean onPrepareActionMode(ActionMode mode, Menu menu) {
			return false;
		}

		@Override
		public boolean onActionItemClicked(ActionMode mode, MenuItem item) {
			int itemID = item.getItemId();
			if (itemID == R.id.action_delete_homepage) {
				deleteLyricFiles();
				return true;
			} else if (itemID == R.id.action_rename_homepage) {
				promptRenameFile();
				return true;
			} else if (itemID == R.id.action_select_all_homepage) {
				selectAll();
				return true;
			}
			return false;
		}

		@Override
		public void onDestroyActionMode(ActionMode mode) {
			adapter.clearSelections();
			actionMode = null;
		}
	}
}
