////////////////////////////////////////////////////////////////////////////
// This file is part of BmiCalc.
// BmiCalc is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// BmiCalc is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with BmiCalc.  If not, see <https://www.gnu.org/licenses/>.
////////////////////////////////////////////////////////////////////////////
package com.ei.bmicalc;

public class BmiUtils {

    private static final double LBS_IN_KG = 2.2046;
    private static final double CM_IN_IN = 2.54;
    public static final double INCHES_IN_FEET = 12;
    /**
     * Return the Bmi for given weight and height.
     * @param weight weight in kilograms.
     * @param height height in cm.
     * @return the BMI or -1 if height is 0.
     */
    public static double getBmi(double weight, double height){
        if(height==0)
            return -1.;
        return weight * 10000 / (height * height);
    }

    public static double getBmi(double weight, boolean is_metric_weight, double height, boolean is_metric_height){
        if(!is_metric_weight)
            weight = BmiUtils.lbToKg(weight);
        if(!is_metric_height)
            height = 100 * BmiUtils.inchesToMeters(height);
        return BmiUtils.getBmi(weight, height);
    }

    /**
     * Convert length in inches to meters.
     * @param length_inches the length in inches.
     * @return the length in meters.
     */
    public static double inchesToMeters(double length_inches){
        return inToCm(length_inches) / 100;
    }

    /**
     * Convert cm to inches.
     * @param length_cm length in centimeters.
     * @return the length in inches.
     */
    public static double cmToIn(double length_cm){
        return length_cm / CM_IN_IN;
    }

    /**
     * Convert inches to centimeters.
     * @param length_in the length in inches.
     * @return the length in centimeters.
     */
    public static double inToCm(double length_in)
    {
        return length_in * CM_IN_IN;
    }

    /**
     * Convert weight in lb to kg.
     * @param weight_lb weight in lb.
     * @return weight in kg.
     */
    public static double lbToKg(double weight_lb){
        return weight_lb / LBS_IN_KG;
    }

    /**
     * Convert weight in kg to lb.
     * @param weight_kg weight in kg.
     * @return weight in lb.
     */
    public static double kgToLb(double weight_kg){
        return weight_kg * LBS_IN_KG;
    }

    /**
     * Return the number of feet in inches.
     * @param inches the number of inches.
     * @return the number of feet.
     */
    public static double inchToFeet(double inches){
        return inches/INCHES_IN_FEET;
    }

    /**
     * Return the number of inches in feet.
     * @param feet the number of feet.
     * @return the number of inches.
     */
    public static double feetToInches(double feet){
        return feet * INCHES_IN_FEET;
    }
}
