////////////////////////////////////////////////////////////////////////////
// This file is part of BmiCalc.
// BmiCalc is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// BmiCalc is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with BmiCalc.  If not, see <https://www.gnu.org/licenses/>.
////////////////////////////////////////////////////////////////////////////
package com.ei.bmicalc;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AlertDialog;
import androidx.appcompat.app.AppCompatActivity;

import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.SharedPreferences;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.Environment;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.widget.AdapterView;
import android.widget.Button;
import android.widget.CalendarView;
import android.widget.EditText;
import android.widget.Spinner;
import android.widget.TableLayout;
import android.widget.Toast;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Calendar;
import java.util.Date;

public class MainActivity extends AppCompatActivity {

    private final static String MIME_TEXT = "text/plain";
    private final static String EXTENSION = ".txt";

    //private EditText m_txtHeight;
    private EditText m_txtWeight;
    private CalendarView m_calViewDate;
    private Spinner m_spnHeightMetrics, m_spnWeightMetrics;
    private BmiDataSet m_bmiHolder;
    private Calendar m_calDate = Calendar.getInstance();
    private String m_OutputFile;
    private boolean m_isModified = false;
    private File m_flePersistent;
    private ImperialHeightFragment m_heightFragment;

    private enum Activity{
      LOAD(1),
      SAVE(2);
      final int value;
      Activity(int value){
          this.value = value;
      }
    }

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);
        m_flePersistent = new File(getExternalFilesDir(null), "CurrentFile.csv");
        initComponents();
        m_bmiHolder.redraw();
        populateData(savedInstanceState);

    }

    /**
     * Set all UI components to their values.
     */
    private void initComponents(){
        m_calViewDate = findViewById(R.id.cal_date);
        m_calViewDate.setOnDateChangeListener(new CalendarView.OnDateChangeListener() {
            @Override
            public void onSelectedDayChange(@NonNull CalendarView calendarView, int year, int month, int day) {
                m_calDate.set(year, month, day);
            }
        });
        //m_txtHeight = findViewById(R.id.txt_height);
        m_heightFragment = (ImperialHeightFragment) getSupportFragmentManager().findFragmentById(R.id.frg_height);
        m_txtWeight = findViewById(R.id.txt_weight);
        m_spnHeightMetrics = findViewById(R.id.spn_height);
        m_spnWeightMetrics = findViewById(R.id.spn_weight);
        final TableLayout tableBmi = findViewById(R.id.table_bmi);
        m_bmiHolder = new BmiDataSet(tableBmi, getApplicationContext());
        Button btnCalculateBmi = findViewById(R.id.btn_calculate);
        btnCalculateBmi.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                double weight, height, bmi;
                try{
                    weight = Double.parseDouble(m_txtWeight.getText().toString());
                }
                catch (NumberFormatException e){
                    Toast.makeText(getBaseContext(), getString(R.string.weight_type), Toast.LENGTH_SHORT).show();
                    return;
                }
                height = m_heightFragment.getHeight();
                if (weight<=0){
                    Toast.makeText(getBaseContext(), getString(R.string.weight_invalid), Toast.LENGTH_SHORT).show();
                    return;
                }
                if(height <= 0) {
                    Toast.makeText(getBaseContext(), getString(R.string.height_invalid), Toast.LENGTH_SHORT).show();
                    return;
                }
                boolean is_imperial_weight = m_spnWeightMetrics.getSelectedItem().equals(getString(R.string.lb));
                bmi = BmiUtils.getBmi(
                        weight, !is_imperial_weight,
                        height, m_spnHeightMetrics.getSelectedItem().equals(getString(R.string.metric)));
                m_bmiHolder.addEntry(m_calDate.getTime(), weight, bmi, is_imperial_weight);
                m_isModified = true;
            }
        });
        m_spnWeightMetrics.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            @Override
            public void onItemSelected(AdapterView<?> adapterView, View view, int i, long l) {
                boolean is_imperial = m_spnWeightMetrics.getItemAtPosition(i).equals(getString(R.string.lb));
                m_bmiHolder.setMetricType(is_imperial);
            }

            @Override
            public void onNothingSelected(AdapterView<?> adapterView) {
                //Nothing here.
            }
        });
        m_spnHeightMetrics.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            @Override
            public void onItemSelected(AdapterView<?> adapterView, View view, int i, long l) {
                m_heightFragment.setMetricType(m_spnHeightMetrics.getItemAtPosition(i).equals(getString(R.string.imperial)));
            }

            @Override
            public void onNothingSelected(AdapterView<?> adapterView) {

            }
        });
        Button btnClear =  findViewById(R.id.btn_clear);
        btnClear.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                m_bmiHolder.clearTable();
                m_isModified = true;
            }
        });
        Button btnSave = findViewById(R.id.btn_save);
        Button btnLoad = findViewById(R.id.btn_load);
        String strStorageState = Environment.getExternalStorageState();
        btnLoad.setEnabled(strStorageState.equals(Environment.MEDIA_MOUNTED_READ_ONLY) || strStorageState.equals(Environment.MEDIA_MOUNTED));
        btnSave.setEnabled(strStorageState.equals(Environment.MEDIA_MOUNTED));

        btnSave.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                if(m_bmiHolder.isEmpty()){
                    Toast.makeText(getBaseContext(), getString(R.string.save_table_empty), Toast.LENGTH_SHORT).show();
                    return;
                }
                if (Build.VERSION.SDK_INT >= 19) {
                    Intent intent = new Intent();
                    intent.setType(MIME_TEXT);
                    intent.setAction(Intent.ACTION_CREATE_DOCUMENT);
                    intent.addCategory(Intent.CATEGORY_OPENABLE);
                    startActivityForResult(Intent.createChooser(intent, getString(R.string.save_tbl_dialog)), Activity.SAVE.value);
                }
                else{
                    // If we are in elder versions of Android, we want to ask user for the file name to save.
                    AlertDialog.Builder builder = new AlertDialog.Builder(MainActivity.this);
                    builder.setTitle(R.string.save_tbl_dialog);
                    final EditText etFile = new EditText(MainActivity.this);
                    etFile.setText(m_OutputFile);
                    builder.setView(etFile);
                    m_OutputFile = null;
                    // Set up the buttons
                    builder.setPositiveButton("OK", new DialogInterface.OnClickListener() {
                        @Override
                        public void onClick(DialogInterface dialog, int which) {
                            m_OutputFile = etFile.getText().toString();
                            if(!m_OutputFile.isEmpty()) {
                                // We have to make sure that the file ends with .txt, because otherwise Android
                                // content resolver will not let us open the file, as its extension does not comply with
                                // MIME type.
                                if(!m_OutputFile.toLowerCase().endsWith(EXTENSION))
                                    m_OutputFile = m_OutputFile + EXTENSION;
                                File out = new File(Environment.getExternalStorageDirectory(), m_OutputFile);

                                String err = m_bmiHolder.writeCsv(out);
                                if(err==null){
                                    Toast.makeText(getBaseContext(), getString(R.string.saved_to_legacy) + out.getPath(), Toast.LENGTH_LONG).show();
                                }
                                else {
                                    Toast.makeText(getBaseContext(), err, Toast.LENGTH_SHORT).show();
                                }
                            }
                        }
                    });
                    builder.setNegativeButton("Cancel", new DialogInterface.OnClickListener() {
                        @Override
                        public void onClick(DialogInterface dialog, int which) {
                            dialog.cancel();
                        }
                    });
                    builder.show();
                }
            }
        });

        btnLoad.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                Intent intent = new Intent();
                intent.setType(MIME_TEXT);
                intent.setAction(Intent.ACTION_GET_CONTENT);
                intent.addCategory(Intent.CATEGORY_OPENABLE);
                String strTitle = getString(R.string.load_tbl_dialog);
                startActivityForResult(Intent.createChooser(intent, strTitle), Activity.LOAD.value);
            }
        });
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, @Nullable Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if(requestCode==Activity.LOAD.value){
            if(resultCode==RESULT_OK){
                invokeHolderMayBe(data, requestCode);
                m_isModified = true;
            }
            else{
                Toast.makeText(getBaseContext(), getString(R.string.load_failed), Toast.LENGTH_SHORT).show();
            }
        }
        else if (requestCode==Activity.SAVE.value){
            if(resultCode==RESULT_OK){
                invokeHolderMayBe(data, requestCode);
            }
            else{
                Toast.makeText(getBaseContext(), getString(R.string.save_failed), Toast.LENGTH_SHORT).show();
            }
        }
    }

    /**
     * Read or write the data file.
     * @param data the intent, holding data.
     * @param act_code the code of action being performed.
     */
    private void invokeHolderMayBe(@Nullable Intent data, int act_code){
        if (data==null)
            return;
        Uri uriFile = data.getData();
        if (uriFile == null)
            return;
        //File fle = new File(uriFile.getPath());
        String err = null;
        try{
            if(act_code==Activity.SAVE.value) {
                OutputStream fos = getContentResolver().openOutputStream(uriFile);
                if(fos!=null)
                    err = m_bmiHolder.writeCsv((FileOutputStream) fos);
            }
            else {
                InputStream is = getContentResolver().openInputStream(uriFile);
                if (is!=null)
                    err = m_bmiHolder.readData(is, m_spnWeightMetrics.getSelectedItem().equals(getString(R.string.lb)));
            }
        }
        catch(FileNotFoundException e){
            err = String.format(getString(R.string.file_not_found_tmpl), uriFile.getPath());
        }
        if (err != null)
            Toast.makeText(getBaseContext(), err, Toast.LENGTH_SHORT).show();
    }

    /**
     * Load the data from bundle or preferences.
     * @param savedState the bundle with saved state.
     */
    private void populateData(Bundle savedState){
        // Date
        m_calViewDate = findViewById(R.id.cal_date);

        // Height
        SharedPreferences sp = getPreferences(Context.MODE_PRIVATE);
        //m_txtHeight.setText(sp.getString(BmiDataContract.HEIGHT.value, ""))
        m_spnHeightMetrics.setSelection(sp.getInt(BmiDataContract.HEIGHT_METRICS.value, 0));

        m_heightFragment.setHeight(sp.getInt(BmiDataContract.HEIGHT.value, 0), m_spnHeightMetrics.getSelectedItem().equals(getString(R.string.imperial)));
        m_txtWeight.setText(sp.getString(BmiDataContract.WEIGHT.value, ""));
        m_spnWeightMetrics.setSelection(sp.getInt(BmiDataContract.WEIGHT_METRICS.value, 0));
        boolean is_imperial = m_spnWeightMetrics.getSelectedItem().equals(getString(R.string.lb));

        Long lngDate = savedState == null?null:savedState.getLong(BmiDataContract.DATE.value);
        if(lngDate == null) {
            Date dtNow = new Date();
            m_calDate.setTimeInMillis(dtNow.getTime());
            m_calViewDate.setDate(dtNow.getTime());
        }
        else{
            m_calDate.setTimeInMillis(lngDate);
            m_calViewDate.setDate(lngDate);
        }
        boolean blnTableLoaded = false;
        if(savedState!=null) {
            m_OutputFile = savedState.getString(BmiDataContract.SAVE_FILE.value);
            Bundle tableData = savedState.getBundle(BmiDataContract.BMI_TABLE.value);
            if (tableData != null) {
                m_bmiHolder.setData(tableData, is_imperial);
                blnTableLoaded = true;
            }
        }
        if(!blnTableLoaded && m_flePersistent.canRead()){
            m_bmiHolder.readData(m_flePersistent, m_spnWeightMetrics.getSelectedItem().equals(getString(R.string.lb)));
        }
        m_bmiHolder.setMetricType(is_imperial);
    }

    @Override
    protected void onStop() {
        super.onStop();
        saveTableToStorageMayBe();
    }

    @Override
    public void onSaveInstanceState(Bundle savedInstanceState){
        savedInstanceState.putLong(BmiDataContract.DATE.value, m_calDate.getTimeInMillis());
        savedInstanceState.putString(BmiDataContract.SAVE_FILE.value, m_OutputFile);
        savedInstanceState.putBundle(BmiDataContract.BMI_TABLE.value, m_bmiHolder.getData());
        saveTableToStorageMayBe();
        SharedPreferences.Editor spe = getPreferences(Context.MODE_PRIVATE).edit();
        spe.putInt(BmiDataContract.HEIGHT.value, (int)m_heightFragment.getHeight());
        spe.putInt(BmiDataContract.HEIGHT_METRICS.value, m_spnHeightMetrics.getSelectedItemPosition());
        spe.putString(BmiDataContract.WEIGHT.value, m_txtWeight.getText().toString());
        spe.putInt(BmiDataContract.WEIGHT_METRICS.value, m_spnWeightMetrics.getSelectedItemPosition());
        spe.apply();
        super.onSaveInstanceState(savedInstanceState);
    }

    /**
     * Save the data table to the persistent storage if needed.
     */
    private void saveTableToStorageMayBe(){
        if(m_isModified){
            // Save the table to the persistent storage.
            m_bmiHolder.writeCsv(m_flePersistent);
            m_isModified = false;
        }
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu){
        MenuInflater inflater = getMenuInflater();
        inflater.inflate(R.menu.main_menu, menu);
        return true;
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item){
        switch(item.getItemId()){
            case R.id.about:
                Intent intAbout = new Intent(getApplicationContext(), AboutActivity.class);
                startActivity(intAbout);
                return true;
            default:
                return super.onOptionsItemSelected(item);
        }
    }
}
