/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { CompilerConfig } from '../config';
import { ViewEncapsulation } from '../core';
import { DirectiveNormalizer } from '../directive_normalizer';
import { DirectiveResolver } from '../directive_resolver';
import { Lexer } from '../expression_parser/lexer';
import { Parser } from '../expression_parser/parser';
import { I18NHtmlParser } from '../i18n/i18n_html_parser';
import { InjectableCompiler } from '../injectable_compiler';
import { CompileMetadataResolver } from '../metadata_resolver';
import { HtmlParser } from '../ml_parser/html_parser';
import { NgModuleCompiler } from '../ng_module_compiler';
import { NgModuleResolver } from '../ng_module_resolver';
import { TypeScriptEmitter } from '../output/ts_emitter';
import { PipeResolver } from '../pipe_resolver';
import { DomElementSchemaRegistry } from '../schema/dom_element_schema_registry';
import { StyleCompiler } from '../style_compiler';
import { TemplateParser } from '../template_parser/template_parser';
import { syntaxError } from '../util';
import { TypeCheckCompiler } from '../view_compiler/type_check_compiler';
import { ViewCompiler } from '../view_compiler/view_compiler';
import { AotCompiler } from './compiler';
import { StaticReflector } from './static_reflector';
import { StaticSymbolCache } from './static_symbol';
import { StaticSymbolResolver } from './static_symbol_resolver';
import { AotSummaryResolver } from './summary_resolver';
export function createAotUrlResolver(host) {
    return {
        resolve: (basePath, url) => {
            const filePath = host.resourceNameToFileName(url, basePath);
            if (!filePath) {
                throw syntaxError(`Couldn't resolve resource ${url} from ${basePath}`);
            }
            return filePath;
        }
    };
}
/**
 * Creates a new AotCompiler based on options and a host.
 */
export function createAotCompiler(compilerHost, options, errorCollector) {
    let translations = options.translations || '';
    const urlResolver = createAotUrlResolver(compilerHost);
    const symbolCache = new StaticSymbolCache();
    const summaryResolver = new AotSummaryResolver(compilerHost, symbolCache);
    const symbolResolver = new StaticSymbolResolver(compilerHost, symbolCache, summaryResolver);
    const staticReflector = new StaticReflector(summaryResolver, symbolResolver, [], [], errorCollector);
    let htmlParser;
    if (!!options.enableIvy) {
        // Ivy handles i18n at the compiler level so we must use a regular parser
        htmlParser = new HtmlParser();
    }
    else {
        htmlParser = new I18NHtmlParser(new HtmlParser(), translations, options.i18nFormat, options.missingTranslation, console);
    }
    const config = new CompilerConfig({
        defaultEncapsulation: ViewEncapsulation.Emulated,
        useJit: false,
        missingTranslation: options.missingTranslation,
        preserveWhitespaces: options.preserveWhitespaces,
        strictInjectionParameters: options.strictInjectionParameters,
    });
    const normalizer = new DirectiveNormalizer({ get: (url) => compilerHost.loadResource(url) }, urlResolver, htmlParser, config);
    const expressionParser = new Parser(new Lexer());
    const elementSchemaRegistry = new DomElementSchemaRegistry();
    const tmplParser = new TemplateParser(config, staticReflector, expressionParser, elementSchemaRegistry, htmlParser, console, []);
    const resolver = new CompileMetadataResolver(config, htmlParser, new NgModuleResolver(staticReflector), new DirectiveResolver(staticReflector), new PipeResolver(staticReflector), summaryResolver, elementSchemaRegistry, normalizer, console, symbolCache, staticReflector, errorCollector);
    // TODO(vicb): do not pass options.i18nFormat here
    const viewCompiler = new ViewCompiler(staticReflector);
    const typeCheckCompiler = new TypeCheckCompiler(options, staticReflector);
    const compiler = new AotCompiler(config, options, compilerHost, staticReflector, resolver, tmplParser, new StyleCompiler(urlResolver), viewCompiler, typeCheckCompiler, new NgModuleCompiler(staticReflector), new InjectableCompiler(staticReflector, !!options.enableIvy), new TypeScriptEmitter(), summaryResolver, symbolResolver);
    return { compiler, reflector: staticReflector };
}
//# sourceMappingURL=data:application/json;base64,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