/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ChangeDetectionStrategy } from '../change_detection/constants';
import { compileComponent as render3CompileComponent, compileDirective as render3CompileDirective } from '../render3/jit/directive';
import { compilePipe as render3CompilePipe } from '../render3/jit/pipe';
import { makeDecorator, makePropDecorator } from '../util/decorators';
import { noop } from '../util/noop';
const ɵ0 = (dir = {}) => dir, ɵ1 = (type, meta) => SWITCH_COMPILE_DIRECTIVE(type, meta);
/**
 * Type of the Directive metadata.
 *
 * @publicApi
 */
export const Directive = makeDecorator('Directive', ɵ0, undefined, undefined, ɵ1);
const ɵ2 = (c = {}) => (Object.assign({ changeDetection: ChangeDetectionStrategy.Default }, c)), ɵ3 = (type, meta) => SWITCH_COMPILE_COMPONENT(type, meta);
/**
 * Component decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export const Component = makeDecorator('Component', ɵ2, Directive, undefined, ɵ3);
const ɵ4 = (p) => (Object.assign({ pure: true }, p)), ɵ5 = (type, meta) => SWITCH_COMPILE_PIPE(type, meta);
/**
 * @Annotation
 * @publicApi
 */
export const Pipe = makeDecorator('Pipe', ɵ4, undefined, undefined, ɵ5);
const ɵ6 = (bindingPropertyName) => ({ bindingPropertyName });
/**
 * @Annotation
 * @publicApi
 */
export const Input = makePropDecorator('Input', ɵ6);
const ɵ7 = (bindingPropertyName) => ({ bindingPropertyName });
/**
 * @Annotation
 * @publicApi
 */
export const Output = makePropDecorator('Output', ɵ7);
const ɵ8 = (hostPropertyName) => ({ hostPropertyName });
/**
 * @Annotation
 * @publicApi
 */
export const HostBinding = makePropDecorator('HostBinding', ɵ8);
const ɵ9 = (eventName, args) => ({ eventName, args });
/**
 * Decorator that binds a DOM event to a host listener and supplies configuration metadata.
 * Angular invokes the supplied handler method when the host element emits the specified event,
 * and updates the bound element with the result.
 *
 * If the handler method returns false, applies `preventDefault` on the bound element.
 *
 * @usageNotes
 *
 * The following example declares a directive
 * that attaches a click listener to a button and counts clicks.
 *
 * ```ts
 * @Directive({selector: 'button[counting]'})
 * class CountClicks {
 *   numberOfClicks = 0;
 *
 *   @HostListener('click', ['$event.target'])
 *   onClick(btn) {
 *     console.log('button', btn, 'number of clicks:', this.numberOfClicks++);
 *  }
 * }
 *
 * @Component({
 *   selector: 'app',
 *   template: '<button counting>Increment</button>',
 * })
 * class App {}
 *
 * ```
 *
 * The following example registers another DOM event handler that listens for key-press events.
 * ``` ts
 * import { HostListener, Component } from "@angular/core";
 *
 * @Component({
 *   selector: 'app',
 *   template: `<h1>Hello, you have pressed keys {{counter}} number of times!</h1> Press any key to
 * increment the counter.
 *   <button (click)="resetCounter()">Reset Counter</button>`
 * })
 * class AppComponent {
 *   counter = 0;
 *   @HostListener('window:keydown', ['$event'])
 *   handleKeyDown(event: KeyboardEvent) {
 *     this.counter++;
 *   }
 *   resetCounter() {
 *     this.counter = 0;
 *   }
 * }
 * ```
 *
 * @Annotation
 * @publicApi
 */
export const HostListener = makePropDecorator('HostListener', ɵ9);
export const SWITCH_COMPILE_COMPONENT__POST_R3__ = render3CompileComponent;
export const SWITCH_COMPILE_DIRECTIVE__POST_R3__ = render3CompileDirective;
export const SWITCH_COMPILE_PIPE__POST_R3__ = render3CompilePipe;
const SWITCH_COMPILE_COMPONENT__PRE_R3__ = noop;
const SWITCH_COMPILE_DIRECTIVE__PRE_R3__ = noop;
const SWITCH_COMPILE_PIPE__PRE_R3__ = noop;
const SWITCH_COMPILE_COMPONENT = SWITCH_COMPILE_COMPONENT__PRE_R3__;
const SWITCH_COMPILE_DIRECTIVE = SWITCH_COMPILE_DIRECTIVE__PRE_R3__;
const SWITCH_COMPILE_PIPE = SWITCH_COMPILE_PIPE__PRE_R3__;
export { ɵ0, ɵ1, ɵ2, ɵ3, ɵ4, ɵ5, ɵ6, ɵ7, ɵ8, ɵ9 };
//# sourceMappingURL=data:application/json;base64,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