/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ElementRef, unwrapElementRef } from '../linker/element_ref';
import { QueryList } from '../linker/query_list';
import { asElementData, asProviderData, asQueryList } from './types';
import { declaredViewContainer, filterQueryId, isEmbeddedView } from './util';
export function queryDef(flags, id, bindings) {
    let bindingDefs = [];
    for (let propName in bindings) {
        const bindingType = bindings[propName];
        bindingDefs.push({ propName, bindingType });
    }
    return {
        // will bet set by the view definition
        nodeIndex: -1,
        parent: null,
        renderParent: null,
        bindingIndex: -1,
        outputIndex: -1,
        // regular values
        // TODO(vicb): check
        checkIndex: -1,
        flags,
        childFlags: 0,
        directChildFlags: 0,
        childMatchedQueries: 0,
        ngContentIndex: -1,
        matchedQueries: {},
        matchedQueryIds: 0,
        references: {},
        childCount: 0,
        bindings: [],
        bindingFlags: 0,
        outputs: [],
        element: null,
        provider: null,
        text: null,
        query: { id, filterId: filterQueryId(id), bindings: bindingDefs },
        ngContent: null
    };
}
export function createQuery(emitDistinctChangesOnly) {
    return new QueryList(emitDistinctChangesOnly);
}
export function dirtyParentQueries(view) {
    const queryIds = view.def.nodeMatchedQueries;
    while (view.parent && isEmbeddedView(view)) {
        let tplDef = view.parentNodeDef;
        view = view.parent;
        // content queries
        const end = tplDef.nodeIndex + tplDef.childCount;
        for (let i = 0; i <= end; i++) {
            const nodeDef = view.def.nodes[i];
            if ((nodeDef.flags & 67108864 /* TypeContentQuery */) &&
                (nodeDef.flags & 536870912 /* DynamicQuery */) &&
                (nodeDef.query.filterId & queryIds) === nodeDef.query.filterId) {
                asQueryList(view, i).setDirty();
            }
            if ((nodeDef.flags & 1 /* TypeElement */ && i + nodeDef.childCount < tplDef.nodeIndex) ||
                !(nodeDef.childFlags & 67108864 /* TypeContentQuery */) ||
                !(nodeDef.childFlags & 536870912 /* DynamicQuery */)) {
                // skip elements that don't contain the template element or no query.
                i += nodeDef.childCount;
            }
        }
    }
    // view queries
    if (view.def.nodeFlags & 134217728 /* TypeViewQuery */) {
        for (let i = 0; i < view.def.nodes.length; i++) {
            const nodeDef = view.def.nodes[i];
            if ((nodeDef.flags & 134217728 /* TypeViewQuery */) && (nodeDef.flags & 536870912 /* DynamicQuery */)) {
                asQueryList(view, i).setDirty();
            }
            // only visit the root nodes
            i += nodeDef.childCount;
        }
    }
}
export function checkAndUpdateQuery(view, nodeDef) {
    const queryList = asQueryList(view, nodeDef.nodeIndex);
    if (!queryList.dirty) {
        return;
    }
    let directiveInstance;
    let newValues = undefined;
    if (nodeDef.flags & 67108864 /* TypeContentQuery */) {
        const elementDef = nodeDef.parent.parent;
        newValues = calcQueryValues(view, elementDef.nodeIndex, elementDef.nodeIndex + elementDef.childCount, nodeDef.query, []);
        directiveInstance = asProviderData(view, nodeDef.parent.nodeIndex).instance;
    }
    else if (nodeDef.flags & 134217728 /* TypeViewQuery */) {
        newValues = calcQueryValues(view, 0, view.def.nodes.length - 1, nodeDef.query, []);
        directiveInstance = view.component;
    }
    queryList.reset(newValues, unwrapElementRef);
    const bindings = nodeDef.query.bindings;
    let notify = false;
    for (let i = 0; i < bindings.length; i++) {
        const binding = bindings[i];
        let boundValue;
        switch (binding.bindingType) {
            case 0 /* First */:
                boundValue = queryList.first;
                break;
            case 1 /* All */:
                boundValue = queryList;
                notify = true;
                break;
        }
        directiveInstance[binding.propName] = boundValue;
    }
    if (notify) {
        queryList.notifyOnChanges();
    }
}
function calcQueryValues(view, startIndex, endIndex, queryDef, values) {
    for (let i = startIndex; i <= endIndex; i++) {
        const nodeDef = view.def.nodes[i];
        const valueType = nodeDef.matchedQueries[queryDef.id];
        if (valueType != null) {
            values.push(getQueryValue(view, nodeDef, valueType));
        }
        if (nodeDef.flags & 1 /* TypeElement */ && nodeDef.element.template &&
            (nodeDef.element.template.nodeMatchedQueries & queryDef.filterId) ===
                queryDef.filterId) {
            const elementData = asElementData(view, i);
            // check embedded views that were attached at the place of their template,
            // but process child nodes first if some match the query (see issue #16568)
            if ((nodeDef.childMatchedQueries & queryDef.filterId) === queryDef.filterId) {
                calcQueryValues(view, i + 1, i + nodeDef.childCount, queryDef, values);
                i += nodeDef.childCount;
            }
            if (nodeDef.flags & 16777216 /* EmbeddedViews */) {
                const embeddedViews = elementData.viewContainer._embeddedViews;
                for (let k = 0; k < embeddedViews.length; k++) {
                    const embeddedView = embeddedViews[k];
                    const dvc = declaredViewContainer(embeddedView);
                    if (dvc && dvc === elementData) {
                        calcQueryValues(embeddedView, 0, embeddedView.def.nodes.length - 1, queryDef, values);
                    }
                }
            }
            const projectedViews = elementData.template._projectedViews;
            if (projectedViews) {
                for (let k = 0; k < projectedViews.length; k++) {
                    const projectedView = projectedViews[k];
                    calcQueryValues(projectedView, 0, projectedView.def.nodes.length - 1, queryDef, values);
                }
            }
        }
        if ((nodeDef.childMatchedQueries & queryDef.filterId) !== queryDef.filterId) {
            // if no child matches the query, skip the children.
            i += nodeDef.childCount;
        }
    }
    return values;
}
export function getQueryValue(view, nodeDef, queryValueType) {
    if (queryValueType != null) {
        // a match
        switch (queryValueType) {
            case 1 /* RenderElement */:
                return asElementData(view, nodeDef.nodeIndex).renderElement;
            case 0 /* ElementRef */:
                return new ElementRef(asElementData(view, nodeDef.nodeIndex).renderElement);
            case 2 /* TemplateRef */:
                return asElementData(view, nodeDef.nodeIndex).template;
            case 3 /* ViewContainerRef */:
                return asElementData(view, nodeDef.nodeIndex).viewContainer;
            case 4 /* Provider */:
                return asProviderData(view, nodeDef.nodeIndex).instance;
        }
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicXVlcnkuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi8uLi8uLi8uLi9wYWNrYWdlcy9jb3JlL3NyYy92aWV3L3F1ZXJ5LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBOzs7Ozs7R0FNRztBQUVILE9BQU8sRUFBQyxVQUFVLEVBQUUsZ0JBQWdCLEVBQUMsTUFBTSx1QkFBdUIsQ0FBQztBQUNuRSxPQUFPLEVBQUMsU0FBUyxFQUFDLE1BQU0sc0JBQXNCLENBQUM7QUFFL0MsT0FBTyxFQUFDLGFBQWEsRUFBRSxjQUFjLEVBQUUsV0FBVyxFQUE0RixNQUFNLFNBQVMsQ0FBQztBQUM5SixPQUFPLEVBQUMscUJBQXFCLEVBQUUsYUFBYSxFQUFFLGNBQWMsRUFBQyxNQUFNLFFBQVEsQ0FBQztBQUU1RSxNQUFNLFVBQVUsUUFBUSxDQUNwQixLQUFnQixFQUFFLEVBQVUsRUFBRSxRQUFnRDtJQUNoRixJQUFJLFdBQVcsR0FBc0IsRUFBRSxDQUFDO0lBQ3hDLEtBQUssSUFBSSxRQUFRLElBQUksUUFBUSxFQUFFO1FBQzdCLE1BQU0sV0FBVyxHQUFHLFFBQVEsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUN2QyxXQUFXLENBQUMsSUFBSSxDQUFDLEVBQUMsUUFBUSxFQUFFLFdBQVcsRUFBQyxDQUFDLENBQUM7S0FDM0M7SUFFRCxPQUFPO1FBQ0wsc0NBQXNDO1FBQ3RDLFNBQVMsRUFBRSxDQUFDLENBQUM7UUFDYixNQUFNLEVBQUUsSUFBSTtRQUNaLFlBQVksRUFBRSxJQUFJO1FBQ2xCLFlBQVksRUFBRSxDQUFDLENBQUM7UUFDaEIsV0FBVyxFQUFFLENBQUMsQ0FBQztRQUNmLGlCQUFpQjtRQUNqQixvQkFBb0I7UUFDcEIsVUFBVSxFQUFFLENBQUMsQ0FBQztRQUNkLEtBQUs7UUFDTCxVQUFVLEVBQUUsQ0FBQztRQUNiLGdCQUFnQixFQUFFLENBQUM7UUFDbkIsbUJBQW1CLEVBQUUsQ0FBQztRQUN0QixjQUFjLEVBQUUsQ0FBQyxDQUFDO1FBQ2xCLGNBQWMsRUFBRSxFQUFFO1FBQ2xCLGVBQWUsRUFBRSxDQUFDO1FBQ2xCLFVBQVUsRUFBRSxFQUFFO1FBQ2QsVUFBVSxFQUFFLENBQUM7UUFDYixRQUFRLEVBQUUsRUFBRTtRQUNaLFlBQVksRUFBRSxDQUFDO1FBQ2YsT0FBTyxFQUFFLEVBQUU7UUFDWCxPQUFPLEVBQUUsSUFBSTtRQUNiLFFBQVEsRUFBRSxJQUFJO1FBQ2QsSUFBSSxFQUFFLElBQUk7UUFDVixLQUFLLEVBQUUsRUFBQyxFQUFFLEVBQUUsUUFBUSxFQUFFLGFBQWEsQ0FBQyxFQUFFLENBQUMsRUFBRSxRQUFRLEVBQUUsV0FBVyxFQUFDO1FBQy9ELFNBQVMsRUFBRSxJQUFJO0tBQ2hCLENBQUM7QUFDSixDQUFDO0FBRUQsTUFBTSxVQUFVLFdBQVcsQ0FBQyx1QkFBZ0M7SUFDMUQsT0FBTyxJQUFJLFNBQVMsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0FBQ2hELENBQUM7QUFFRCxNQUFNLFVBQVUsa0JBQWtCLENBQUMsSUFBYztJQUMvQyxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLGtCQUFrQixDQUFDO0lBQzdDLE9BQU8sSUFBSSxDQUFDLE1BQU0sSUFBSSxjQUFjLENBQUMsSUFBSSxDQUFDLEVBQUU7UUFDMUMsSUFBSSxNQUFNLEdBQUcsSUFBSSxDQUFDLGFBQWMsQ0FBQztRQUNqQyxJQUFJLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQztRQUNuQixrQkFBa0I7UUFDbEIsTUFBTSxHQUFHLEdBQUcsTUFBTSxDQUFDLFNBQVMsR0FBRyxNQUFNLENBQUMsVUFBVSxDQUFDO1FBQ2pELEtBQUssSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsSUFBSSxHQUFHLEVBQUUsQ0FBQyxFQUFFLEVBQUU7WUFDN0IsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDbEMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLGtDQUE2QixDQUFDO2dCQUM1QyxDQUFDLE9BQU8sQ0FBQyxLQUFLLCtCQUF5QixDQUFDO2dCQUN4QyxDQUFDLE9BQU8sQ0FBQyxLQUFNLENBQUMsUUFBUSxHQUFHLFFBQVEsQ0FBQyxLQUFLLE9BQU8sQ0FBQyxLQUFNLENBQUMsUUFBUSxFQUFFO2dCQUNwRSxXQUFXLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDLFFBQVEsRUFBRSxDQUFDO2FBQ2pDO1lBQ0QsSUFBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLHNCQUF3QixJQUFJLENBQUMsR0FBRyxPQUFPLENBQUMsVUFBVSxHQUFHLE1BQU0sQ0FBQyxTQUFTLENBQUM7Z0JBQ3BGLENBQUMsQ0FBQyxPQUFPLENBQUMsVUFBVSxrQ0FBNkIsQ0FBQztnQkFDbEQsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxVQUFVLCtCQUF5QixDQUFDLEVBQUU7Z0JBQ2xELHFFQUFxRTtnQkFDckUsQ0FBQyxJQUFJLE9BQU8sQ0FBQyxVQUFVLENBQUM7YUFDekI7U0FDRjtLQUNGO0lBRUQsZUFBZTtJQUNmLElBQUksSUFBSSxDQUFDLEdBQUcsQ0FBQyxTQUFTLGdDQUEwQixFQUFFO1FBQ2hELEtBQUssSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxNQUFNLEVBQUUsQ0FBQyxFQUFFLEVBQUU7WUFDOUMsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDbEMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLGdDQUEwQixDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsS0FBSywrQkFBeUIsQ0FBQyxFQUFFO2dCQUN6RixXQUFXLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDLFFBQVEsRUFBRSxDQUFDO2FBQ2pDO1lBQ0QsNEJBQTRCO1lBQzVCLENBQUMsSUFBSSxPQUFPLENBQUMsVUFBVSxDQUFDO1NBQ3pCO0tBQ0Y7QUFDSCxDQUFDO0FBRUQsTUFBTSxVQUFVLG1CQUFtQixDQUFDLElBQWMsRUFBRSxPQUFnQjtJQUNsRSxNQUFNLFNBQVMsR0FBRyxXQUFXLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUN2RCxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssRUFBRTtRQUNwQixPQUFPO0tBQ1I7SUFDRCxJQUFJLGlCQUFzQixDQUFDO0lBQzNCLElBQUksU0FBUyxHQUFVLFNBQVUsQ0FBQztJQUNsQyxJQUFJLE9BQU8sQ0FBQyxLQUFLLGtDQUE2QixFQUFFO1FBQzlDLE1BQU0sVUFBVSxHQUFHLE9BQU8sQ0FBQyxNQUFPLENBQUMsTUFBTyxDQUFDO1FBQzNDLFNBQVMsR0FBRyxlQUFlLENBQ3ZCLElBQUksRUFBRSxVQUFVLENBQUMsU0FBUyxFQUFFLFVBQVUsQ0FBQyxTQUFTLEdBQUcsVUFBVSxDQUFDLFVBQVUsRUFBRSxPQUFPLENBQUMsS0FBTSxFQUN4RixFQUFFLENBQUMsQ0FBQztRQUNSLGlCQUFpQixHQUFHLGNBQWMsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLE1BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQyxRQUFRLENBQUM7S0FDOUU7U0FBTSxJQUFJLE9BQU8sQ0FBQyxLQUFLLGdDQUEwQixFQUFFO1FBQ2xELFNBQVMsR0FBRyxlQUFlLENBQUMsSUFBSSxFQUFFLENBQUMsRUFBRSxJQUFJLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFLE9BQU8sQ0FBQyxLQUFNLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDcEYsaUJBQWlCLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQztLQUNwQztJQUNELFNBQVMsQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLGdCQUFnQixDQUFDLENBQUM7SUFDN0MsTUFBTSxRQUFRLEdBQUcsT0FBTyxDQUFDLEtBQU0sQ0FBQyxRQUFRLENBQUM7SUFDekMsSUFBSSxNQUFNLEdBQUcsS0FBSyxDQUFDO0lBQ25CLEtBQUssSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxRQUFRLENBQUMsTUFBTSxFQUFFLENBQUMsRUFBRSxFQUFFO1FBQ3hDLE1BQU0sT0FBTyxHQUFHLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUM1QixJQUFJLFVBQWUsQ0FBQztRQUNwQixRQUFRLE9BQU8sQ0FBQyxXQUFXLEVBQUU7WUFDM0I7Z0JBQ0UsVUFBVSxHQUFHLFNBQVMsQ0FBQyxLQUFLLENBQUM7Z0JBQzdCLE1BQU07WUFDUjtnQkFDRSxVQUFVLEdBQUcsU0FBUyxDQUFDO2dCQUN2QixNQUFNLEdBQUcsSUFBSSxDQUFDO2dCQUNkLE1BQU07U0FDVDtRQUNELGlCQUFpQixDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsR0FBRyxVQUFVLENBQUM7S0FDbEQ7SUFDRCxJQUFJLE1BQU0sRUFBRTtRQUNWLFNBQVMsQ0FBQyxlQUFlLEVBQUUsQ0FBQztLQUM3QjtBQUNILENBQUM7QUFFRCxTQUFTLGVBQWUsQ0FDcEIsSUFBYyxFQUFFLFVBQWtCLEVBQUUsUUFBZ0IsRUFBRSxRQUFrQixFQUN4RSxNQUFhO0lBQ2YsS0FBSyxJQUFJLENBQUMsR0FBRyxVQUFVLEVBQUUsQ0FBQyxJQUFJLFFBQVEsRUFBRSxDQUFDLEVBQUUsRUFBRTtRQUMzQyxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNsQyxNQUFNLFNBQVMsR0FBRyxPQUFPLENBQUMsY0FBYyxDQUFDLFFBQVEsQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUN0RCxJQUFJLFNBQVMsSUFBSSxJQUFJLEVBQUU7WUFDckIsTUFBTSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLE9BQU8sRUFBRSxTQUFTLENBQUMsQ0FBQyxDQUFDO1NBQ3REO1FBQ0QsSUFBSSxPQUFPLENBQUMsS0FBSyxzQkFBd0IsSUFBSSxPQUFPLENBQUMsT0FBUSxDQUFDLFFBQVE7WUFDbEUsQ0FBQyxPQUFPLENBQUMsT0FBUSxDQUFDLFFBQVUsQ0FBQyxrQkFBa0IsR0FBRyxRQUFRLENBQUMsUUFBUSxDQUFDO2dCQUNoRSxRQUFRLENBQUMsUUFBUSxFQUFFO1lBQ3pCLE1BQU0sV0FBVyxHQUFHLGFBQWEsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDM0MsMEVBQTBFO1lBQzFFLDJFQUEyRTtZQUMzRSxJQUFJLENBQUMsT0FBTyxDQUFDLG1CQUFtQixHQUFHLFFBQVEsQ0FBQyxRQUFRLENBQUMsS0FBSyxRQUFRLENBQUMsUUFBUSxFQUFFO2dCQUMzRSxlQUFlLENBQUMsSUFBSSxFQUFFLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLE9BQU8sQ0FBQyxVQUFVLEVBQUUsUUFBUSxFQUFFLE1BQU0sQ0FBQyxDQUFDO2dCQUN2RSxDQUFDLElBQUksT0FBTyxDQUFDLFVBQVUsQ0FBQzthQUN6QjtZQUNELElBQUksT0FBTyxDQUFDLEtBQUssK0JBQTBCLEVBQUU7Z0JBQzNDLE1BQU0sYUFBYSxHQUFHLFdBQVcsQ0FBQyxhQUFjLENBQUMsY0FBYyxDQUFDO2dCQUNoRSxLQUFLLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsYUFBYSxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsRUFBRTtvQkFDN0MsTUFBTSxZQUFZLEdBQUcsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDO29CQUN0QyxNQUFNLEdBQUcsR0FBRyxxQkFBcUIsQ0FBQyxZQUFZLENBQUMsQ0FBQztvQkFDaEQsSUFBSSxHQUFHLElBQUksR0FBRyxLQUFLLFdBQVcsRUFBRTt3QkFDOUIsZUFBZSxDQUFDLFlBQVksRUFBRSxDQUFDLEVBQUUsWUFBWSxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRSxRQUFRLEVBQUUsTUFBTSxDQUFDLENBQUM7cUJBQ3ZGO2lCQUNGO2FBQ0Y7WUFDRCxNQUFNLGNBQWMsR0FBRyxXQUFXLENBQUMsUUFBUSxDQUFDLGVBQWUsQ0FBQztZQUM1RCxJQUFJLGNBQWMsRUFBRTtnQkFDbEIsS0FBSyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLGNBQWMsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxFQUFFLEVBQUU7b0JBQzlDLE1BQU0sYUFBYSxHQUFHLGNBQWMsQ0FBQyxDQUFDLENBQUMsQ0FBQztvQkFDeEMsZUFBZSxDQUFDLGFBQWEsRUFBRSxDQUFDLEVBQUUsYUFBYSxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRSxRQUFRLEVBQUUsTUFBTSxDQUFDLENBQUM7aUJBQ3pGO2FBQ0Y7U0FDRjtRQUNELElBQUksQ0FBQyxPQUFPLENBQUMsbUJBQW1CLEdBQUcsUUFBUSxDQUFDLFFBQVEsQ0FBQyxLQUFLLFFBQVEsQ0FBQyxRQUFRLEVBQUU7WUFDM0Usb0RBQW9EO1lBQ3BELENBQUMsSUFBSSxPQUFPLENBQUMsVUFBVSxDQUFDO1NBQ3pCO0tBQ0Y7SUFDRCxPQUFPLE1BQU0sQ0FBQztBQUNoQixDQUFDO0FBRUQsTUFBTSxVQUFVLGFBQWEsQ0FDekIsSUFBYyxFQUFFLE9BQWdCLEVBQUUsY0FBOEI7SUFDbEUsSUFBSSxjQUFjLElBQUksSUFBSSxFQUFFO1FBQzFCLFVBQVU7UUFDVixRQUFRLGNBQWMsRUFBRTtZQUN0QjtnQkFDRSxPQUFPLGFBQWEsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDLGFBQWEsQ0FBQztZQUM5RDtnQkFDRSxPQUFPLElBQUksVUFBVSxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDLGFBQWEsQ0FBQyxDQUFDO1lBQzlFO2dCQUNFLE9BQU8sYUFBYSxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsU0FBUyxDQUFDLENBQUMsUUFBUSxDQUFDO1lBQ3pEO2dCQUNFLE9BQU8sYUFBYSxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsU0FBUyxDQUFDLENBQUMsYUFBYSxDQUFDO1lBQzlEO2dCQUNFLE9BQU8sY0FBYyxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsU0FBUyxDQUFDLENBQUMsUUFBUSxDQUFDO1NBQzNEO0tBQ0Y7QUFDSCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBAbGljZW5zZVxuICogQ29weXJpZ2h0IEdvb2dsZSBMTEMgQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiBVc2Ugb2YgdGhpcyBzb3VyY2UgY29kZSBpcyBnb3Zlcm5lZCBieSBhbiBNSVQtc3R5bGUgbGljZW5zZSB0aGF0IGNhbiBiZVxuICogZm91bmQgaW4gdGhlIExJQ0VOU0UgZmlsZSBhdCBodHRwczovL2FuZ3VsYXIuaW8vbGljZW5zZVxuICovXG5cbmltcG9ydCB7RWxlbWVudFJlZiwgdW53cmFwRWxlbWVudFJlZn0gZnJvbSAnLi4vbGlua2VyL2VsZW1lbnRfcmVmJztcbmltcG9ydCB7UXVlcnlMaXN0fSBmcm9tICcuLi9saW5rZXIvcXVlcnlfbGlzdCc7XG5cbmltcG9ydCB7YXNFbGVtZW50RGF0YSwgYXNQcm92aWRlckRhdGEsIGFzUXVlcnlMaXN0LCBOb2RlRGVmLCBOb2RlRmxhZ3MsIFF1ZXJ5QmluZGluZ0RlZiwgUXVlcnlCaW5kaW5nVHlwZSwgUXVlcnlEZWYsIFF1ZXJ5VmFsdWVUeXBlLCBWaWV3RGF0YX0gZnJvbSAnLi90eXBlcyc7XG5pbXBvcnQge2RlY2xhcmVkVmlld0NvbnRhaW5lciwgZmlsdGVyUXVlcnlJZCwgaXNFbWJlZGRlZFZpZXd9IGZyb20gJy4vdXRpbCc7XG5cbmV4cG9ydCBmdW5jdGlvbiBxdWVyeURlZihcbiAgICBmbGFnczogTm9kZUZsYWdzLCBpZDogbnVtYmVyLCBiaW5kaW5nczoge1twcm9wTmFtZTogc3RyaW5nXTogUXVlcnlCaW5kaW5nVHlwZX0pOiBOb2RlRGVmIHtcbiAgbGV0IGJpbmRpbmdEZWZzOiBRdWVyeUJpbmRpbmdEZWZbXSA9IFtdO1xuICBmb3IgKGxldCBwcm9wTmFtZSBpbiBiaW5kaW5ncykge1xuICAgIGNvbnN0IGJpbmRpbmdUeXBlID0gYmluZGluZ3NbcHJvcE5hbWVdO1xuICAgIGJpbmRpbmdEZWZzLnB1c2goe3Byb3BOYW1lLCBiaW5kaW5nVHlwZX0pO1xuICB9XG5cbiAgcmV0dXJuIHtcbiAgICAvLyB3aWxsIGJldCBzZXQgYnkgdGhlIHZpZXcgZGVmaW5pdGlvblxuICAgIG5vZGVJbmRleDogLTEsXG4gICAgcGFyZW50OiBudWxsLFxuICAgIHJlbmRlclBhcmVudDogbnVsbCxcbiAgICBiaW5kaW5nSW5kZXg6IC0xLFxuICAgIG91dHB1dEluZGV4OiAtMSxcbiAgICAvLyByZWd1bGFyIHZhbHVlc1xuICAgIC8vIFRPRE8odmljYik6IGNoZWNrXG4gICAgY2hlY2tJbmRleDogLTEsXG4gICAgZmxhZ3MsXG4gICAgY2hpbGRGbGFnczogMCxcbiAgICBkaXJlY3RDaGlsZEZsYWdzOiAwLFxuICAgIGNoaWxkTWF0Y2hlZFF1ZXJpZXM6IDAsXG4gICAgbmdDb250ZW50SW5kZXg6IC0xLFxuICAgIG1hdGNoZWRRdWVyaWVzOiB7fSxcbiAgICBtYXRjaGVkUXVlcnlJZHM6IDAsXG4gICAgcmVmZXJlbmNlczoge30sXG4gICAgY2hpbGRDb3VudDogMCxcbiAgICBiaW5kaW5nczogW10sXG4gICAgYmluZGluZ0ZsYWdzOiAwLFxuICAgIG91dHB1dHM6IFtdLFxuICAgIGVsZW1lbnQ6IG51bGwsXG4gICAgcHJvdmlkZXI6IG51bGwsXG4gICAgdGV4dDogbnVsbCxcbiAgICBxdWVyeToge2lkLCBmaWx0ZXJJZDogZmlsdGVyUXVlcnlJZChpZCksIGJpbmRpbmdzOiBiaW5kaW5nRGVmc30sXG4gICAgbmdDb250ZW50OiBudWxsXG4gIH07XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBjcmVhdGVRdWVyeShlbWl0RGlzdGluY3RDaGFuZ2VzT25seTogYm9vbGVhbik6IFF1ZXJ5TGlzdDxhbnk+IHtcbiAgcmV0dXJuIG5ldyBRdWVyeUxpc3QoZW1pdERpc3RpbmN0Q2hhbmdlc09ubHkpO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gZGlydHlQYXJlbnRRdWVyaWVzKHZpZXc6IFZpZXdEYXRhKSB7XG4gIGNvbnN0IHF1ZXJ5SWRzID0gdmlldy5kZWYubm9kZU1hdGNoZWRRdWVyaWVzO1xuICB3aGlsZSAodmlldy5wYXJlbnQgJiYgaXNFbWJlZGRlZFZpZXcodmlldykpIHtcbiAgICBsZXQgdHBsRGVmID0gdmlldy5wYXJlbnROb2RlRGVmITtcbiAgICB2aWV3ID0gdmlldy5wYXJlbnQ7XG4gICAgLy8gY29udGVudCBxdWVyaWVzXG4gICAgY29uc3QgZW5kID0gdHBsRGVmLm5vZGVJbmRleCArIHRwbERlZi5jaGlsZENvdW50O1xuICAgIGZvciAobGV0IGkgPSAwOyBpIDw9IGVuZDsgaSsrKSB7XG4gICAgICBjb25zdCBub2RlRGVmID0gdmlldy5kZWYubm9kZXNbaV07XG4gICAgICBpZiAoKG5vZGVEZWYuZmxhZ3MgJiBOb2RlRmxhZ3MuVHlwZUNvbnRlbnRRdWVyeSkgJiZcbiAgICAgICAgICAobm9kZURlZi5mbGFncyAmIE5vZGVGbGFncy5EeW5hbWljUXVlcnkpICYmXG4gICAgICAgICAgKG5vZGVEZWYucXVlcnkhLmZpbHRlcklkICYgcXVlcnlJZHMpID09PSBub2RlRGVmLnF1ZXJ5IS5maWx0ZXJJZCkge1xuICAgICAgICBhc1F1ZXJ5TGlzdCh2aWV3LCBpKS5zZXREaXJ0eSgpO1xuICAgICAgfVxuICAgICAgaWYgKChub2RlRGVmLmZsYWdzICYgTm9kZUZsYWdzLlR5cGVFbGVtZW50ICYmIGkgKyBub2RlRGVmLmNoaWxkQ291bnQgPCB0cGxEZWYubm9kZUluZGV4KSB8fFxuICAgICAgICAgICEobm9kZURlZi5jaGlsZEZsYWdzICYgTm9kZUZsYWdzLlR5cGVDb250ZW50UXVlcnkpIHx8XG4gICAgICAgICAgIShub2RlRGVmLmNoaWxkRmxhZ3MgJiBOb2RlRmxhZ3MuRHluYW1pY1F1ZXJ5KSkge1xuICAgICAgICAvLyBza2lwIGVsZW1lbnRzIHRoYXQgZG9uJ3QgY29udGFpbiB0aGUgdGVtcGxhdGUgZWxlbWVudCBvciBubyBxdWVyeS5cbiAgICAgICAgaSArPSBub2RlRGVmLmNoaWxkQ291bnQ7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgLy8gdmlldyBxdWVyaWVzXG4gIGlmICh2aWV3LmRlZi5ub2RlRmxhZ3MgJiBOb2RlRmxhZ3MuVHlwZVZpZXdRdWVyeSkge1xuICAgIGZvciAobGV0IGkgPSAwOyBpIDwgdmlldy5kZWYubm9kZXMubGVuZ3RoOyBpKyspIHtcbiAgICAgIGNvbnN0IG5vZGVEZWYgPSB2aWV3LmRlZi5ub2Rlc1tpXTtcbiAgICAgIGlmICgobm9kZURlZi5mbGFncyAmIE5vZGVGbGFncy5UeXBlVmlld1F1ZXJ5KSAmJiAobm9kZURlZi5mbGFncyAmIE5vZGVGbGFncy5EeW5hbWljUXVlcnkpKSB7XG4gICAgICAgIGFzUXVlcnlMaXN0KHZpZXcsIGkpLnNldERpcnR5KCk7XG4gICAgICB9XG4gICAgICAvLyBvbmx5IHZpc2l0IHRoZSByb290IG5vZGVzXG4gICAgICBpICs9IG5vZGVEZWYuY2hpbGRDb3VudDtcbiAgICB9XG4gIH1cbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGNoZWNrQW5kVXBkYXRlUXVlcnkodmlldzogVmlld0RhdGEsIG5vZGVEZWY6IE5vZGVEZWYpIHtcbiAgY29uc3QgcXVlcnlMaXN0ID0gYXNRdWVyeUxpc3Qodmlldywgbm9kZURlZi5ub2RlSW5kZXgpO1xuICBpZiAoIXF1ZXJ5TGlzdC5kaXJ0eSkge1xuICAgIHJldHVybjtcbiAgfVxuICBsZXQgZGlyZWN0aXZlSW5zdGFuY2U6IGFueTtcbiAgbGV0IG5ld1ZhbHVlczogYW55W10gPSB1bmRlZmluZWQhO1xuICBpZiAobm9kZURlZi5mbGFncyAmIE5vZGVGbGFncy5UeXBlQ29udGVudFF1ZXJ5KSB7XG4gICAgY29uc3QgZWxlbWVudERlZiA9IG5vZGVEZWYucGFyZW50IS5wYXJlbnQhO1xuICAgIG5ld1ZhbHVlcyA9IGNhbGNRdWVyeVZhbHVlcyhcbiAgICAgICAgdmlldywgZWxlbWVudERlZi5ub2RlSW5kZXgsIGVsZW1lbnREZWYubm9kZUluZGV4ICsgZWxlbWVudERlZi5jaGlsZENvdW50LCBub2RlRGVmLnF1ZXJ5ISxcbiAgICAgICAgW10pO1xuICAgIGRpcmVjdGl2ZUluc3RhbmNlID0gYXNQcm92aWRlckRhdGEodmlldywgbm9kZURlZi5wYXJlbnQhLm5vZGVJbmRleCkuaW5zdGFuY2U7XG4gIH0gZWxzZSBpZiAobm9kZURlZi5mbGFncyAmIE5vZGVGbGFncy5UeXBlVmlld1F1ZXJ5KSB7XG4gICAgbmV3VmFsdWVzID0gY2FsY1F1ZXJ5VmFsdWVzKHZpZXcsIDAsIHZpZXcuZGVmLm5vZGVzLmxlbmd0aCAtIDEsIG5vZGVEZWYucXVlcnkhLCBbXSk7XG4gICAgZGlyZWN0aXZlSW5zdGFuY2UgPSB2aWV3LmNvbXBvbmVudDtcbiAgfVxuICBxdWVyeUxpc3QucmVzZXQobmV3VmFsdWVzLCB1bndyYXBFbGVtZW50UmVmKTtcbiAgY29uc3QgYmluZGluZ3MgPSBub2RlRGVmLnF1ZXJ5IS5iaW5kaW5ncztcbiAgbGV0IG5vdGlmeSA9IGZhbHNlO1xuICBmb3IgKGxldCBpID0gMDsgaSA8IGJpbmRpbmdzLmxlbmd0aDsgaSsrKSB7XG4gICAgY29uc3QgYmluZGluZyA9IGJpbmRpbmdzW2ldO1xuICAgIGxldCBib3VuZFZhbHVlOiBhbnk7XG4gICAgc3dpdGNoIChiaW5kaW5nLmJpbmRpbmdUeXBlKSB7XG4gICAgICBjYXNlIFF1ZXJ5QmluZGluZ1R5cGUuRmlyc3Q6XG4gICAgICAgIGJvdW5kVmFsdWUgPSBxdWVyeUxpc3QuZmlyc3Q7XG4gICAgICAgIGJyZWFrO1xuICAgICAgY2FzZSBRdWVyeUJpbmRpbmdUeXBlLkFsbDpcbiAgICAgICAgYm91bmRWYWx1ZSA9IHF1ZXJ5TGlzdDtcbiAgICAgICAgbm90aWZ5ID0gdHJ1ZTtcbiAgICAgICAgYnJlYWs7XG4gICAgfVxuICAgIGRpcmVjdGl2ZUluc3RhbmNlW2JpbmRpbmcucHJvcE5hbWVdID0gYm91bmRWYWx1ZTtcbiAgfVxuICBpZiAobm90aWZ5KSB7XG4gICAgcXVlcnlMaXN0Lm5vdGlmeU9uQ2hhbmdlcygpO1xuICB9XG59XG5cbmZ1bmN0aW9uIGNhbGNRdWVyeVZhbHVlcyhcbiAgICB2aWV3OiBWaWV3RGF0YSwgc3RhcnRJbmRleDogbnVtYmVyLCBlbmRJbmRleDogbnVtYmVyLCBxdWVyeURlZjogUXVlcnlEZWYsXG4gICAgdmFsdWVzOiBhbnlbXSk6IGFueVtdIHtcbiAgZm9yIChsZXQgaSA9IHN0YXJ0SW5kZXg7IGkgPD0gZW5kSW5kZXg7IGkrKykge1xuICAgIGNvbnN0IG5vZGVEZWYgPSB2aWV3LmRlZi5ub2Rlc1tpXTtcbiAgICBjb25zdCB2YWx1ZVR5cGUgPSBub2RlRGVmLm1hdGNoZWRRdWVyaWVzW3F1ZXJ5RGVmLmlkXTtcbiAgICBpZiAodmFsdWVUeXBlICE9IG51bGwpIHtcbiAgICAgIHZhbHVlcy5wdXNoKGdldFF1ZXJ5VmFsdWUodmlldywgbm9kZURlZiwgdmFsdWVUeXBlKSk7XG4gICAgfVxuICAgIGlmIChub2RlRGVmLmZsYWdzICYgTm9kZUZsYWdzLlR5cGVFbGVtZW50ICYmIG5vZGVEZWYuZWxlbWVudCEudGVtcGxhdGUgJiZcbiAgICAgICAgKG5vZGVEZWYuZWxlbWVudCEudGVtcGxhdGUgIS5ub2RlTWF0Y2hlZFF1ZXJpZXMgJiBxdWVyeURlZi5maWx0ZXJJZCkgPT09XG4gICAgICAgICAgICBxdWVyeURlZi5maWx0ZXJJZCkge1xuICAgICAgY29uc3QgZWxlbWVudERhdGEgPSBhc0VsZW1lbnREYXRhKHZpZXcsIGkpO1xuICAgICAgLy8gY2hlY2sgZW1iZWRkZWQgdmlld3MgdGhhdCB3ZXJlIGF0dGFjaGVkIGF0IHRoZSBwbGFjZSBvZiB0aGVpciB0ZW1wbGF0ZSxcbiAgICAgIC8vIGJ1dCBwcm9jZXNzIGNoaWxkIG5vZGVzIGZpcnN0IGlmIHNvbWUgbWF0Y2ggdGhlIHF1ZXJ5IChzZWUgaXNzdWUgIzE2NTY4KVxuICAgICAgaWYgKChub2RlRGVmLmNoaWxkTWF0Y2hlZFF1ZXJpZXMgJiBxdWVyeURlZi5maWx0ZXJJZCkgPT09IHF1ZXJ5RGVmLmZpbHRlcklkKSB7XG4gICAgICAgIGNhbGNRdWVyeVZhbHVlcyh2aWV3LCBpICsgMSwgaSArIG5vZGVEZWYuY2hpbGRDb3VudCwgcXVlcnlEZWYsIHZhbHVlcyk7XG4gICAgICAgIGkgKz0gbm9kZURlZi5jaGlsZENvdW50O1xuICAgICAgfVxuICAgICAgaWYgKG5vZGVEZWYuZmxhZ3MgJiBOb2RlRmxhZ3MuRW1iZWRkZWRWaWV3cykge1xuICAgICAgICBjb25zdCBlbWJlZGRlZFZpZXdzID0gZWxlbWVudERhdGEudmlld0NvbnRhaW5lciEuX2VtYmVkZGVkVmlld3M7XG4gICAgICAgIGZvciAobGV0IGsgPSAwOyBrIDwgZW1iZWRkZWRWaWV3cy5sZW5ndGg7IGsrKykge1xuICAgICAgICAgIGNvbnN0IGVtYmVkZGVkVmlldyA9IGVtYmVkZGVkVmlld3Nba107XG4gICAgICAgICAgY29uc3QgZHZjID0gZGVjbGFyZWRWaWV3Q29udGFpbmVyKGVtYmVkZGVkVmlldyk7XG4gICAgICAgICAgaWYgKGR2YyAmJiBkdmMgPT09IGVsZW1lbnREYXRhKSB7XG4gICAgICAgICAgICBjYWxjUXVlcnlWYWx1ZXMoZW1iZWRkZWRWaWV3LCAwLCBlbWJlZGRlZFZpZXcuZGVmLm5vZGVzLmxlbmd0aCAtIDEsIHF1ZXJ5RGVmLCB2YWx1ZXMpO1xuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfVxuICAgICAgY29uc3QgcHJvamVjdGVkVmlld3MgPSBlbGVtZW50RGF0YS50ZW1wbGF0ZS5fcHJvamVjdGVkVmlld3M7XG4gICAgICBpZiAocHJvamVjdGVkVmlld3MpIHtcbiAgICAgICAgZm9yIChsZXQgayA9IDA7IGsgPCBwcm9qZWN0ZWRWaWV3cy5sZW5ndGg7IGsrKykge1xuICAgICAgICAgIGNvbnN0IHByb2plY3RlZFZpZXcgPSBwcm9qZWN0ZWRWaWV3c1trXTtcbiAgICAgICAgICBjYWxjUXVlcnlWYWx1ZXMocHJvamVjdGVkVmlldywgMCwgcHJvamVjdGVkVmlldy5kZWYubm9kZXMubGVuZ3RoIC0gMSwgcXVlcnlEZWYsIHZhbHVlcyk7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG4gICAgaWYgKChub2RlRGVmLmNoaWxkTWF0Y2hlZFF1ZXJpZXMgJiBxdWVyeURlZi5maWx0ZXJJZCkgIT09IHF1ZXJ5RGVmLmZpbHRlcklkKSB7XG4gICAgICAvLyBpZiBubyBjaGlsZCBtYXRjaGVzIHRoZSBxdWVyeSwgc2tpcCB0aGUgY2hpbGRyZW4uXG4gICAgICBpICs9IG5vZGVEZWYuY2hpbGRDb3VudDtcbiAgICB9XG4gIH1cbiAgcmV0dXJuIHZhbHVlcztcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGdldFF1ZXJ5VmFsdWUoXG4gICAgdmlldzogVmlld0RhdGEsIG5vZGVEZWY6IE5vZGVEZWYsIHF1ZXJ5VmFsdWVUeXBlOiBRdWVyeVZhbHVlVHlwZSk6IGFueSB7XG4gIGlmIChxdWVyeVZhbHVlVHlwZSAhPSBudWxsKSB7XG4gICAgLy8gYSBtYXRjaFxuICAgIHN3aXRjaCAocXVlcnlWYWx1ZVR5cGUpIHtcbiAgICAgIGNhc2UgUXVlcnlWYWx1ZVR5cGUuUmVuZGVyRWxlbWVudDpcbiAgICAgICAgcmV0dXJuIGFzRWxlbWVudERhdGEodmlldywgbm9kZURlZi5ub2RlSW5kZXgpLnJlbmRlckVsZW1lbnQ7XG4gICAgICBjYXNlIFF1ZXJ5VmFsdWVUeXBlLkVsZW1lbnRSZWY6XG4gICAgICAgIHJldHVybiBuZXcgRWxlbWVudFJlZihhc0VsZW1lbnREYXRhKHZpZXcsIG5vZGVEZWYubm9kZUluZGV4KS5yZW5kZXJFbGVtZW50KTtcbiAgICAgIGNhc2UgUXVlcnlWYWx1ZVR5cGUuVGVtcGxhdGVSZWY6XG4gICAgICAgIHJldHVybiBhc0VsZW1lbnREYXRhKHZpZXcsIG5vZGVEZWYubm9kZUluZGV4KS50ZW1wbGF0ZTtcbiAgICAgIGNhc2UgUXVlcnlWYWx1ZVR5cGUuVmlld0NvbnRhaW5lclJlZjpcbiAgICAgICAgcmV0dXJuIGFzRWxlbWVudERhdGEodmlldywgbm9kZURlZi5ub2RlSW5kZXgpLnZpZXdDb250YWluZXI7XG4gICAgICBjYXNlIFF1ZXJ5VmFsdWVUeXBlLlByb3ZpZGVyOlxuICAgICAgICByZXR1cm4gYXNQcm92aWRlckRhdGEodmlldywgbm9kZURlZi5ub2RlSW5kZXgpLmluc3RhbmNlO1xuICAgIH1cbiAgfVxufVxuIl19