/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
// Called before each cycle of a view's check to detect whether this is in the
// initState for which we need to call ngOnInit, ngAfterContentInit or ngAfterViewInit
// lifecycle methods. Returns true if this check cycle should call lifecycle
// methods.
export function shiftInitState(view, priorInitState, newInitState) {
    // Only update the InitState if we are currently in the prior state.
    // For example, only move into CallingInit if we are in BeforeInit. Only
    // move into CallingContentInit if we are in CallingInit. Normally this will
    // always be true because of how checkCycle is called in checkAndUpdateView.
    // However, if checkAndUpdateView is called recursively or if an exception is
    // thrown while checkAndUpdateView is running, checkAndUpdateView starts over
    // from the beginning. This ensures the state is monotonically increasing,
    // terminating in the AfterInit state, which ensures the Init methods are called
    // at least once and only once.
    const state = view.state;
    const initState = state & 1792 /* InitState_Mask */;
    if (initState === priorInitState) {
        view.state = (state & ~1792 /* InitState_Mask */) | newInitState;
        view.initIndex = -1;
        return true;
    }
    return initState === newInitState;
}
// Returns true if the lifecycle init method should be called for the node with
// the given init index.
export function shouldCallLifecycleInitHook(view, initState, index) {
    if ((view.state & 1792 /* InitState_Mask */) === initState && view.initIndex <= index) {
        view.initIndex = index + 1;
        return true;
    }
    return false;
}
/**
 * Node instance data.
 *
 * We have a separate type per NodeType to save memory
 * (TextData | ElementData | ProviderData | PureExpressionData | QueryList<any>)
 *
 * To keep our code monomorphic,
 * we prohibit using `NodeData` directly but enforce the use of accessors (`asElementData`, ...).
 * This way, no usage site can get a `NodeData` from view.nodes and then use it for different
 * purposes.
 */
export class NodeData {
}
/**
 * Accessor for view.nodes, enforcing that every usage site stays monomorphic.
 */
export function asTextData(view, index) {
    return view.nodes[index];
}
/**
 * Accessor for view.nodes, enforcing that every usage site stays monomorphic.
 */
export function asElementData(view, index) {
    return view.nodes[index];
}
/**
 * Accessor for view.nodes, enforcing that every usage site stays monomorphic.
 */
export function asProviderData(view, index) {
    return view.nodes[index];
}
/**
 * Accessor for view.nodes, enforcing that every usage site stays monomorphic.
 */
export function asPureExpressionData(view, index) {
    return view.nodes[index];
}
/**
 * Accessor for view.nodes, enforcing that every usage site stays monomorphic.
 */
export function asQueryList(view, index) {
    return view.nodes[index];
}
export class DebugContext {
}
/**
 * This object is used to prevent cycles in the source files and to have a place where
 * debug mode can hook it. It is lazily filled when `isDevMode` is known.
 */
export const Services = {
    setCurrentNode: undefined,
    createRootView: undefined,
    createEmbeddedView: undefined,
    createComponentView: undefined,
    createNgModuleRef: undefined,
    overrideProvider: undefined,
    overrideComponentView: undefined,
    clearOverrides: undefined,
    checkAndUpdateView: undefined,
    checkNoChangesView: undefined,
    destroyView: undefined,
    resolveDep: undefined,
    createDebugContext: undefined,
    handleEvent: undefined,
    updateDirectives: undefined,
    updateRenderer: undefined,
    dirtyParentQueries: undefined,
};
//# sourceMappingURL=data:application/json;base64,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