"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addCommand = void 0;
const tslib_1 = require("tslib");
const utils_fs_1 = require("@ionic/utils-fs");
const utils_terminal_1 = require("@ionic/utils-terminal");
const add_1 = require("../android/add");
const common_1 = require("../android/common");
const colors_1 = tslib_1.__importDefault(require("../colors"));
const common_2 = require("../common");
const errors_1 = require("../errors");
const add_2 = require("../ios/add");
const common_3 = require("../ios/common");
const log_1 = require("../log");
const sync_1 = require("./sync");
async function addCommand(config, selectedPlatformName) {
    var _a;
    if (selectedPlatformName && !(await common_2.isValidPlatform(selectedPlatformName))) {
        const platformDir = common_2.resolvePlatform(config, selectedPlatformName);
        if (platformDir) {
            await common_2.runPlatformHook(config, platformDir, 'capacitor:add');
        }
        else {
            let msg = `Platform ${colors_1.default.input(selectedPlatformName)} not found.`;
            if (await common_2.isValidCommunityPlatform(selectedPlatformName)) {
                msg += `\nTry installing ${colors_1.default.strong(`@capacitor-community/${selectedPlatformName}`)} and adding the platform again.`;
            }
            log_1.logger.error(msg);
        }
    }
    else {
        const knownPlatforms = await common_2.getKnownPlatforms();
        const platformName = await common_2.promptForPlatform(knownPlatforms, `Please choose a platform to add:`, selectedPlatformName);
        if (platformName === config.web.name) {
            webWarning();
            return;
        }
        const existingPlatformDir = await common_2.getProjectPlatformDirectory(config, platformName);
        if (existingPlatformDir) {
            errors_1.fatal(`${colors_1.default.input(platformName)} platform already exists.\n` +
                `To re-add this platform, first remove ${colors_1.default.strong(utils_terminal_1.prettyPath(existingPlatformDir))}, then run this command again.\n` +
                `${colors_1.default.strong('WARNING')}: Your native project will be completely removed.`);
        }
        try {
            await common_2.check([
                () => common_2.checkPackage(),
                () => common_2.checkAppConfig(config),
                ...addChecks(config, platformName),
            ]);
            await doAdd(config, platformName);
            await editPlatforms(config, platformName);
            if (shouldSync(config, platformName)) {
                if (await utils_fs_1.pathExists(config.app.webDirAbs)) {
                    sync_1.sync(config, platformName, false);
                }
                else {
                    log_1.logger.warn(`${colors_1.default.success(colors_1.default.strong('sync'))} could not run--missing ${colors_1.default.strong(config.app.webDir)} directory.`);
                }
            }
            printNextSteps(platformName);
        }
        catch (e) {
            if (!errors_1.isFatal(e)) {
                errors_1.fatal((_a = e.stack) !== null && _a !== void 0 ? _a : e);
            }
            throw e;
        }
    }
}
exports.addCommand = addCommand;
function printNextSteps(platformName) {
    log_1.logSuccess(`${colors_1.default.strong(platformName)} platform added!`);
    log_1.output.write(`Follow the Developer Workflow guide to get building:\n${colors_1.default.strong(`https://capacitorjs.com/docs/v3/basics/workflow`)}\n`);
}
function addChecks(config, platformName) {
    if (platformName === config.ios.name) {
        return [() => common_3.checkIOSPackage(config), () => common_3.checkCocoaPods(config)];
    }
    else if (platformName === config.android.name) {
        return [() => common_1.checkAndroidPackage(config)];
    }
    else if (platformName === config.web.name) {
        return [];
    }
    else {
        throw `Platform ${platformName} is not valid.`;
    }
}
async function doAdd(config, platformName) {
    await common_2.runTask(colors_1.default.success(colors_1.default.strong('add')), async () => {
        if (platformName === config.ios.name) {
            await add_2.addIOS(config);
        }
        else if (platformName === config.android.name) {
            await add_1.addAndroid(config);
        }
    });
}
async function editPlatforms(config, platformName) {
    if (platformName === config.ios.name) {
        await common_3.editProjectSettingsIOS(config);
    }
    else if (platformName === config.android.name) {
        await common_1.editProjectSettingsAndroid(config);
    }
}
function shouldSync(config, platformName) {
    // Don't sync if we're adding the iOS platform not on a mac
    if (config.cli.os !== "mac" /* Mac */ && platformName === 'ios') {
        return false;
    }
    return true;
}
function webWarning() {
    log_1.logger.error(`Not adding platform ${colors_1.default.strong('web')}.\n` +
        `In Capacitor, the web platform is just your web app! For example, if you have a React or Angular project, the web platform is that project.\n` +
        `To add Capacitor functionality to your web app, follow the Web Getting Started Guide: ${colors_1.default.strong('https://capacitorjs.com/docs/v3/web')}`);
}
