/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { getClosureSafeProperty } from '../../util/property';
/**
 * Construct an `InjectableDef` which defines how a token will be constructed by the DI system, and
 * in which injectors (if any) it will be available.
 *
 * This should be assigned to a static `ɵprov` field on a type, which will then be an
 * `InjectableType`.
 *
 * Options:
 * * `providedIn` determines which injectors will include the injectable, by either associating it
 *   with an `@NgModule` or other `InjectorType`, or by specifying that this injectable should be
 *   provided in the `'root'` injector, which will be the application-level injector in most apps.
 * * `factory` gives the zero argument function which will create an instance of the injectable.
 *   The factory can call `inject` to access the `Injector` and request injection of dependencies.
 *
 * @codeGenApi
 */
export function ɵɵdefineInjectable(opts) {
    return {
        token: opts.token, providedIn: opts.providedIn || null, factory: opts.factory,
        value: undefined,
    };
}
/**
 * @deprecated in v8, delete after v10. This API should be used only be generated code, and that
 * code should now use ɵɵdefineInjectable instead.
 * @publicApi
 */
export const defineInjectable = ɵɵdefineInjectable;
/**
 * Construct an `InjectorDef` which configures an injector.
 *
 * This should be assigned to a static injector def (`ɵinj`) field on a type, which will then be an
 * `InjectorType`.
 *
 * Options:
 *
 * * `factory`: an `InjectorType` is an instantiable type, so a zero argument `factory` function to
 *   create the type must be provided. If that factory function needs to inject arguments, it can
 *   use the `inject` function.
 * * `providers`: an optional array of providers to add to the injector. Each provider must
 *   either have a factory or point to a type which has a `ɵprov` static property (the
 *   type must be an `InjectableType`).
 * * `imports`: an optional array of imports of other `InjectorType`s or `InjectorTypeWithModule`s
 *   whose providers will also be added to the injector. Locally provided types will override
 *   providers from imports.
 *
 * @publicApi
 */
export function ɵɵdefineInjector(options) {
    return {
        factory: options.factory, providers: options.providers || [], imports: options.imports || [],
    };
}
/**
 * Read the injectable def (`ɵprov`) for `type` in a way which is immune to accidentally reading
 * inherited value.
 *
 * @param type A type which may have its own (non-inherited) `ɵprov`.
 */
export function getInjectableDef(type) {
    return getOwnDefinition(type, type[NG_PROV_DEF]) ||
        getOwnDefinition(type, type[NG_INJECTABLE_DEF]);
}
/**
 * Return `def` only if it is defined directly on `type` and is not inherited from a base
 * class of `type`.
 *
 * The function `Object.hasOwnProperty` is not sufficient to distinguish this case because in older
 * browsers (e.g. IE10) static property inheritance is implemented by copying the properties.
 *
 * Instead, the definition's `token` is compared to the `type`, and if they don't match then the
 * property was not defined directly on the type itself, and was likely inherited. The definition
 * is only returned if the `type` matches the `def.token`.
 */
function getOwnDefinition(type, def) {
    return def && def.token === type ? def : null;
}
/**
 * Read the injectable def (`ɵprov`) for `type` or read the `ɵprov` from one of its ancestors.
 *
 * @param type A type which may have `ɵprov`, via inheritance.
 *
 * @deprecated Will be removed in v10, where an error will occur in the scenario if we find the
 * `ɵprov` on an ancestor only.
 */
export function getInheritedInjectableDef(type) {
    // See `jit/injectable.ts#compileInjectable` for context on NG_PROV_DEF_FALLBACK.
    const def = type && (type[NG_PROV_DEF] || type[NG_INJECTABLE_DEF] ||
        (type[NG_PROV_DEF_FALLBACK] && type[NG_PROV_DEF_FALLBACK]()));
    if (def) {
        const typeName = getTypeName(type);
        // TODO(FW-1307): Re-add ngDevMode when closure can handle it
        // ngDevMode &&
        console.warn(`DEPRECATED: DI is instantiating a token "${typeName}" that inherits its @Injectable decorator but does not provide one itself.\n` +
            `This will become an error in v10. Please add @Injectable() to the "${typeName}" class.`);
        return def;
    }
    else {
        return null;
    }
}
/** Gets the name of a type, accounting for some cross-browser differences. */
function getTypeName(type) {
    // `Function.prototype.name` behaves differently between IE and other browsers. In most browsers
    // it'll always return the name of the function itself, no matter how many other functions it
    // inherits from. On IE the function doesn't have its own `name` property, but it takes it from
    // the lowest level in the prototype chain. E.g. if we have `class Foo extends Parent` most
    // browsers will evaluate `Foo.name` to `Foo` while IE will return `Parent`. We work around
    // the issue by converting the function to a string and parsing its name out that way via a regex.
    if (type.hasOwnProperty('name')) {
        return type.name;
    }
    const match = ('' + type).match(/^function\s*([^\s(]+)/);
    return match === null ? '' : match[1];
}
/**
 * Read the injector def type in a way which is immune to accidentally reading inherited value.
 *
 * @param type type which may have an injector def (`ɵinj`)
 */
export function getInjectorDef(type) {
    return type && (type.hasOwnProperty(NG_INJ_DEF) || type.hasOwnProperty(NG_INJECTOR_DEF)) ?
        type[NG_INJ_DEF] :
        null;
}
export const NG_PROV_DEF = getClosureSafeProperty({ ɵprov: getClosureSafeProperty });
export const NG_INJ_DEF = getClosureSafeProperty({ ɵinj: getClosureSafeProperty });
// On IE10 properties defined via `defineProperty` won't be inherited by child classes,
// which will break inheriting the injectable definition from a grandparent through an
// undecorated parent class. We work around it by defining a fallback method which will be
// used to retrieve the definition. This should only be a problem in JIT mode, because in
// AOT TypeScript seems to have a workaround for static properties. When inheriting from an
// undecorated parent is no longer supported in v10, this can safely be removed.
export const NG_PROV_DEF_FALLBACK = getClosureSafeProperty({ ɵprovFallback: getClosureSafeProperty });
// We need to keep these around so we can read off old defs if new defs are unavailable
export const NG_INJECTABLE_DEF = getClosureSafeProperty({ ngInjectableDef: getClosureSafeProperty });
export const NG_INJECTOR_DEF = getClosureSafeProperty({ ngInjectorDef: getClosureSafeProperty });
//# sourceMappingURL=data:application/json;base64,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