/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { __read } from "tslib";
import { ViewEncapsulation } from '../metadata/view';
import { SecurityContext } from '../sanitization/security';
import { asElementData } from './types';
import { NOOP, calcBindingFlags, checkAndUpdateBinding, dispatchEvent, elementEventFullName, getParentRenderElement, resolveDefinition, resolveRendererType2, splitMatchedQueriesDsl, splitNamespace } from './util';
export function anchorDef(flags, matchedQueriesDsl, ngContentIndex, childCount, handleEvent, templateFactory) {
    flags |= 1 /* TypeElement */;
    var _a = splitMatchedQueriesDsl(matchedQueriesDsl), matchedQueries = _a.matchedQueries, references = _a.references, matchedQueryIds = _a.matchedQueryIds;
    var template = templateFactory ? resolveDefinition(templateFactory) : null;
    return {
        // will bet set by the view definition
        nodeIndex: -1,
        parent: null,
        renderParent: null,
        bindingIndex: -1,
        outputIndex: -1,
        // regular values
        flags: flags,
        checkIndex: -1,
        childFlags: 0,
        directChildFlags: 0,
        childMatchedQueries: 0, matchedQueries: matchedQueries, matchedQueryIds: matchedQueryIds, references: references, ngContentIndex: ngContentIndex, childCount: childCount,
        bindings: [],
        bindingFlags: 0,
        outputs: [],
        element: {
            ns: null,
            name: null,
            attrs: null, template: template,
            componentProvider: null,
            componentView: null,
            componentRendererType: null,
            publicProviders: null,
            allProviders: null,
            handleEvent: handleEvent || NOOP
        },
        provider: null,
        text: null,
        query: null,
        ngContent: null
    };
}
export function elementDef(checkIndex, flags, matchedQueriesDsl, ngContentIndex, childCount, namespaceAndName, fixedAttrs, bindings, outputs, handleEvent, componentView, componentRendererType) {
    var _a;
    if (fixedAttrs === void 0) { fixedAttrs = []; }
    if (!handleEvent) {
        handleEvent = NOOP;
    }
    var _b = splitMatchedQueriesDsl(matchedQueriesDsl), matchedQueries = _b.matchedQueries, references = _b.references, matchedQueryIds = _b.matchedQueryIds;
    var ns = null;
    var name = null;
    if (namespaceAndName) {
        _a = __read(splitNamespace(namespaceAndName), 2), ns = _a[0], name = _a[1];
    }
    bindings = bindings || [];
    var bindingDefs = [];
    for (var i = 0; i < bindings.length; i++) {
        var _c = __read(bindings[i], 3), bindingFlags = _c[0], namespaceAndName_1 = _c[1], suffixOrSecurityContext = _c[2];
        var _d = __read(splitNamespace(namespaceAndName_1), 2), ns_1 = _d[0], name_1 = _d[1];
        var securityContext = undefined;
        var suffix = undefined;
        switch (bindingFlags & 15 /* Types */) {
            case 4 /* TypeElementStyle */:
                suffix = suffixOrSecurityContext;
                break;
            case 1 /* TypeElementAttribute */:
            case 8 /* TypeProperty */:
                securityContext = suffixOrSecurityContext;
                break;
        }
        bindingDefs[i] =
            { flags: bindingFlags, ns: ns_1, name: name_1, nonMinifiedName: name_1, securityContext: securityContext, suffix: suffix };
    }
    outputs = outputs || [];
    var outputDefs = [];
    for (var i = 0; i < outputs.length; i++) {
        var _e = __read(outputs[i], 2), target = _e[0], eventName = _e[1];
        outputDefs[i] = {
            type: 0 /* ElementOutput */,
            target: target, eventName: eventName,
            propName: null
        };
    }
    fixedAttrs = fixedAttrs || [];
    var attrs = fixedAttrs.map(function (_a) {
        var _b = __read(_a, 2), namespaceAndName = _b[0], value = _b[1];
        var _c = __read(splitNamespace(namespaceAndName), 2), ns = _c[0], name = _c[1];
        return [ns, name, value];
    });
    componentRendererType = resolveRendererType2(componentRendererType);
    if (componentView) {
        flags |= 33554432 /* ComponentView */;
    }
    flags |= 1 /* TypeElement */;
    return {
        // will bet set by the view definition
        nodeIndex: -1,
        parent: null,
        renderParent: null,
        bindingIndex: -1,
        outputIndex: -1,
        // regular values
        checkIndex: checkIndex,
        flags: flags,
        childFlags: 0,
        directChildFlags: 0,
        childMatchedQueries: 0, matchedQueries: matchedQueries, matchedQueryIds: matchedQueryIds, references: references, ngContentIndex: ngContentIndex, childCount: childCount,
        bindings: bindingDefs,
        bindingFlags: calcBindingFlags(bindingDefs),
        outputs: outputDefs,
        element: {
            ns: ns,
            name: name,
            attrs: attrs,
            template: null,
            // will bet set by the view definition
            componentProvider: null,
            componentView: componentView || null,
            componentRendererType: componentRendererType,
            publicProviders: null,
            allProviders: null,
            handleEvent: handleEvent || NOOP,
        },
        provider: null,
        text: null,
        query: null,
        ngContent: null
    };
}
export function createElement(view, renderHost, def) {
    var elDef = def.element;
    var rootSelectorOrNode = view.root.selectorOrNode;
    var renderer = view.renderer;
    var el;
    if (view.parent || !rootSelectorOrNode) {
        if (elDef.name) {
            el = renderer.createElement(elDef.name, elDef.ns);
        }
        else {
            el = renderer.createComment('');
        }
        var parentEl = getParentRenderElement(view, renderHost, def);
        if (parentEl) {
            renderer.appendChild(parentEl, el);
        }
    }
    else {
        // when using native Shadow DOM, do not clear the root element contents to allow slot projection
        var preserveContent = (!!elDef.componentRendererType &&
            elDef.componentRendererType.encapsulation === ViewEncapsulation.ShadowDom);
        el = renderer.selectRootElement(rootSelectorOrNode, preserveContent);
    }
    if (elDef.attrs) {
        for (var i = 0; i < elDef.attrs.length; i++) {
            var _a = __read(elDef.attrs[i], 3), ns = _a[0], name_2 = _a[1], value = _a[2];
            renderer.setAttribute(el, name_2, value, ns);
        }
    }
    return el;
}
export function listenToElementOutputs(view, compView, def, el) {
    for (var i = 0; i < def.outputs.length; i++) {
        var output = def.outputs[i];
        var handleEventClosure = renderEventHandlerClosure(view, def.nodeIndex, elementEventFullName(output.target, output.eventName));
        var listenTarget = output.target;
        var listenerView = view;
        if (output.target === 'component') {
            listenTarget = null;
            listenerView = compView;
        }
        var disposable = listenerView.renderer.listen(listenTarget || el, output.eventName, handleEventClosure);
        view.disposables[def.outputIndex + i] = disposable;
    }
}
function renderEventHandlerClosure(view, index, eventName) {
    return function (event) { return dispatchEvent(view, index, eventName, event); };
}
export function checkAndUpdateElementInline(view, def, v0, v1, v2, v3, v4, v5, v6, v7, v8, v9) {
    var bindLen = def.bindings.length;
    var changed = false;
    if (bindLen > 0 && checkAndUpdateElementValue(view, def, 0, v0))
        changed = true;
    if (bindLen > 1 && checkAndUpdateElementValue(view, def, 1, v1))
        changed = true;
    if (bindLen > 2 && checkAndUpdateElementValue(view, def, 2, v2))
        changed = true;
    if (bindLen > 3 && checkAndUpdateElementValue(view, def, 3, v3))
        changed = true;
    if (bindLen > 4 && checkAndUpdateElementValue(view, def, 4, v4))
        changed = true;
    if (bindLen > 5 && checkAndUpdateElementValue(view, def, 5, v5))
        changed = true;
    if (bindLen > 6 && checkAndUpdateElementValue(view, def, 6, v6))
        changed = true;
    if (bindLen > 7 && checkAndUpdateElementValue(view, def, 7, v7))
        changed = true;
    if (bindLen > 8 && checkAndUpdateElementValue(view, def, 8, v8))
        changed = true;
    if (bindLen > 9 && checkAndUpdateElementValue(view, def, 9, v9))
        changed = true;
    return changed;
}
export function checkAndUpdateElementDynamic(view, def, values) {
    var changed = false;
    for (var i = 0; i < values.length; i++) {
        if (checkAndUpdateElementValue(view, def, i, values[i]))
            changed = true;
    }
    return changed;
}
function checkAndUpdateElementValue(view, def, bindingIdx, value) {
    if (!checkAndUpdateBinding(view, def, bindingIdx, value)) {
        return false;
    }
    var binding = def.bindings[bindingIdx];
    var elData = asElementData(view, def.nodeIndex);
    var renderNode = elData.renderElement;
    var name = binding.name;
    switch (binding.flags & 15 /* Types */) {
        case 1 /* TypeElementAttribute */:
            setElementAttribute(view, binding, renderNode, binding.ns, name, value);
            break;
        case 2 /* TypeElementClass */:
            setElementClass(view, renderNode, name, value);
            break;
        case 4 /* TypeElementStyle */:
            setElementStyle(view, binding, renderNode, name, value);
            break;
        case 8 /* TypeProperty */:
            var bindView = (def.flags & 33554432 /* ComponentView */ &&
                binding.flags & 32 /* SyntheticHostProperty */) ?
                elData.componentView :
                view;
            setElementProperty(bindView, binding, renderNode, name, value);
            break;
    }
    return true;
}
function setElementAttribute(view, binding, renderNode, ns, name, value) {
    var securityContext = binding.securityContext;
    var renderValue = securityContext ? view.root.sanitizer.sanitize(securityContext, value) : value;
    renderValue = renderValue != null ? renderValue.toString() : null;
    var renderer = view.renderer;
    if (value != null) {
        renderer.setAttribute(renderNode, name, renderValue, ns);
    }
    else {
        renderer.removeAttribute(renderNode, name, ns);
    }
}
function setElementClass(view, renderNode, name, value) {
    var renderer = view.renderer;
    if (value) {
        renderer.addClass(renderNode, name);
    }
    else {
        renderer.removeClass(renderNode, name);
    }
}
function setElementStyle(view, binding, renderNode, name, value) {
    var renderValue = view.root.sanitizer.sanitize(SecurityContext.STYLE, value);
    if (renderValue != null) {
        renderValue = renderValue.toString();
        var unit = binding.suffix;
        if (unit != null) {
            renderValue = renderValue + unit;
        }
    }
    else {
        renderValue = null;
    }
    var renderer = view.renderer;
    if (renderValue != null) {
        renderer.setStyle(renderNode, name, renderValue);
    }
    else {
        renderer.removeStyle(renderNode, name);
    }
}
function setElementProperty(view, binding, renderNode, name, value) {
    var securityContext = binding.securityContext;
    var renderValue = securityContext ? view.root.sanitizer.sanitize(securityContext, value) : value;
    view.renderer.setProperty(renderNode, name, renderValue);
}
//# sourceMappingURL=data:application/json;base64,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