package com.ghostsq.commander.samba;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetAddress;
import java.util.Arrays;

import android.content.Context;
import android.content.pm.PackageManager;
import android.content.res.Resources;
import android.net.DhcpInfo;
import android.net.Uri;
import android.net.wifi.WifiManager;
import android.util.Log;
import android.util.SparseBooleanArray;

import jcifs.smb.SmbFile;
import jcifs.smb.NtlmPasswordAuthentication;
import jcifs.Config;
import jcifs.util.LogStream;

import com.ghostsq.commander.Commander;
import com.ghostsq.commander.adapters.CA;
import com.ghostsq.commander.adapters.CommanderAdapter;
import com.ghostsq.commander.adapters.CommanderAdapterBase;
import com.ghostsq.commander.adapters.Engine;
import com.ghostsq.commander.adapters.Engines;
import com.ghostsq.commander.adapters.Engines.IReciever;
import com.ghostsq.commander.adapters.FSAdapter;
import com.ghostsq.commander.favorites.Favorite;
import com.ghostsq.commander.utils.Credentials;
import com.ghostsq.commander.utils.Utils;

//import org.apache.http.auth.UsernamePasswordCredentials;

public class SMBAdapter extends CommanderAdapterBase implements Engines.IReciever {
    private static final String TAG = "SMBAdapter"; 
    public  Uri       uri = null;
    private SmbItem[] items;
    public  NtlmPasswordAuthentication credentials = null;
   
    static {
        Config.setProperty( "jcifs.resolveOrder", "BCAST" );    //,DNS
        Config.setProperty( "jcifs.smb.client.responseTimeout", "120000" );
        Config.setProperty( "jcifs.netbios.retryTimeout", "5000" );
        Config.setProperty( "jcifs.netbios.retryCount", "5" );
//        LogStream.setLevel( 4 );
    }
    
    // taken from http://code.google.com/p/boxeeremote/wiki/AndroidUDP
    private final InetAddress getBroadcastAddress( DhcpInfo dhcp ) throws IOException {
        if( dhcp == null ) return null;
        int broadcast = (dhcp.ipAddress & dhcp.netmask) | ~dhcp.netmask;
        byte[] quads = new byte[4];
        for (int k = 0; k < 4; k++)
          quads[k] = (byte) ((broadcast >> k * 8) & 0xFF);
        return InetAddress.getByAddress(quads);
    }    

    public SMBAdapter() {
    }
    
    public SMBAdapter( Context ctx_ ) {
        super( ctx_ );
    }
    
    @Override
    public String toString() {
        if( uri == null )
            return "";
        String ui = uri.getUserInfo();
        if( ui != null && credentials == null )
            return Favorite.screenPwd( uri );
        if( credentials == null )
            return uri.toString();
        Uri uri_with_cred_scr = Utils.getUriWithAuth( uri, credentials.getName(), Credentials.pwScreen );
        return uri_with_cred_scr.toString();    
    }
    @Override
    public Uri getUri() {
        return Utils.updateUserInfo( uri, null );
    }
    @Override
    public void setUri( Uri uri_ ) {
        uri = uri_;
        if( uri == null ) return;
        
        String ui = uri.getUserInfo();
        if( Utils.str( ui ) ) {
            setCredentials( new Credentials( ui ) );
            uri = Utils.updateUserInfo( uri, null );
        }
        String host = uri.getHost();
        if( !Utils.str( host ) ) return;
        String path = uri.getPath(); // JCIFS requires slash on the end to successful copy operations
        boolean ch = path == null || path.length() == 0;
        if( ch )
            path = SLS;
        else {
            ch = path.lastIndexOf( SLC ) != path.length()-1;
            if( ch ) 
                path = path + SLS;
        }
        if( ch )
            uri = uri.buildUpon().encodedPath( path ).build();
    }

    @Override
    public String getScheme() {
        return "smb";
    }

    @Override
    public boolean hasFeature( Feature feature ) {
        switch( feature ) {
        case REAL:
            return true;
        default: return super.hasFeature( feature );
        }
    }
    
    @Override
    public boolean readSource( Uri tmp_uri, String pass_back_on_done ) {
        try {
            try {
                String ba_s = null;
                WifiManager wifi = (WifiManager)commander.getContext().getSystemService( Context.WIFI_SERVICE );
                if( wifi != null && wifi.isWifiEnabled() ) {
                    InetAddress ba = getBroadcastAddress( wifi.getDhcpInfo() );
                    if( ba != null )
                        ba_s = ba.getHostAddress();
                }
                if( ba_s == null ) {
                    Log.w( TAG, "Can't get the broadcast address. Assuming /32" );
                    ba_s = "255.255.255.255";
                }
                Config.setProperty( "jcifs.netbios.baddr", ba_s );
            } catch( Exception e ) {
                Log.e( TAG, "setting the broadcast IP", e );
            }
            PackageManager pm = ctx.getPackageManager();
            Resources smb_res = pm.getResourcesForApplication( "com.ghostsq.commander.samba" );
            Utils.changeLanguage( ctx, smb_res );
            
            if( tmp_uri != null )
                setUri( tmp_uri );
            if( uri == null )
                return false;
            
            if( reader != null ) { // that's not good.
                if( reader.isAlive() ) {
                    Log.w( TAG, "Busy..." );
                    reader.interrupt();
                    Thread.sleep( 500 );
                    if( reader.isAlive() ) 
                        return false;      
                }
            }
            String host = uri.getHost();
            NtlmPasswordAuthentication use_auth = credentials;
            if( !Utils.str( host ) || use_auth == null && !Utils.str( uri.getUserInfo() ) )
                use_auth = NtlmPasswordAuthentication.ANONYMOUS;

            String uri_s = uri.toString();
            String unesc = Utils.unEscape( uri_s );            
            reader = new ListEngine( unesc, use_auth, mode, this, commander.getContext(), pass_back_on_done );
            reader.setHandler( readerHandler );
            reader.setName( TAG + ".ListEngine" );
            reader.start();
            return true;
        }
        catch( Exception e ) {
            Log.e( TAG, null, e );
        }
        catch( NoSuchFieldError e ) {
            Log.e( TAG, null, e );
            notify( "Try to install the latest version of the plugin", Commander.OPERATION_FAILED );
            return false;
        }
        notify( "Fail", Commander.OPERATION_FAILED );
        return false;
    }
    @Override
    protected void onReadComplete() {
        if( reader instanceof ListEngine ) {
            parentLink = uri == null || uri.getHost() == null || uri.getHost().length() <= 1 ? SLS : PLS;
            ListEngine list_engine = (ListEngine)reader;
            items = list_engine.getItems();
            numItems = items != null ? items.length + 1 : 1;
            reSort();
            notifyDataSetChanged();
        }
    }
    private final SmbFile[] bitsToItems( SparseBooleanArray cis ) {
        try {
            int counter = 0;
            for( int i = 0; i < cis.size(); i++ )
                if( cis.valueAt( i ) )
                    counter++;
            SmbFile[] subItems = new SmbFile[counter];
            int j = 0;
            for( int i = 0; i < cis.size(); i++ )
                if( cis.valueAt( i ) )
                    subItems[j++] = items[ cis.keyAt( i ) - 1 ].f;
            return subItems;
        } catch( Exception e ) {
            Log.e( TAG, "bitsToNames()", e );
        }
        return null;
    }
    
    @Override
    public boolean copyItems( SparseBooleanArray cis, CommanderAdapter to, boolean move ) {
        try {
            SmbFile[] subItems = bitsToItems( cis );
            if( subItems == null ) {
                notify( s( Utils.RR.copy_err.r() ), Commander.OPERATION_FAILED );
                return false;
            }
            File dest = null;
            Engines.IReciever recipient = null;
            if( move && getScheme().equals( to.getScheme() ) ) {
                Uri to_uri = to.getUri();
                if( to_uri.getHost().equalsIgnoreCase( uri.getHost() ) ) {
                    notify( Commander.OPERATION_STARTED );
                    MoveEngine me = new MoveEngine( ctx, subItems, new SmbFile( to_uri.toString(), credentials ) ); 
                    commander.startEngine( me );
                    return true;
                }
            } else if( to instanceof FSAdapter  ) {
                dest = new File( to.toString() );
            } else {
                dest = new File( createTempDir() );
                recipient = to.getReceiver(); 
            }
            notify( Commander.OPERATION_STARTED );
            Engine eng = new CopyFromEngine( commander, subItems, dest, move, recipient );
            commander.startEngine( eng );
            return true;
        }
        catch( Exception e ) {
            notify( "Failed to proceed.", Commander.OPERATION_FAILED );
        }
        return false;
    }
   
    @Override
    public boolean createFile( String fileURI ) {
        return false;
    }

    @Override
    public void createFolder( String name ) {
        notify( Commander.OPERATION_STARTED );
        commander.startEngine( new MkDirEngine( commander.getContext(), uri.toString() + SLS + name, credentials ) );
    }

    @Override
    public boolean deleteItems( SparseBooleanArray cis ) {
        try {
            SmbFile[] subItems = bitsToItems( cis );
            if( subItems != null ) {
                notify( Commander.OPERATION_STARTED );
                commander.startEngine( new DelEngine( commander.getContext(), subItems ) );
                return true;
            }
        }
        catch( Exception e ) {
            commander.showError( "Exception: " + e.getMessage() );
        }
        return false;
    }

    @Override
    public Uri getItemUri( int position ) {
        Uri u = getUri();
        if( u == null ) return null;
        return u.buildUpon().appendEncodedPath( getItemName( position, false ) ).build();
    }
    @Override
    public String getItemName( int position, boolean full ) {
        if( items != null && position > 0 && position <= items.length ) {
            if( full ) {
                Uri u = getItemUri( position );
                if( u != null ) return u.toString();
            }
            else return items[position-1].name;
        }
        return null;
    }
    
    @Override
    public void openItem( int position ) {
        if( position == 0 ) { // ..
            if( uri != null && parentLink != SLS ) {
                String path = uri.getPath();
                int len_ = path.length()-1;
                if( len_ > 0 ) {
                    if( path.charAt( len_ ) == SLC )
                        path = path.substring( 0, len_ );
                    path = path.substring( 0, path.lastIndexOf( SLC ) );
                    if( path.length() == 0 )
                        path = SLS;
                    commander.Navigate( uri.buildUpon().encodedPath( path ).build(), null, uri.getLastPathSegment() + SLS );
                }
                else {
                    commander.Navigate( Uri.parse( "smb://" ), null, null );
                }
            }
            return;
        }
        if( items == null || position < 0 || position > items.length )
            return;
        SmbItem item = items[position - 1];
        SmbFile f = item.f;
        synchronized( f ) {
            if( item.dir ) {
                String can = f.getCanonicalPath();
                String fix = SmbItem.fixName( can );
                String esc = escapeSMBcanonical( fix );
                commander.Navigate( Uri.parse( esc ), null, null );
            } else {
                Uri auth_item_uri = getUri().buildUpon().appendEncodedPath( f.getName() ).build();
                commander.Open( auth_item_uri, credentials != null ? new Credentials( credentials.getName(), credentials.getPassword() ) : null );
            }
        }
    }

    private final static String escapeSMBcanonical( String s ) {
        return s.replaceAll( "%", "%25" )
                .replaceAll( "#", "%23" )
                .replaceAll( "\\?", "%3F" )
                .replaceAll( "@", "%40" );
    }
    
    @Override
    public boolean receiveItems( String[] fileURIs, int move_mode ) {
        try {
            if( fileURIs == null || fileURIs.length == 0 ) {
                notify( s( Utils.RR.copy_err.r() ), Commander.OPERATION_FAILED );
                return false;
            }
            File[] list = Utils.getListOfFiles( fileURIs );
            if( list == null ) {
                notify( "Something wrong with the files", Commander.OPERATION_FAILED );
                return false;
            }
            notify( Commander.OPERATION_STARTED );
            String cur_str = uri.toString();
            String unesc = Utils.unEscape( cur_str );
            SmbFile cur_smb = credentials != null ? new SmbFile( unesc, credentials ) : new SmbFile( unesc );
            commander.startEngine( new CopyToEngine( commander, list, cur_smb, move_mode ) );
            return true;
        } catch( Exception e ) {
            notify( "Exception: " + e.getMessage(), Commander.OPERATION_FAILED );
        }
        return false;
    }

    @Override
    public boolean renameItem( int position, String newName, boolean copy ) {
        if( position <= 0 || position > items.length )
            return false;
        if( copy ) {
            notify( s( Utils.RR.not_supported.r() ), Commander.OPERATION_FAILED );
            return false;
        }
        notify( Commander.OPERATION_STARTED );
        commander.startEngine( new RenEngine( commander.getContext(), items[position - 1].f, 
                                uri.toString() + SLS + newName, credentials ) );
        return true;
    }

    @Override
    public void reqItemsSize( SparseBooleanArray cis ) {
        try {
            SmbFile[] list = bitsToItems( cis );
            notify( Commander.OPERATION_STARTED );
            commander.startEngine( new CalcSizesEngine( ctx, list, credentials ) );
        }
        catch(Exception e) {
        }
    }

    public final void setIdentities( String name, String pass ) {
        String domain = null;
        int dsp = name.indexOf( ';' );
        if( dsp < 0 )
            dsp = name.indexOf( '\\' );
        if( dsp >= 0 ) {
            domain = name.substring( 0, dsp );
            name = name.substring( dsp+1 );
            credentials = new NtlmPasswordAuthentication( domain, name, pass );
        }
        else
            credentials = new NtlmPasswordAuthentication( null, name, pass );  // ???
    }
    @Override
    public synchronized void setCredentials( Credentials crd ) {
        if( crd == null )
            credentials = null;
        else
            setIdentities( crd.getUserName(), crd.getPassword() );
    }
    @Override
    public Credentials getCredentials() {
        if( credentials == null ) return null;
        return new Credentials( credentials.getName(), credentials.getPassword() );
    }

    @Override
    public Object getItem( int position ) {
        if( position == 0 ) {
            Item item = new Item();
            item.name = parentLink;
            return item;
        }
        else {
            if( items != null && position > 0 && position <= items.length )
                return items[position - 1];
        }
        return null;
    }

    @Override
    protected void reSort() {
        if( items == null ) return;
        SmbItemPropComparator comp = new SmbItemPropComparator( mode & MODE_SORTING, (mode & MODE_CASE) != CASE_SENS, ascending );
        Arrays.sort( items, comp );
    }

    @Override
    public Item getItem( Uri u ) {
        try {
            if( uri != null && !uri.getHost().equals( u.getHost() ) )
                return null;
            String uri_s = u.toString();
            String unesc = Utils.unEscape( uri_s );            
            SmbFile smb_file = credentials != null ? new SmbFile( unesc, credentials ) : 
                                                     new SmbFile( unesc, NtlmPasswordAuthentication.ANONYMOUS );
            if( smb_file.exists() ) 
                return new SmbItem( smb_file, 0, null );
        } catch( Throwable e ) {
            e.printStackTrace();
        }
        return null;
    }
    
    @Override
    public InputStream getContent( Uri u, long offset ) {
        try {
            if( uri != null && !uri.getHost().equals( u.getHost() ) )
                return null;
            //Log.d( TAG, "Content for " + u + " offset=" + offset );
            
            String uri_s = u.toString();
            String unesc = Utils.unEscape( uri_s );            
            
            SmbFile smb_file = credentials != null ? new SmbFile( unesc, credentials ) : 
                                                     new SmbFile( unesc, NtlmPasswordAuthentication.ANONYMOUS );
            smb_file.connect();
            if( smb_file.exists() && smb_file.isFile() ) {
                InputStream is = smb_file.getInputStream();
                if( offset > 0 )
                    is.skip( offset );
                return is;
            }
        } catch( Throwable e ) {
            Log.e( TAG, Utils.getCause( e ), e );
        }
        return null;
    }

    @Override
    public OutputStream saveContent( Uri u ) {
        try {
            if( uri != null && !uri.getHost().equals( u.getHost() ) )
                return null;
            String uri_s = u.toString();
            String unesc = Utils.unEscape( uri_s );            
            SmbFile smb_file = new SmbFile( unesc, credentials );
            smb_file.connect();
            if( smb_file.exists() && smb_file.isFile() )
                return smb_file.getOutputStream();
        } catch( Throwable e ) {
            e.printStackTrace();
        }
        return null;
    }
    
    @Override
    public void closeStream( Closeable s ) {
        try {
            s.close();
        } catch( IOException e ) {
            e.printStackTrace();
        }
    }
    static SMBAdapter createInstance() {
        return new SMBAdapter();
    }
    
    @Override
    public IReciever getReceiver() {
        return this;
    }
}
