/*
 * Copyright (c) 2016 Marien Raat <marienraat@riseup.net>
 * Copyright (c) 2017 Stephen Michel <s@smichel.me>
 * SPDX-License-Identifier: GPL-3.0+
 */

package com.jmstudios.redmoon.filter

import android.content.ComponentName
import android.content.Intent

import com.jmstudios.redmoon.util.*

private const val DURATION_LONG = 1000f // One second
private const val DURATION_SHORT = 250f
private const val DURATION_INSTANT = 0f

enum class Command(val time: Float) {
    ON(DURATION_LONG) {
        override val turnOn: Boolean = true
        override fun onAnimationStart(service: FilterService) {
            service.start(true)
        }
    },
    OFF(DURATION_LONG) {
        override val turnOn: Boolean = false
        override fun onAnimationStart(service: FilterService) {
            service.stopForeground(true)
        }
        override fun onAnimationEnd(service: FilterService) {
            service.stopSelf()
        }
    },
    PAUSE(DURATION_SHORT) {
        override val turnOn: Boolean = false
        override fun onAnimationStart(service: FilterService) {
            service.start(false)
        }
    },
    RESUME(DURATION_SHORT) {
        override val turnOn: Boolean = true
        override fun onAnimationStart(service: FilterService) {
            service.start(true)
        }
    },
    SHOW_PREVIEW(DURATION_INSTANT) {
        override val turnOn: Boolean = true
        override fun onAnimationStart(service: FilterService) {
            if (filterWasOn == null) {
                filterWasOn = filterIsOn
            }
            service.start(true)
        }
    },
    HIDE_PREVIEW(DURATION_INSTANT) {
        override val turnOn: Boolean
            get() = filterWasOn == true

        override fun onAnimationEnd(service: FilterService) {
            if (filterWasOn != true) {
                service.stopForeground(true)
                service.stopSelf()
                filterWasOn = null
            }
        }
    };

    val intent: Intent
        get() = intent(FilterService::class).putExtra(EXTRA_COMMAND, name)

    fun send(): ComponentName? = appContext.startService(intent)

    abstract val turnOn: Boolean

    //override fun toString(): String = javaClass.simpleName

    open fun onAnimationStart(service: FilterService) {}
    open fun onAnimationCancel(service: FilterService) {}
    open fun onAnimationEnd(service: FilterService) {}
    open fun onAnimationRepeat(service: FilterService) {}

    companion object : Logger() {
        private const val EXTRA_COMMAND = "jmstudios.bundle.key.command"

        private var filterWasOn: Boolean? = null

        fun getCommand(intent: Intent): Command {
            val commandName = intent.getStringExtra(EXTRA_COMMAND)
            Log.i("Recieved flag: $commandName")
            return valueOf(commandName)
        }

        fun toggle(on: Boolean) {
            if (on) ON.send() else OFF.send()
        }

        fun preview(on: Boolean) {
            if (on) SHOW_PREVIEW.send() else HIDE_PREVIEW.send()
        }
    }
}
