/*******************************************************************************
 * Copyright 2011 See AUTHORS file.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ******************************************************************************/

package com.badlogic.gdx.scenes.scene2d;

import com.badlogic.gdx.math.Vector2;

/** Listener for actor input events. */
public class InputListener implements EventListener {
	public boolean handle (Event e) {
		if (!(e instanceof InputEvent)) return false;
		InputEvent event = (InputEvent)e;

		switch (event.getType()) {
		case keyDown:
			return keyDown(event, event.getKeyCode());
		case keyUp:
			return keyUp(event, event.getKeyCode());
		case keyTyped:
			return keyTyped(event, event.getCharacter());
		}

		Vector2 coords = Vector2.tmp.set(event.getStageX(), event.getStageY());
		event.getListenerActor().stageToLocalCoordinates(coords);

		switch (event.getType()) {
		case touchDown:
			return touchDown(event, coords.x, coords.y, event.getPointer(), event.getButton());
		case touchUp:
			touchUp(event, coords.x, coords.y, event.getPointer(), event.getButton());
			return true;
		case touchDragged:
			touchDragged(event, coords.x, coords.y, event.getPointer());
			return true;
		case mouseMoved:
			return mouseMoved(event, coords.x, coords.y);
		case scrolled:
			return scrolled(event, coords.x, coords.y, event.getScrollAmount());
		case enter:
			enter(event, coords.x, coords.y, event.getPointer(), event.getRelatedActor());
			return false;
		case exit:
			exit(event, coords.x, coords.y, event.getPointer(), event.getRelatedActor());
			return false;
		}
		return false;
	}

	/** Called when a mouse button or a finger touch goes down on the actor. If true is returned, this listener will receive all
	 * touchDragged and touchUp events, even those not over this actor, until touchUp is received. Also when true is returned, the
	 * event is {@link Event#handle() handled}.
	 * @see InputEvent */
	public boolean touchDown (InputEvent event, float x, float y, int pointer, int button) {
		return false;
	}

	/** Called when a mouse button or a finger touch goes up anywhere, but only if touchDown previously returned true for the mouse
	 * button or touch. The touchUp event is always {@link Event#handle() handled}.
	 * @see InputEvent */
	public void touchUp (InputEvent event, float x, float y, int pointer, int button) {
	}

	/** Called when a mouse button or a finger touch is moved anywhere, but only if touchDown previously returned true for the mouse
	 * button or touch. The touchDragged event is always {@link Event#handle() handled}.
	 * @see InputEvent */
	public void touchDragged (InputEvent event, float x, float y, int pointer) {
	}

	/** Called any time the mouse is moved when a button is not down. This event only occurs on the desktop. When true is returned,
	 * the event is {@link Event#handle() handled}.
	 * @see InputEvent */
	public boolean mouseMoved (InputEvent event, float x, float y) {
		return false;
	}

	/** Called any time the mouse cursor or a finger touch is moved over an actor. On the desktop, this event occurs even when no
	 * mouse buttons are pressed (pointer will be -1).
	 * @see InputEvent */
	public void enter (InputEvent event, float x, float y, int pointer, Actor fromActor) {
	}

	/** Called any time the mouse cursor or a finger touch is moved out of an actor. On the desktop, this event occurs even when no
	 * mouse buttons are pressed (pointer will be -1).
	 * @see InputEvent */
	public void exit (InputEvent event, float x, float y, int pointer, Actor toActor) {
	}

	/** Called when the mouse wheel has been scrolled. When true is returned, the event is {@link Event#handle() handled}. */
	public boolean scrolled (InputEvent event, float x, float y, int amount) {
		return false;
	}

	/** Called when a key goes down. When true is returned, the event is {@link Event#handle() handled}. */
	public boolean keyDown (InputEvent event, int keycode) {
		return false;
	}

	/** Called when a key goes up. When true is returned, the event is {@link Event#handle() handled}. */
	public boolean keyUp (InputEvent event, int keycode) {
		return false;
	}

	/** Called when a key is typed. When true is returned, the event is {@link Event#handle() handled}. */
	public boolean keyTyped (InputEvent event, char character) {
		return false;
	}
}
