// @flow

import { NativeModules, Vibration, Platform } from 'react-native';

/**
 * Notification Feedback Type
 * The type of notification generated by a UINotificationFeedbackGenerator object.
 * https://developer.apple.com/documentation/uikit/uinotificationfeedbacktype
 */
export type NotificationFeedbackType = $Enum<{
  /**
   * A notification feedback type, indicating that a task has completed successfully.
   */
  success: string,
  /**
   * A notification feedback type, indicating that a task has produced a warning.
   */
  warning: string,
  /**
   * A notification feedback type, indicating that a task has failed.
   */
  error: string,
}>;

/**
 * Notification Feedback Type
 * Convenient constants
 */
export const NotificationTypes = {
  Success: 'success',
  Warning: 'warning',
  Error: 'error',
};

/**
 * Impact Feedback Style
 * The mass of the objects in the collision simulated by a UIImpactFeedbackGenerator object.
 * https://developer.apple.com/documentation/uikit/uiimpactfeedbackstyle
 */
export type ImpactFeedbackStyle = $Enum<{
  /**
   * A collision between small, light user interface elements.
   */
  light: string,
  /**
   * A collision between moderately sized user interface elements.
   */
  medium: string,
  /**
   * A collision between large, heavy user interface elements.
   */
  heavy: string,
}>;

/**
 * Impact Feedback Style
 * Convenient constants
 */
export const ImpactStyles = {
  Light: 'light',
  Medium: 'medium',
  Heavy: 'heavy',
};

/**
 * Triggers notification feedback.
 */
export function notification(type?: NotificationFeedbackType) {
  if (Platform.OS === 'ios') {
    NativeModules.ExponentHaptic.notification(type || 'success');
  } else {
    console.warn('`Expo.Haptic` is only available on iOS');
    Vibration.vibrate();
  }
}

/**
 * Triggers impact feedback.
 */
export function impact(style?: ImpactFeedbackStyle) {
  if (Platform.OS === 'ios') {
    NativeModules.ExponentHaptic.impact(style || 'medium');
  } else {
    console.warn('`Expo.Haptic` is only available on iOS');
    Vibration.vibrate();
  }
}

/**
 * Triggers selection feedback.
 */
export function selection() {
  if (Platform.OS === 'ios') {
    NativeModules.ExponentHaptic.selection();
  } else {
    console.warn('`Expo.Haptic` is only available on iOS');
    Vibration.vibrate();
  }
}
