
package com.matt.outfield.controller

import kotlin.coroutines.CoroutineContext
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.launch

import net.twasi.obsremotejava.OBSRemoteController
import net.twasi.obsremotejava.callbacks.Callback
import net.twasi.obsremotejava.events.responses.SwitchScenesResponse
import net.twasi.obsremotejava.requests.GetCurrentScene.GetCurrentSceneResponse
import net.twasi.obsremotejava.requests.GetSceneList.GetSceneListResponse
import net.twasi.obsremotejava.requests.GetStreamingStatus.GetStreamingStatusResponse
import net.twasi.obsremotejava.requests.TakeSourceScreenshot.TakeSourceScreenshotResponse
import net.twasi.obsremotejava.requests.ResponseBase

import com.matt.outfield.util.buildWeakSet

public class OBSController(val config : OBSConfig)
        : CoroutineScope {

    override val coroutineContext: CoroutineContext
        get() = Dispatchers.IO + job

    private val job : Job = Job()

    private var obsRemote : OBSRemoteController? = null
    private var isConnected : Boolean = false

    /**
     * @param hostname hostname with port e.g. ws://1.2.3.4:4444"
     * @param connection password or null
     */
    public class OBSConfig(val hostname : String,
                           val password : String?)

    public interface OBSListener {
        fun onConnect() { }
        fun onConnectionFailed(message : String) { }
        fun onDisconnect() { }
        fun onConnectError(message : String, e : Throwable) { }
        fun onRecordingStatusChange(on : Boolean) { }
        fun onStreamingStatusChange(on : Boolean) { }
        fun onSwitchScenes(sceneName : String?) { }
    }

    val listeners : MutableSet<OBSListener> = buildWeakSet()

    init {
        val conn = getConnection()
        conn.registerSwitchScenesCallback({ res : ResponseBase? ->
            val sceneRes = res as? SwitchScenesResponse
            if (sceneRes != null) {
                notifyListeners({ listener ->
                    listener.onSwitchScenes(sceneRes.getSceneName())
                })
            }
        })
        conn.registerRecordingStartedCallback({ _ ->
            notifyListeners({ listener ->
                listener.onRecordingStatusChange(true)
            })
        })
        conn.registerRecordingStoppedCallback({ _ ->
            notifyListeners({ listener ->
                listener.onRecordingStatusChange(false)
            })
        })
        conn.registerStreamStartedCallback({ _ ->
            notifyListeners({ listener ->
                listener.onStreamingStatusChange(true)
            })
        })
        conn.registerStreamStoppedCallback({ _ ->
            notifyListeners({ listener ->
                listener.onStreamingStatusChange(false)
            })
        })
        conn.registerConnectCallback({ _ ->
            isConnected = true
            notifyListeners({ listener ->
                listener.onConnect()
            })
        })
        conn.registerDisconnectCallback({ _ ->
            isConnected = false
            notifyListeners({ listener ->
                listener.onDisconnect()
            })
        })
        conn.registerConnectionFailedCallback({ message ->
            isConnected = false
            notifyListeners({ listener ->
                listener.onConnectionFailed(message)
            })
        })
        conn.registerOnError({ message : String, e : Throwable ->
            isConnected = false
            notifyListeners({ listener ->
                listener.onConnectError(message, e)
            })
        })
    }

    public fun connect() {
        getConnection().connect()
    }

    public fun disconnect(onError : () -> Unit = { }) {
        job.cancel()
        dispatchObs(onError, { conn ->
            conn.disconnect()
        })
    }

    public fun getIsConnected() : Boolean {
        return isConnected
    }

    public fun addListener(listener : OBSListener) {
        listeners.add(listener)
    }

    public fun removeListener(listener : OBSListener) {
        listeners.remove(listener)
    }

    public fun switchScene(sceneName : String,
                           onError : () -> Unit = { }) {
        dispatchObs(onError, { conn ->
            conn.setCurrentScene(sceneName, { _ -> })
        })
    }

    public fun getCurrentScene(callback : (String) -> Unit,
                               onError : () -> Unit = { }) {
        dispatchObs(onError, { conn ->
            conn.getCurrentScene({res : ResponseBase? ->
                val sceneResponse = res as? GetCurrentSceneResponse
                val name = sceneResponse?.getName()
                if (name != null)
                    dispatchMain { callback(name) }
                else
                    onError()
            })
        })
    }

    public fun getScenes(callback : (List<String>) -> Unit,
                         onError : () -> Unit = { }) {
        dispatchObs(onError, { conn ->
            conn.getScenes({ res : ResponseBase? ->
                val scenesResponse = res as? GetSceneListResponse
                val scenes = mutableListOf<String>()
                val gotScenes = scenesResponse?.getScenes()
                if (gotScenes != null) {
                    for (scene in gotScenes)
                        scenes.add(scene.getName())
                    dispatchMain { callback(scenes) }
                } else {
                    onError()
                }
            })
        })
    }

    /**
     * Get whether recording or streaming
     *
     * @param callback function(isStreaming, isRecording)
     * @param onError callback if error occurred
     */
    public fun getRecordingStreamingStatus(
        callback : (Boolean, Boolean) -> Unit,
        onError : () -> Unit = { }
    ) {
        dispatchObs(onError, { conn ->
            conn.getStreamingStatus({ res ->
                val statusRes = res as? GetStreamingStatusResponse
                if (statusRes != null) {
                    dispatchMain {
                        callback(statusRes.isStreaming(),
                                 statusRes.isRecording())
                    }
                }
            })
        })
    }

    public fun startStreaming(onError : () -> Unit = { }) {
        dispatchObs(onError, { conn ->
            conn.startStreaming({ _ -> })
        })
    }

    public fun stopStreaming(onError : () -> Unit = { }) {
        dispatchObs(onError, { conn ->
            conn.stopStreaming({ _ -> })
        })
    }

    public fun startRecording(onError : () -> Unit = { }) {
        dispatchObs(onError, { conn ->
            conn.startRecording({ _ -> })
        })
    }

    public fun stopRecording(onError : () -> Unit = { }) {
        dispatchObs(onError, { conn ->
            conn.stopRecording({ _ -> })
        })
    }

    /**
     * Get a screenshot of a source or scene (scenes are sources)
     *
     * @param sourceName name of source or scene
     * @param imgFormat png, jpg, jpeg, or any supported format
     * @param callback with image string in base64
     * @param onError callback if error
     */
    public fun takeSourceScreenshot(sourceName : String,
                                    imgFormat : String,
                                    callback : (String) -> Unit,
                                    onError : () -> Unit = { }) {
        dispatchObs(onError, { conn ->
            conn.takeSourceScreenshot(sourceName,
                                                   imgFormat,
                                                   { res ->
                val shotRes = res as? TakeSourceScreenshotResponse
                if (shotRes != null) {
                    val img = shotRes.getImg()
                    if (img != null)
                        dispatchMain { callback(img) }
                    else
                        onError()
                }
            })
        })
    }

    private fun getConnection() : OBSRemoteController {
        val curRemote = obsRemote
        if (curRemote != null)
            return curRemote

        val controller = OBSRemoteController(config.hostname,
                                             false,
                                             config.password,
                                             false);

        obsRemote = controller

        return controller
    }

    /**
     * Make an OBS call on the IO thread
     *
     * @param onError callback if an error occurred
     * @param run command to run
     */
    private fun dispatchObs(onError : () -> Unit,
                            run : (conn : OBSRemoteController) -> Unit) {
        launch {
            if (isConnected) {
                val conn = getConnection()
                if (conn.isFailed())
                    onError()
                else
                    run(conn)
            } else {
                onError()
            }
        }
    }

    private fun dispatchMain(run : () -> Unit) {
        launch (Dispatchers.Main) { run() }
    }

    private fun notifyListeners(f : (OBSListener) -> Unit) {
        dispatchMain {
            for (listener in listeners)
                f(listener)
        }
    }

}
