/*
* Copyright (C) 2017 Yonnji Nyyoka, yonnji@miqote.com
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 3
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/
package com.miqote.shanawp;

import android.content.SharedPreferences;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.Picture;
import android.graphics.Rect;
import android.os.Handler;
import android.service.wallpaper.WallpaperService;
import android.view.SurfaceHolder;
import android.util.Log;
import com.larvalabs.svgandroid.*;

public class LiveWallpaperService extends WallpaperService {

    @Override
    public void onCreate() {
        super.onCreate();
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
    }

    @Override
    public Engine onCreateEngine() {
        return new LiveWallpaperEngine();
    }

    public class LiveWallpaperEngine extends Engine implements Runnable,
                                                               SharedPreferences.OnSharedPreferenceChangeListener {
        public static final String TAG = "SHANAWP";
        private Handler handler = new Handler();
        private SharedPreferences preferences;
        private String view = "close";
        private boolean isVisible = true;
        private Rect background = null;
        private Rect foreground = null;
        private Picture layerSeireiden;
        private Picture layerBody;
        private Picture layerEyes;
        private Picture layerEyesClosed;
        private Bitmap[] layerHair = null;
        private Bitmap[][] sparksFrames = null;
        private Picture layerHairBase;
        private Picture layerClothes;
        private Spark[] sparks = null;
        private int hairID = 0;
        private int eyesID = 0;

        public LiveWallpaperEngine() {
            preferences = LiveWallpaperService.this.getSharedPreferences("settings", 0);
            preferences.registerOnSharedPreferenceChangeListener(this);
            onSharedPreferenceChanged(preferences, null);
            layerSeireiden = getPicture(R.raw.seireiden);
            layerBody = getPicture(R.raw.body);
            layerEyes = getPicture(R.raw.eyes);
            layerEyesClosed = getPicture(R.raw.eyes_closed);
            layerHairBase = getPicture(R.raw.hair_base);
            layerClothes = getPicture(R.raw.clothes);
        }

        private Bitmap getBitmap(int resource) {
            final int IMAGE_MAX_SIZE = 100000;
            // Decode image size
            BitmapFactory.Options o = new BitmapFactory.Options();
            o.inJustDecodeBounds = true;
            BitmapFactory.decodeResource(getResources(), resource, o);
            int scale = 1;
            while ((o.outWidth * o.outHeight) / Math.pow(scale, 2) > IMAGE_MAX_SIZE) {
                scale++;
            }
            Log.d(TAG, "scale = " + scale + ", orig-width: " + o.outWidth + ", orig-height: " + o.outHeight);
            Bitmap b = null;
            o = new BitmapFactory.Options();
            if (scale > 1) {
                scale--;
                // scale to max possible inSampleSize that still yields an image
                // larger than target
                o.inSampleSize = scale;
                b = BitmapFactory.decodeResource(getResources(), resource, o);
                // resize to desired dimensions
                int height = b.getHeight();
                int width = b.getWidth();
                Log.d(TAG, "1th scale operation dimenions - width: " + width + ", height: " + height);
                double y = Math.sqrt(IMAGE_MAX_SIZE / (((double) width) / height));
                double x = (y / height) * width;
                Bitmap scaledBitmap = Bitmap.createScaledBitmap(b, (int) x, (int) y, true);
                b.recycle();
                b = scaledBitmap;
                System.gc();
            } else {
                b = BitmapFactory.decodeResource(getResources(), resource);
            }
            Log.d(TAG, "bitmap size - width: " + b.getWidth() + ", height: " + b.getHeight());
            return b;
        }

        private Picture getPicture(int resource) {
            SVG svg = SVGParser.getSVGFromResource(getResources(), resource);
            return svg.getPicture();
        }

        @Override
        public void onCreate(SurfaceHolder surfaceHolder) {
            super.onCreate(surfaceHolder);
            if (layerHair == null) {
                if (isPreview()) {
                    layerHair = new Bitmap[] {
                        getBitmap(R.drawable.hair_h5617),
                    };
                } else {
                    layerHair = new Bitmap[] {
                        getBitmap(R.drawable.hair_h5617),
                        getBitmap(R.drawable.hair_h5622),
                        getBitmap(R.drawable.hair_h5627),
                        getBitmap(R.drawable.hair_h5632),
                        getBitmap(R.drawable.hair_h5637),
                        getBitmap(R.drawable.hair_h5642),
                        getBitmap(R.drawable.hair_h5647),
                        getBitmap(R.drawable.hair_h5652),
                        getBitmap(R.drawable.hair_h5657),
                        getBitmap(R.drawable.hair_h5662),
                        getBitmap(R.drawable.hair_h5667),
                        getBitmap(R.drawable.hair_h5672),
                        getBitmap(R.drawable.hair_h5677),
                        getBitmap(R.drawable.hair_h5682),
                        getBitmap(R.drawable.hair_h5687)
                    };
                }
            }
            if (sparksFrames == null) {
                sparksFrames = new Bitmap[][] {
                    {
                        getBitmap(R.drawable.spark1a),
                        getBitmap(R.drawable.spark1b),
                        getBitmap(R.drawable.spark1c)
                    }, {
                        getBitmap(R.drawable.spark2a),
                        getBitmap(R.drawable.spark2b)
                    }, {
                        getBitmap(R.drawable.spark3a),
                        getBitmap(R.drawable.spark3b),
                        getBitmap(R.drawable.spark3c)
                    }, {
                        getBitmap(R.drawable.spark4a),
                        getBitmap(R.drawable.spark4b),
                        getBitmap(R.drawable.spark4c)
                    }, {
                        getBitmap(R.drawable.spark5a),
                        getBitmap(R.drawable.spark5b),
                        getBitmap(R.drawable.spark5c)
                    }
                };
            }
        }

        @Override
        public void onDestroy() {
            super.onDestroy();
            handler.removeCallbacks(this);
            if (layerHair != null) {
                for (int i = 0; i < layerHair.length; i++) {
                    layerHair[i].recycle();
                    layerHair[i] = null;
                }
                layerHair = null;
            }
            if (sparksFrames != null) {
                for (int i = 0; i < sparksFrames.length; i++) {
                    for (int j = 0; j < sparksFrames[i].length; j++) {
                        sparksFrames[i][j].recycle();
                        sparksFrames[i][j] = null;
                    }
                }
                sparksFrames = null;
            }

        }

        @Override
        public void onVisibilityChanged(boolean visible) {
            isVisible = visible;
            if (isVisible) {
                run();
            } else {
                handler.removeCallbacks(this);
            }
        }

        @Override
        public void onSurfaceCreated(SurfaceHolder holder) {
            super.onSurfaceCreated(holder);
        }

        @Override
        public void onSurfaceDestroyed(SurfaceHolder holder) {
            super.onSurfaceDestroyed(holder);
            isVisible = false;
            handler.removeCallbacks(this);
        }

        @Override
        public void onSurfaceChanged(SurfaceHolder holder, int format, int width, int height) {
            super.onSurfaceChanged(holder, format, width, height);
        }

        @Override
        public void onOffsetsChanged(float xOffset, float yOffset,
                                     float xStep, float yStep, int xPixels, int yPixels) {
        }

        @Override
        public final void onSharedPreferenceChanged(SharedPreferences prefs, String key) {
            view = prefs.getString("view", "close");
            foreground = null;
            sparks = null;
        }

        private void doDraw(Canvas canvas) {
            if (background == null) {
                int x = (int) (canvas.getClipBounds().left + (canvas.getWidth() - canvas.getHeight()) / 2f);
                int y = canvas.getClipBounds().top;
                background = new Rect(x, y, x + canvas.getHeight(), y + canvas.getHeight());
            }
            if (foreground == null) {
                foreground = background;
                if (view.equals("far")) {
                    int x = canvas.getClipBounds().left;
                    int y = canvas.getHeight() - canvas.getWidth();
                    foreground = new Rect(x, y, x + canvas.getWidth(), y + canvas.getWidth());
                } else if (view.equals("medium")) {
                    int x = (int) (canvas.getClipBounds().left + (canvas.getWidth() - canvas.getHeight() / 3f * 2f) / 2f);
                    int y = (int) (canvas.getHeight() / 3f);
                    foreground = new Rect(x, y, (int) (x + canvas.getHeight() / 3f * 2f), (int) (y + canvas.getHeight() / 3f * 2f));
                }
            }
            if (sparks == null) {
                sparks = new Spark[20 + 25 + 15 + 30 + 35];
                int[] packs = new int[] {
                    20, 25, 15, 30, 35
                };
                for (int pack = 0, i = 0; pack < packs.length; pack++) {
                    for (int j = 0; j < packs[pack]; j++, i++) {
                        sparks[i] = new Spark(sparksFrames[pack],
                                              foreground.width() / 20, foreground.height() / 20);
                        sparks[i].setPosition((float) (foreground.left + foreground.width() * Math.random()),
                                              (float) (foreground.top + foreground.height() * Math.random()));
                    }
                }
            }
            canvas.drawPicture(layerSeireiden, background);
            canvas.drawPicture(layerBody, foreground);
            if (eyesID >= 100) {
                eyesID = 0;
                canvas.drawPicture(layerEyesClosed, foreground);
            } else {
                eyesID++;
                canvas.drawPicture(layerEyes, foreground);
            }
            canvas.drawBitmap(layerHair[hairID], null, foreground, null);
            if (hairID >= layerHair.length - 1) {
                hairID = 0;
            } else {
                hairID++;
            }
            canvas.drawPicture(layerHairBase, foreground);
            canvas.drawPicture(layerClothes, foreground);
            for (Spark spark: sparks) {
                if (spark.getX() < foreground.left || spark.getX() > (foreground.left + foreground.width()) ||
                    spark.getY() > (foreground.top + foreground.height())) {
                    spark.reset();
                    spark.setPosition((float) (foreground.left + (foreground.width() / 5) +
                                               (foreground.width() - foreground.width() / 5 * 2) * Math.random()),
                                      (float) (foreground.top - foreground.height() / 10));
                }
                spark.draw(canvas);
            }
        }

        @Override
        public final void run() {
            SurfaceHolder holder = getSurfaceHolder();
            Canvas canvas = null;
            try {
                canvas = holder.lockCanvas();
                if (canvas != null) {
                    doDraw(canvas);
                }
            } finally {
                if (canvas != null) {
                    holder.unlockCanvasAndPost(canvas);
                }
            }
            handler.removeCallbacks(this);
            if (isVisible) {
                handler.postDelayed(this, 1000 / 20);
            }
        }
    }
}
