"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const p_map_1 = __importDefault(require("p-map"));
const state_constants_1 = require("./constants/state.constants");
const listr_interface_1 = require("./interfaces/listr.interface");
const task_1 = require("./lib/task");
const task_wrapper_1 = require("./lib/task-wrapper");
const renderer_1 = require("./utils/renderer");
class Listr {
    constructor(task, options) {
        this.task = task;
        this.options = options;
        this.tasks = [];
        this.err = [];
        this.options = Object.assign({
            showSubtasks: true,
            concurrent: false,
            renderer: 'default',
            nonTTYRenderer: 'verbose',
            exitOnError: true,
            collapse: true,
            collapseSkips: true,
            clearOutput: false
        }, options);
        this.concurrency = 1;
        if (this.options.concurrent === true) {
            this.concurrency = Infinity;
        }
        else if (typeof this.options.concurrent === 'number') {
            this.concurrency = this.options.concurrent;
        }
        this.rendererClass = renderer_1.getRenderer(this.options.renderer, this.options.nonTTYRenderer);
        this.add(task || []);
        process.on('SIGINT', async () => {
            await Promise.all(this.tasks.map(async (task) => {
                if (task.isPending()) {
                    task.state$ = state_constants_1.stateConstants.FAILED;
                }
            }));
            this.renderer.end(new Error('Interrupted.'));
            process.exit(127);
        }).setMaxListeners(0);
    }
    add(task) {
        const tasks = Array.isArray(task) ? task : [task];
        tasks.forEach((task) => {
            this.tasks.push(new task_1.Task(this, task, this.options));
        });
    }
    async run(context) {
        var _a;
        if (!this.renderer) {
            this.renderer = new this.rendererClass(this.tasks, this.options);
        }
        this.renderer.render();
        context = context || ((_a = this.options) === null || _a === void 0 ? void 0 : _a.ctx) || Object.create({});
        const errors = [];
        await this.checkAll(context);
        try {
            await p_map_1.default(this.tasks, async (task) => {
                await this.checkAll(context);
                return this.runTask(task, context, errors);
            }, { concurrency: this.concurrency });
            if (errors.length > 0) {
                const err = new listr_interface_1.ListrError('Something went wrong');
                err.errors = errors;
                throw err;
            }
            this.renderer.end();
        }
        catch (error) {
            error.context = context;
            this.renderer.end(error);
            if (this.options.exitOnError !== false) {
                throw error;
            }
        }
        return context;
    }
    checkAll(context) {
        return Promise.all(this.tasks.map((task) => {
            task.check(context);
        }));
    }
    runTask(task, context, errors) {
        if (!task.isEnabled()) {
            return Promise.resolve();
        }
        return new task_wrapper_1.TaskWrapper(task, errors).run(context);
    }
}
exports.Listr = Listr;
//# sourceMappingURL=listr.js.map