/*
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the LICENSE
 * file in the root directory of this source tree.
 */
#ifndef HERMES_PUBLIC_RUNTIMECONFIG_H
#define HERMES_PUBLIC_RUNTIMECONFIG_H

#include "hermes/Public/CrashManager.h"
#include "hermes/Public/CtorConfig.h"
#include "hermes/Public/GCConfig.h"

#include <memory>

#ifdef HERMESVM_SERIALIZE
#include <vector>

namespace llvm {
class MemoryBuffer;
class raw_ostream;
} // namespace llvm
#endif

namespace hermes {
namespace vm {

class PinnedHermesValue;
#ifdef HERMESVM_SERIALIZE
class Serializer;
class Deserializer;
#endif

// Parameters for Runtime initialisation.  Check documentation in README.md
#define RUNTIME_FIELDS_BASE(F)                                         \
  /* Parameters to be passed on to the GC. */                          \
  F(vm::GCConfig, GCConfig)                                            \
                                                                       \
  /* Pre-allocated Register Stack */                                   \
  F(PinnedHermesValue *, RegisterStack, nullptr)                       \
                                                                       \
  /* Register Stack Size */                                            \
  F(unsigned, MaxNumRegisters, 1024 * 1024)                            \
                                                                       \
  /* Whether or not the JIT is enabled */                              \
  F(bool, EnableJIT, false)                                            \
                                                                       \
  /* Whether to allow eval and Function ctor */                        \
  F(bool, EnableEval, true)                                            \
                                                                       \
  /* Whether to verify the IR generated by eval and Function ctor */   \
  F(bool, VerifyEvalIR, false)                                         \
                                                                       \
  /* Support for ES6 Symbol. */                                        \
  F(bool, ES6Symbol, true)                                             \
                                                                       \
  /* Trace non-deterministic JS behavior */                            \
  F(bool, TraceEnvironmentInteractions, false)                         \
                                                                       \
  /* Enable sampling certain statistics. */                            \
  F(bool, EnableSampledStats, false)                                   \
                                                                       \
  /* Whether to enable sampling profiler */                            \
  F(bool, EnableSampleProfiling, false)                                \
                                                                       \
  /* Whether to randomize stack placement etc. */                      \
  F(bool, RandomizeMemoryLayout, false)                                \
                                                                       \
  /* Eagerly read bytecode into page cache. */                         \
  F(unsigned, BytecodeWarmupPercent, 0)                                \
                                                                       \
  /* Signal-based I/O tracking. Slows down execution. If enabled, */   \
  /* all bytecode buffers > 64 kB passed to Hermes must be mmap:ed. */ \
  F(bool, TrackIO, false)                                              \
                                                                       \
  /* An interface for managing crashes. */                             \
  F(std::shared_ptr<CrashManager>, CrashMgr, new NopCrashManager)      \
                                                                       \
  /* The flags passed from a VM experiment */                          \
  F(uint32_t, VMExperimentFlags, 0)                                    \
  /* RUNTIME_FIELDS END */

#ifdef HERMESVM_SERIALIZE
using ExternalPointersVectorFunction = std::vector<void *>();
#define RUNTIME_FIELDS_SD(F)                                             \
  /* Should serialize after initialization */                            \
  F(std::shared_ptr<llvm::raw_ostream>, SerializeAfterInitFile, nullptr) \
  /* Should deserialize instead of initialization */                     \
  F(std::shared_ptr<llvm::MemoryBuffer>, DeserializeFile, nullptr)       \
  /* A function to get pointer values not visible to Runtime. e.g.       \
   * function pointers defined in ConsoleHost*/                          \
  F(ExternalPointersVectorFunction *, ExternalPointersVectorCallBack, nullptr)

#define RUNTIME_FIELDS(F) \
  RUNTIME_FIELDS_BASE(F)  \
  RUNTIME_FIELDS_SD(F)
#else // ifndef HERMESVM_SERIALIZE
#define RUNTIME_FIELDS(F) RUNTIME_FIELDS_BASE(F)
#endif // HERMESVM_SERIALIZE

_HERMES_CTORCONFIG_STRUCT(RuntimeConfig, RUNTIME_FIELDS, {});

#undef RUNTIME_FIELDS

} // namespace vm
} // namespace hermes

#endif // HERMES_PUBLIC_RUNTIMECONFIG_H
