/*
 * Copyright (C) 2012 Mykola Kondratenko
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.mykola.lexinproject.data;

import java.util.ArrayList;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteException;
import android.database.sqlite.SQLiteOpenHelper;
import android.os.AsyncTask;

import com.mykola.lexinproject.providers.TranslationManager;
import com.mykola.lexinproject.utils.LogUtils;

public class DictionaryDatabase extends SQLiteOpenHelper {
	private final static String TAG = "DictionaryDatabase";
	private static DictionaryDatabase mInstance = null;
	private static final int DATABASE_VERSION = 6;
	private static final String DATABASE_NAME = "lexinDictionary";
	private static final String TABLE_DICTIONARY = "dictionary";
	private static final String INDEX_DICTIONARY_1 = "idx_dictionary_1";
	private static final String KEY_ID = "_id";
	private static final String KEY_LANGUAGE_ID = "language";
	private static final String KEY_TYPE_ID = "type";
	private static final String FIELD_TRANSLATION = "translation";
	private static final String FIELD_WORD = "value";
	private static final String FIELD_ATTACHMENTS = "attachments";
	private static final String PRAGMA_ENCODING = "PRAGMA encoding ='utf-8'";
	private static final String CREATE_DICTIONARY_TABLE = "create table if not exists "
			+ TABLE_DICTIONARY
			+ "("
			+ KEY_ID
			+ " integer primary key,"
			+ KEY_LANGUAGE_ID
			+ " integer,"
			+ KEY_TYPE_ID
			+ " integer,"
			+ FIELD_WORD
			+ " text,"
			+ FIELD_TRANSLATION
			+ " text,"
			+ FIELD_ATTACHMENTS + " text)";
	private static final String CREATE_DICTIONARY_INDEX = "create index if not exists "
			+ INDEX_DICTIONARY_1
			+ " on "
			+ TABLE_DICTIONARY
			+ " ("
			+ KEY_LANGUAGE_ID
			+ " asc,"
			+ KEY_TYPE_ID
			+ " asc,"
			+ FIELD_WORD
			+ " asc)";

	private AsyncTask<?, ?, ?> mTask;

	public interface IListener {
		public void responseAdd(Translation translation);

		public void responseGet(Translation translation);

		public void responseGetAll(ArrayList<Translation> translations);

		public void responseDelete();

		public void responseGetAllWords(ArrayList<String> translations);
	}

	public static synchronized DictionaryDatabase getInstance(Context ctx) {
		// Use the application context, which will ensure that you
		// don't accidentally leak an Activity's context.
		if (mInstance == null) {
			mInstance = new DictionaryDatabase(ctx.getApplicationContext());
		}
		return mInstance;
	}

	private DictionaryDatabase(Context context) {
		super(context, DATABASE_NAME, null, DATABASE_VERSION);
	}

	// Creating Tables
	@Override
	public void onCreate(SQLiteDatabase db) {
		try {
			db.execSQL(PRAGMA_ENCODING);
			db.execSQL(CREATE_DICTIONARY_TABLE);
			db.execSQL(CREATE_DICTIONARY_INDEX);
		} catch (SQLiteException excpetion) {
			LogUtils.LOGE(TAG, "requestAdd", excpetion);
		} finally {
		}
	}

	@Override
	public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
		try {
			// Drop older table if existed
			db.execSQL("drop table if exists " + TABLE_DICTIONARY);
			db.execSQL("drop index if exists " + INDEX_DICTIONARY_1);
			// Create tables again
			onCreate(db);
		} catch (SQLiteException excpetion) {
			LogUtils.LOGE(TAG, "requestAdd", excpetion);
		} finally {
		}
	}

	public void stop() {
		LogUtils.LOGD(TAG, "stop >>");
		if (mTask != null) {
			mTask.cancel(true);
		}
		LogUtils.LOGD(TAG, "stop <<");
	}

	/**
	 * All CRUD(Create, Read, Update, Delete) Operations
	 */
	public void requestAdd(final Translation translation,
			final IListener listener) {
		// Inserting Row
		mTask = new AsyncTask<Void, Void, Void>() {
			@Override
			protected Void doInBackground(Void... params) {
				final ContentValues values = new ContentValues();
				values.put(KEY_LANGUAGE_ID, translation.getLanguage().getId());
				values.put(KEY_TYPE_ID, translation.getType());
				values.put(FIELD_WORD, translation.getWord());
				values.put(FIELD_TRANSLATION, translation.getTranslation());
				values.put(FIELD_ATTACHMENTS, translation.getAttachments());
				SQLiteDatabase db = null;
				try {
					db = getWritableDatabase();
					long rowId = db.insert(TABLE_DICTIONARY, null, values);
					translation.setId(rowId);
					LogUtils.LOGD(TAG, "requestAdd id = [" + rowId + "]");
				} catch (SQLiteException excpetion) {
					LogUtils.LOGE(TAG, "requestAdd", excpetion);
				} finally {
					if (db != null) {
						db.close();
					}
				}
				return null;
			}

			@Override
			protected void onPostExecute(Void l) {
				mTask = null;
				listener.responseAdd(translation);
			}
		}.execute();
	}

	public void requestGet(final Translation translation,
			final IListener listener) {
		mTask = new AsyncTask<Void, Void, Translation>() {
			@Override
			protected Translation doInBackground(Void... params) {
				SQLiteDatabase db = null;
				try {
					db = getReadableDatabase();
					Cursor cursor = db.query(
							TABLE_DICTIONARY,
							new String[] { KEY_ID, FIELD_TRANSLATION,
									FIELD_ATTACHMENTS },
							KEY_LANGUAGE_ID + "=? and " + KEY_TYPE_ID
									+ "=? and " + FIELD_WORD + "=? ",
							new String[] {
									String.valueOf(translation.getLanguage()
											.getId()),
									String.valueOf(translation.getType()),
									String.valueOf(translation.getWord()) },
							null, null, null, null);
					if (cursor != null) {
						cursor.moveToFirst();
						if (cursor.getCount() > 0) {
							LogUtils.LOGD(TAG,
									"requestGet id = [" + cursor.getLong(0)
											+ "]");
							translation.setId(cursor.getLong(0));
							translation.setTranslation(cursor.getString(1));
							translation.setAttachments(cursor.getString(2));
						}
						cursor.close();
					}
				} catch (SQLiteException exception) {
					LogUtils.LOGE(TAG, "requestGet", exception);
				} finally {
					if (db != null) {
						db.close();
					}
				}
				return translation;
			}

			@Override
			protected void onPostExecute(Translation translation) {
				mTask = null;
				listener.responseGet(translation);
			}
		}.execute();
	}

	public void requestDelete(final Translation translation,
			final IListener listener) {
		mTask = new AsyncTask<Void, Void, Void>() {
			@Override
			protected Void doInBackground(Void... params) {
				SQLiteDatabase db = null;
				try {
					db = getWritableDatabase();
					int passed = db
							.delete(TABLE_DICTIONARY, KEY_ID + "=?",
									new String[] { String.valueOf(translation
											.getId()) });
					LogUtils.LOGD(TAG, "requestDelete rows " + passed
							+ " id = [" + translation.getId() + "]");
				} catch (SQLiteException exception) {
					LogUtils.LOGE(TAG, "requestDelete", exception);
				} finally {
					if (db != null) {
						db.close();
					}
				}
				return null;
			}

			@Override
			protected void onPostExecute(Void params) {
				mTask = null;
				listener.responseDelete();
			}

		}.execute();
	}

	public void requestGetAll(final IListener listener) {
		mTask = new AsyncTask<Void, Void, ArrayList<Translation>>() {
			@Override
			protected ArrayList<Translation> doInBackground(Void... params) {
				ArrayList<Translation> contactList = new ArrayList<Translation>();
				final String selectQuery = "select " + KEY_ID + ","
						+ KEY_LANGUAGE_ID + "," + KEY_TYPE_ID + ","
						+ FIELD_WORD + " from " + TABLE_DICTIONARY
						+ " order by " + FIELD_WORD + "," + KEY_LANGUAGE_ID;
				SQLiteDatabase db = null;
				try {
					db = getReadableDatabase();
					Cursor cursor = db.rawQuery(selectQuery, null);
					if (cursor != null) {
						if (cursor.moveToFirst()) {
							do {
								Language language = TranslationManager.getLanguages()[cursor
										.getInt(1)];
								Translation translation = new Translation(
										language, cursor.getInt(2),
										cursor.getString(3));
								translation.setId(cursor.getLong(0));
								contactList.add(translation);
							} while (!isCancelled() && cursor.moveToNext());
						}
						cursor.close();
					}
				} catch (SQLiteException exception) {
					LogUtils.LOGE(TAG, "requestGetAll", exception);
				} finally {
					if (db != null) {
						db.close();
					}
				}
				return contactList;
			}

			@Override
			protected void onPostExecute(ArrayList<Translation> translations) {
				mTask = null;
				listener.responseGetAll(translations);
			}
		}.execute();
	}

	public void requestGetAllWords(final IListener listener) {
		mTask = new AsyncTask<Void, Void, ArrayList<String>>() {
			@Override
			protected ArrayList<String> doInBackground(Void... params) {
				ArrayList<String> wordsList = new ArrayList<String>();
				final String selectQuery = "select distinct " + FIELD_WORD
						+ " from " + TABLE_DICTIONARY;
				SQLiteDatabase db = null;
				try {
					db = getReadableDatabase();
					Cursor cursor = db.rawQuery(selectQuery, null);
					if (cursor != null) {
						if (cursor.moveToFirst()) {
							do {
								wordsList.add(cursor.getString(0));
							} while (!isCancelled() && cursor.moveToNext());
						}
						cursor.close();
					}
				} catch (SQLiteException exception) {
					LogUtils.LOGE(TAG, "requestGetAllWords", exception);
				} finally {
					if (db != null) {
						db.close();
					}
				}
				return wordsList;
			}

			@Override
			protected void onPostExecute(ArrayList<String> words) {
				mTask = null;
				listener.responseGetAllWords(words);
			}
		}.execute();
	}
}
