/*
 * Copyright (C) 2012 Mykola Kondratenko
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.mykola.lexinproject.data;

import java.util.ArrayList;

import android.os.Parcel;
import android.os.Parcelable;
import android.text.TextUtils;

import com.mykola.lexinproject.utils.LogUtils;

public class Translation implements Parcelable {
	private static final String STRING_DELIMETER = "|";
	private static final String STRING_DELIMETER_REGEXP = "\\|";
	private static final String TAG = "Translation";

	public static class Attachment implements Parcelable {
		public String mTitle;
		public String mUrl;

		@Override
		public int describeContents() {
			// TODO Auto-generated method stub
			return 0;
		}

		@Override
		public void writeToParcel(Parcel dest, int flags) {
			dest.writeString(mTitle);
			dest.writeString(mUrl);
		}

		public Attachment(String title, String url) {
			this.mTitle = title;
			this.mUrl = url;
		}

		public Attachment(Parcel in) {
			mTitle = in.readString();
			mUrl = in.readString();
		}

		public static final Parcelable.Creator<Attachment> CREATOR = new Creator<Attachment>() {
			public Attachment createFromParcel(Parcel source) {
				return new Attachment(source);
			}

			public Attachment[] newArray(int size) {
				return new Attachment[size];
			}
		};
	}

	public final static long INVALID_ID = -1;
	private Language mLanguage;
	final private int mType;
	final private String mWord;
	private String mTranslation;
	private ArrayList<Attachment> mAttachments;
	private long mId;
	private boolean mDeleted = false;

	public Translation(Language language, int type, String word) {
		this.mLanguage = language;
		this.mType = type;
		this.mWord = word;
		this.mId = INVALID_ID;
		this.mAttachments = new ArrayList<Attachment>();
	}

	public String getWord() {
		return mWord;
	}

	public Language getLanguage() {
		return mLanguage;
	}

	public int getType() {
		return mType;
	}

	public ArrayList<Attachment> getAttachmentsArray() {
		return mAttachments;
	}

	public void setDeleted(boolean delete) {
		mDeleted = delete;
	}

	public boolean isDeleted() {
		return mDeleted;
	}

	public String getAttachments() {
		StringBuilder content = new StringBuilder();
		if (mAttachments != null) {
			for (int i = 0; i < mAttachments.size(); i++) {
				content.append(mAttachments.get(i).mTitle);
				content.append(STRING_DELIMETER);
				content.append(mAttachments.get(i).mUrl);
				content.append(STRING_DELIMETER);
			}
		}
		return content.toString();
	}

	public void setAttachments(String content) {
		LogUtils.LOGD(TAG, "setAttachments " + content);
		mAttachments = new ArrayList<Attachment>();
		final String[] items = content.split(STRING_DELIMETER_REGEXP);
		for (int i = 0; i < items.length; i += 2) {
			final String title = items[i];
			if (!TextUtils.isEmpty(title) && items.length > 1) {
				mAttachments.add(new Attachment(title, items[i + 1]));
			}
		}
	}

	public void setAttachments(ArrayList<Translation.Attachment> attachments) {
		mAttachments = attachments;
	}

	public String getTranslation() {
		return mTranslation;
	}

	public void setTranslation(String translation) {
		this.mTranslation = translation;
	}

	public long getId() {
		return mId;
	}

	public void setId(long id) {
		this.mId = id;
	}

	@Override
	public int describeContents() {
		return 0;
	}

	public Translation(Parcel in) {
		mType = in.readInt();
		mWord = in.readString();
		mTranslation = in.readString();
		mAttachments = new ArrayList<Translation.Attachment>();
		in.readTypedList(mAttachments, Translation.Attachment.CREATOR);
		mId = in.readLong();
		mLanguage = Language.CREATOR.createFromParcel(in);
	}

	@Override
	public void writeToParcel(Parcel outParcel, int flags) {
		outParcel.writeInt(mType);
		outParcel.writeString(mWord);
		outParcel.writeString(mTranslation);
		outParcel.writeTypedList(mAttachments);
		outParcel.writeLong(mId);
		mLanguage.writeToParcel(outParcel, flags);
	}

	public static final Parcelable.Creator<Translation> CREATOR = new Creator<Translation>() {
		public Translation createFromParcel(Parcel source) {
			return new Translation(source);
		}

		public Translation[] newArray(int size) {
			return new Translation[size];
		}
	};
}
