/*
 * Copyright (C) 2012 Mykola Kondratenko
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.mykola.lexinproject.data;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Set;

import android.app.Activity;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.ViewGroup;
import android.widget.ArrayAdapter;
import android.widget.Button;
import android.widget.SectionIndexer;
import android.widget.TextView;

import com.mykola.lexinproject.R;
import com.mykola.lexinproject.providers.TranslationManager;
import com.mykola.lexinproject.utils.UIUtils;

public class TranslationsAdapter extends ArrayAdapter<Translation> implements
		SectionIndexer {
	public interface IListener {
		public void onDelete(Translation translation);
	}

	private final static String TAG = "TranslationsAdapter";
	private final Activity mActivity;
	private final ArrayList<Translation> mTranslations;
	private final IListener mListener;
	private final HashMap<String, Integer> mAlphaIndexer;
	private final String[] mSections;

	public TranslationsAdapter(Activity activity,
			ArrayList<Translation> translations, IListener listener) {
		super(activity, android.R.layout.simple_list_item_1, translations);
		this.mActivity = activity;
		this.mTranslations = translations;
		this.mListener = listener;
		this.mAlphaIndexer = new HashMap<String, Integer>();
		final int size = translations.size();
		for (int i = 0; i < size; i++) {
			final String ch = translations.get(i).getWord().substring(0, 1)
					.toUpperCase();
			mAlphaIndexer.put(ch, i);
		}
		Set<String> sectionLetters = mAlphaIndexer.keySet();
		ArrayList<String> sectionList = new ArrayList<String>(sectionLetters);
		Collections.sort(sectionList);
		mSections = new String[sectionList.size()];
		sectionList.toArray(mSections);
	}

	public static class ViewHolder {
		public TextView mItemTitle;
		public TextView mItemSubTitle;
		public Button mDeleteButton;
	}

	public int getCount() {
		return mTranslations.size();
	}

	public Translation getItem(int position) {
		return mTranslations.get(position);
	}

	public IListener getListener() {
		return mListener;
	}

	@Override
	public View getView(int position, View convertView, ViewGroup parent) {
		View view = convertView;
		ViewHolder holder;
		if (view == null) {
			holder = new ViewHolder();
			view = mActivity.getLayoutInflater().inflate(
					R.layout.list_item_history, null);
			holder.mItemTitle = (TextView) view
					.findViewById(R.id.history_title);
			holder.mItemSubTitle = (TextView) view
					.findViewById(R.id.history_subtitle);
			holder.mDeleteButton = (Button) view
					.findViewById(R.id.history_button_delete);
			view.setTag(holder);
		} else {
			holder = (ViewHolder) view.getTag();
		}
		final Translation translation = mTranslations.get(position);
		if (translation != null) {
			StringBuilder historyItem = new StringBuilder();
			historyItem.append(TranslationManager.id2Type(translation.getType()));
			historyItem.append(" ");
			historyItem.append(translation.getLanguage().getName());
			holder.mItemTitle.setText(translation.getWord());
			holder.mItemSubTitle.setText(historyItem.toString());
			holder.mDeleteButton.setOnClickListener(new OnClickListener() {
				public void onClick(View v) {
					remove(translation);
					getListener().onDelete(translation);
				}
			});
			if (UIUtils.hasICS()) {
				if (translation.isDeleted()) {
					holder.mDeleteButton.setVisibility(View.VISIBLE);
				} else {
					holder.mDeleteButton.setVisibility(View.INVISIBLE);
				}
			}
		}
		return view;
	}

	@Override
	public long getItemId(int position) {
		return getItem(position).getId();
	}

	public ArrayList<Translation> getTranslations() {
		return mTranslations;
	}

	@Override
	public int getPositionForSection(int arg0) {
		return mAlphaIndexer.get(mSections[arg0]);
	}

	@Override
	public int getSectionForPosition(int arg0) {
		return 1;
	}

	@Override
	public Object[] getSections() {
		return mSections;
	}
}