/*
 * Copyright (C) 2012 Mykola Kondratenko
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.mykola.lexinproject.providers;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.StringReader;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.HttpVersion;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.conn.ClientConnectionManager;
import org.apache.http.conn.params.ConnManagerParams;
import org.apache.http.conn.scheme.PlainSocketFactory;
import org.apache.http.conn.scheme.Scheme;
import org.apache.http.conn.scheme.SchemeRegistry;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.impl.conn.tsccm.ThreadSafeClientConnManager;
import org.apache.http.params.BasicHttpParams;
import org.apache.http.params.HttpConnectionParams;
import org.apache.http.params.HttpParams;
import org.apache.http.params.HttpProtocolParams;
import org.htmlcleaner.TagNode;
import org.htmlcleaner.XPatherException;

import android.os.AsyncTask;

import com.actionbarsherlock.app.SherlockFragment;
import com.mykola.lexinproject.data.D;
import com.mykola.lexinproject.data.Translation;
import com.mykola.lexinproject.utils.LogUtils;
import com.mykola.lexinproject.utils.XmlUtils;

public class LexinTranslator extends TranslatorBase {
	private static final String STRING_INGEN_UNIK = " - Ingen ";
	private static final String XPATH_BODY = "//body";
	private static final String XPATH_BODY_URLS = "//body//a";
	private static final String TAG = "LexinTranslator";
	private final static String HTTP_URL_1 = "http://lexin.nada.kth.se/lexin/service?searchinfo=";
	private final static String HTTP_URL_2 = "http://folkets-lexikon.csc.kth.se/folkets/service?lang=";
	private final HttpClient mHttpClient;
	private AsyncTask<?, ?, ?> mTask;

	public LexinTranslator(SherlockFragment parent) {
		super(parent);
		HttpParams params = new BasicHttpParams();
		HttpConnectionParams.setConnectionTimeout(params, 40 * 1000);
		HttpConnectionParams.setSoTimeout(params, 40 * 1000);
		ConnManagerParams.setMaxTotalConnections(params, 100);
		HttpProtocolParams.setVersion(params, HttpVersion.HTTP_1_1);
		SchemeRegistry schemeRegistry = new SchemeRegistry();
		schemeRegistry.register(new Scheme("http", PlainSocketFactory
				.getSocketFactory(), 80));
		ClientConnectionManager cm = new ThreadSafeClientConnManager(params,
				schemeRegistry);
		mHttpClient = new DefaultHttpClient(cm, params);
	}

	@Override
	public void translate(Translation translation, IListener listener) {
		LogUtils.LOGD(TAG, "translate >>");
		final StringBuilder request = new StringBuilder();
		if (translation.getLanguage().getName().equals(D.STR_ENGELSKT)) {
			request.append(HTTP_URL_2);
			if (translation.getType() == 0) {
				request.append("sv");
			} else {
				request.append("en");
			}
			request.append("&interface=en&word=");
			try {
				request.append(URLEncoder.encode(translation.getWord(), "UTF-8"));
			} catch (UnsupportedEncodingException ex) {
				LogUtils.LOGE(TAG, "translate", ex);
			}
		} else {
			request.append(HTTP_URL_1);
			request.append(TranslationManager.id2Type(translation.getType()));
			request.append(",");
			request.append(translation.getLanguage().getShortcut());
			request.append(",");
			try {
				request.append(URLEncoder.encode(translation.getWord(), "UTF-8"));
			} catch (UnsupportedEncodingException ex) {
				LogUtils.LOGE(TAG, "translate", ex);
			}
			request.append(";");
		}		
		setState(STATE.eTranslate);
		loadUrl(request.toString(), translation, listener);
		LogUtils.LOGD(TAG, "translate <<");
	}

	protected void loadUrl(final String url, final Translation translation,
			final IListener listener) {
		LogUtils.LOGD(TAG, "loadUrl >> " +url);
		mTask = new AsyncTask<Void, Void, Boolean>() {

			@Override
			protected void onPreExecute() {
				super.onPreExecute();
			}

			@Override
			protected Boolean doInBackground(Void... params) {
				LogUtils.LOGD(TAG, "loadUrl >>");
				StringBuilder sb = new StringBuilder();
				try {
					HttpGet httpget = new HttpGet(url);
					HttpResponse response = mHttpClient.execute(httpget);
					HttpEntity entity = response.getEntity();
					if (entity != null) {
						InputStream instream = entity.getContent();
						try {
							BufferedReader reader = new BufferedReader(
									new InputStreamReader(instream));
							String line = null;
							while ((line = reader.readLine()) != null) {
								sb.append(line);
								sb.append("\n");
							}
						} catch (IOException ex) {
							LogUtils.LOGE(TAG, "loadUrl", ex);
						} catch (RuntimeException ex) {
							LogUtils.LOGE(TAG, "loadUrl", ex);
							httpget.abort();
						} finally {
							try {
								instream.close();
								entity.consumeContent();
							} catch (Exception ex) {
								LogUtils.LOGE(TAG, "loadUrl", ex);
							}
						}
					}
				} catch (Exception ex) {
					LogUtils.LOGE(TAG, "loadUrl", ex);
				}
				LogUtils.LOGD(TAG, "loadUrl <<");
				return parseHTML(sb.toString(), translation);
			}

			@Override
			protected void onPostExecute(Boolean found) {
				super.onPostExecute(found);
				setState(STATE.eStop);
				if (listener != null) {
					if (found) {
						listener.responseWebGet(translation);
					} else {
						listener.responseWebGetNotFound(translation
								.getTranslation());
					}
				}
				mTask = null;
			}
		}.execute();
	}

	private boolean parseHTML(final String html, Translation translation) {
		LogUtils.LOGD(TAG, "parseHTML >>");
		if (getState() == STATE.eStop) {
			return false;
		}
		TagNode node = null;
		boolean found = false;
		try {
			LogUtils.LOGD(TAG, "HtmlCleaner >>");
			node = getCleaner().clean(new StringReader(html));
			LogUtils.LOGD(TAG, "HtmlCleaner <<");
		} catch (IOException exception) {
			LogUtils.LOGE(TAG, "parseHTML", exception);
		}
		if (node != null) {
			Object[] translateNodes = null;
			try {
				LogUtils.LOGD(TAG, "xPath translates >> ");
				translateNodes = node.evaluateXPath(XPATH_BODY);
				LogUtils.LOGD(TAG, "xPath translates << ");
			} catch (XPatherException exception) {
				LogUtils.LOGE(TAG, "parseHTML", exception);
			}
			if (translateNodes != null && translateNodes.length > 0) {
				final TagNode translateNode = (TagNode) translateNodes[0];
				translation.setTranslation(XmlUtils.toXml(translateNode));
				found = html.indexOf(translation.getWord()+STRING_INGEN_UNIK) == -1;
				if (found) {
					try {
						LogUtils.LOGD(TAG, "xPath translates >> ");
						Object[] linkNodes = node
								.evaluateXPath(XPATH_BODY_URLS);
						LogUtils.LOGD(TAG, "xPath translates << ");
						translation.setAttachments(XmlUtils.getUrls(linkNodes));
					} catch (XPatherException exception) {
						LogUtils.LOGE(TAG, "parseHTML", exception);
					}
				} else {
					LogUtils.LOGE(TAG, "Server returned unexpected result");
				}
			} else {
				LogUtils.LOGE(TAG, "Server returned unexpected result");
			}
		} else {
			LogUtils.LOGE(TAG, "Server returned unexpected result");
		}
		LogUtils.LOGD(TAG, "parseHTML <<");
		return found;
	}

	public void stop() {
		setState(STATE.eStop);
		if (mTask != null) {
			mTask.cancel(true);
		}
	}
}
