/*
 * Copyright (C) 2012 Mykola Kondratenko
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.mykola.lexinproject.providers;

import java.util.ArrayList;

import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.text.TextUtils;
import android.widget.Toast;

import com.actionbarsherlock.app.SherlockFragment;
import com.mykola.lexinproject.R;
import com.mykola.lexinproject.data.D;
import com.mykola.lexinproject.data.DictionaryDatabase;
import com.mykola.lexinproject.data.Language;
import com.mykola.lexinproject.data.Translation;
import com.mykola.lexinproject.ui.IUIListener;
import com.mykola.lexinproject.utils.LogUtils;

public class TranslationManager implements DictionaryDatabase.IListener,
		TranslatorBase.IListener {
	private final static String TAG = "TranslationManager";

	private final TranslatorBase mLexinClient;
	private final DictionaryDatabase mDictionary;
	private final IUIListener mListener;
	private final SherlockFragment mParent;

	public TranslationManager(SherlockFragment parent) {
		this.mParent = parent;
		this.mDictionary = DictionaryDatabase.getInstance(getParent()
				.getActivity());
		this.mLexinClient = new LexinTranslator(getParent());
		try {
			this.mListener = (IUIListener) getParent();
		} catch (ClassCastException e) {
			throw new ClassCastException(parent.toString()
					+ " must implement IUIListener");
		}
	}

	public void translate(final String word, final int type,
			final Language language) {
		LogUtils.LOGD(TAG, "translate word='" + word + "' type=" + type
				+ " language=" + language.getName());
		if (language != null && (!TextUtils.isEmpty(word))) {
			getListener().startProgress();
			mDictionary.requestGet(new Translation(language, type, word), this);
		}
	}

	@Override
	public void responseWebGet(Translation translation) {
		LogUtils.LOGD(TAG, "responseWebGet >>");
		addTranslation(translation);
	}

	@Override
	public void responseWebGetVariants(ArrayList<String> words) {
		LogUtils.LOGD(TAG, "responseWebEmpty >>");
		getListener().onFoundVariants(words);
		getListener().stopProgress();
	}
	
	@Override
	public void responseWebGetNotFound(String message) {
		LogUtils.LOGD(TAG, "responseWebGetNotFound >>");
		getListener().onNotFound(message);	
		getListener().stopProgress();
	}
	
	public void addTranslation(Translation translation) {
		LogUtils.LOGD(TAG, "addTranslation >>");
		mDictionary.requestAdd(translation, this);
	}

	public void getAllTranslations() {
		LogUtils.LOGD(TAG, "getAllTranslations >>");
		getListener().startProgress();
		mDictionary.requestGetAll(this);
	}

	public void getAllWords() {
		LogUtils.LOGD(TAG, "getAllWords >>");
		getListener().startProgress();
		mDictionary.requestGetAllWords(this);
	}

	public void requestDelete(Translation translation) {
		LogUtils.LOGD(TAG, "requestDelete >>");
		getListener().startProgress();
		mDictionary.requestDelete(translation, this);
	}

	@Override
	public void responseAdd(Translation translation) {
		LogUtils.LOGD(TAG, "responseAdd >>");
		getListener().onFound(translation);
		getListener().stopProgress();
	}

	@Override
	public void responseGet(Translation translation) {
		LogUtils.LOGD(TAG, "responseGet >>");
		if (translation.getId() != Translation.INVALID_ID) {
			getListener().onFound(translation);
			getListener().stopProgress();
		} else {
			ConnectivityManager cm = (ConnectivityManager) getParent()
					.getActivity().getSystemService(
							android.content.Context.CONNECTIVITY_SERVICE);
			NetworkInfo activeNetwork = cm.getActiveNetworkInfo();
			if (activeNetwork == null || !activeNetwork.isConnected()) {
				Toast.makeText(getParent().getActivity(),
						R.string.no_connection, Toast.LENGTH_SHORT).show();
			} else {
				mLexinClient.translate(translation, this);
			}
		}
		LogUtils.LOGD(TAG, "responseGet <<");
	}

	@Override
	public void responseGetAll(ArrayList<Translation> translations) {
		LogUtils.LOGD(TAG, "responseGetAll >>");
		getListener().onFound(translations);
		getListener().stopProgress();
	}

	@Override
	public void responseGetAllWords(ArrayList<String> words) {
		LogUtils.LOGD(TAG, "responseGetAllWords >>");
		getListener().onFoundWords(words);
		getListener().stopProgress();
	}

	@Override
	public void responseDelete() {
		LogUtils.LOGD(TAG, "responseDelete >>");
		getListener().stopProgress();
	}

	public void stop() {
		LogUtils.LOGD(TAG, "stop >>");
		mLexinClient.stop();
		mDictionary.stop();
		getListener().stopProgress();
		LogUtils.LOGD(TAG, "stop <<");
	}

	public static String[] getTypes(final Language language) {
		if (language.getName().equals(D.STR_ENGELSKT)) {
			return D.TYPES_ENGELSKT;
		} else {
			return D.TYPES_ALL;
		}
	}

	public static String id2Type(int type) {
		if (type >= 0 && type < D.TYPES_ALL.length)
			return D.TYPES_ALL[type];
		return "Null";
	}

	public static Language[] getLanguages() {
		return D.LANGUAGES;
	}

	private IUIListener getListener() {
		return mListener;
	}

	private SherlockFragment getParent() {
		return mParent;
	}

}
