/*
 * Copyright (C) 2012 Mykola Kondratenko
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.mykola.lexinproject.ui;

import java.util.ArrayList;

import android.app.Activity;
import android.content.Intent;
import android.content.res.Configuration;
import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.AdapterView;
import android.widget.AdapterView.OnItemClickListener;
import android.widget.ListView;

import com.actionbarsherlock.app.SherlockFragment;
import com.example.android.swipedismiss.SwipeDismissListViewTouchListener;
import com.mykola.lexinproject.R;
import com.mykola.lexinproject.data.Translation;
import com.mykola.lexinproject.data.TranslationsAdapter;
import com.mykola.lexinproject.providers.TranslationManager;
import com.mykola.lexinproject.utils.LogUtils;
import com.mykola.lexinproject.utils.UIUtils;

public class HistoryFragment extends SherlockFragment implements IUIListener,
		TranslationsAdapter.IListener {
	private final static String TAG = "HistoryFragment";
	private static final String STATE_TRANSLATIONS = "translations";
	private ListView mListView;
	private TranslationManager mTranslationFactory;
	private TranslationsAdapter mAdapter;
	OnItemClickListener onItemClickListener = new OnItemClickListener() {
		public void onItemClick(AdapterView<?> parent, View view, int position,
				long id) {
			if (parent.getCount() > position) {
				Intent intent = new Intent(getActivity(), LexinActivity.class);
				Translation translation = (Translation) mListView
						.getItemAtPosition(position);
				intent.putExtra(PreviewFragment.ID_REQUEST, translation.getWord());
				intent.putExtra(PreviewFragment.ID_TYPE, translation.getType());
				intent.putExtra(PreviewFragment.ID_LANGUAGE, translation
						.getLanguage().getId());
				intent.putExtra(PreviewFragment.ID_SEARCH, true);
				startActivity(intent);
			}
		}
	};

	public View onCreateView(LayoutInflater inflater, ViewGroup container,
			Bundle savedInstanceState) {
		LogUtils.LOGD(TAG, "onCreateView >>");
		View view = inflater.inflate(R.layout.history_fragment, container,
				false);
		getSherlockActivity().getSupportActionBar().setHomeButtonEnabled(true);
		getSherlockActivity().getSupportActionBar().setDisplayHomeAsUpEnabled(
				true);
		mListView = (ListView) view.findViewById(R.id.listView2);
		mTranslationFactory = new TranslationManager(this);
		if (savedInstanceState != null) {
			LogUtils.LOGD(TAG, "savedInstanceState available");
			if (savedInstanceState.containsKey(STATE_TRANSLATIONS)) {
				ArrayList<Translation> translations = savedInstanceState
						.getParcelableArrayList(STATE_TRANSLATIONS);
				mAdapter = new TranslationsAdapter(getActivity(), translations,
						this);
				mListView.setAdapter(mAdapter);
			}
		} else {
			mTranslationFactory.getAllTranslations();
		}
		mListView.setOnItemClickListener(onItemClickListener);
		// Create a ListView-specific touch listener. ListViews are given
		// special treatment because
		// by default they handle touches for their list items... i.e. they're
		// in charge of drawing
		// the pressed state (the list selector), handling list item clicks,
		// etc.
		if (UIUtils.hasICS()) {
			SwipeDismissListViewTouchListener touchListener = new SwipeDismissListViewTouchListener(
					mListView,
					new SwipeDismissListViewTouchListener.OnDismissCallback() {
						@Override
						public void onDismiss(ListView listView,
								int[] reverseSortedPositions) {
							if (mAdapter != null) {
								for (int position : reverseSortedPositions) {
									mAdapter.getItem(position).setDeleted(true);
								}
								mAdapter.notifyDataSetChanged();
								mListView
										.setOnItemClickListener(onItemClickListener);
							}
						}
					});
			mListView.setOnTouchListener(touchListener);
			// Setting this scroll listener is required to ensure that during
			// ListView scrolling,
			// we don't look for swipes.
			mListView.setOnScrollListener(touchListener.makeScrollListener());
		}
		getSherlockActivity()
			.setSupportProgressBarIndeterminateVisibility(
				Boolean.FALSE);
		LogUtils.LOGD(TAG, "onCreateView <<");
		return view;
	}

	@Override
	public void onSaveInstanceState(Bundle outState) {
		super.onSaveInstanceState(outState);
		LogUtils.LOGD(TAG, "onSaveInstanceState >>");
		outState.putParcelableArrayList(STATE_TRANSLATIONS,
				mAdapter.getTranslations());
		LogUtils.LOGD(TAG, "onSaveInstanceState <<");
	}

	@Override
	public void onPause() {
		super.onPause();
		LogUtils.LOGD(TAG, "onPause >>");
		stopProgress();
		mTranslationFactory.stop();
		LogUtils.LOGD(TAG, "onPause <<");
	}

	@Override
	public void onResume() {
		super.onResume();
		LogUtils.LOGD(TAG, "onResume >>");
		LogUtils.LOGD(TAG, "onResume <<");
	}

	@Override
	public void onConfigurationChanged(Configuration newConfig) {
		super.onConfigurationChanged(newConfig);
		LogUtils.LOGD(TAG, "onConfigurationChanged >>");
		mTranslationFactory.stop();
		LogUtils.LOGD(TAG, "onConfigurationChanged <<");
	}

	@Override
	public void onAttach(Activity activity) {
		LogUtils.LOGD(TAG, "onAttach");
		super.onAttach(activity);
	}

	@Override
	public void startProgress() {
		if (getActivity() != null) {
			getActivity().runOnUiThread(new Runnable() {
				public void run() {
					LogUtils.LOGD(TAG, "startProgress >>");
					getSherlockActivity()
							.setSupportProgressBarIndeterminateVisibility(
									Boolean.TRUE);
					LogUtils.LOGD(TAG, "startProgress <<");
				}
			});
		}
	}

	@Override
	public void stopProgress() {
		if (getActivity() != null) {
			getActivity().runOnUiThread(new Runnable() {
				public void run() {
					LogUtils.LOGD(TAG, "stopProgress >>");
					getSherlockActivity()
							.setSupportProgressBarIndeterminateVisibility(
									Boolean.FALSE);
					LogUtils.LOGD(TAG, "stopProgress <<");
				}
			});
		}
	}

	@Override
	public void showOnView(String content) {
		LogUtils.LOGE(TAG, "showOnWebView");
	}

	@Override
	public void onFoundVariants(ArrayList<String> words) {
		LogUtils.LOGE(TAG, "onFoundVariants");
	}

	@Override
	public void onFound(Translation translation) {
		LogUtils.LOGE(TAG, "onFound");
	}

	@Override
	public void onFound(ArrayList<Translation> translations) {
		LogUtils.LOGE(TAG, "onFound size=" + translations.size());
		mAdapter = new TranslationsAdapter(getActivity(), translations, this);
		mListView.setAdapter(mAdapter);
	}

	@Override
	public void onFoundWords(ArrayList<String> words) {
		LogUtils.LOGE(TAG, "onFoundWords");
	}

	@Override
	public void onDelete(Translation translation) {
		LogUtils.LOGD(TAG, "onDelete >>");
		mTranslationFactory.requestDelete(translation);
		LogUtils.LOGD(TAG, "onDelete <<");
	}

	@Override
	public void onNotFound(String message) {
		LogUtils.LOGD(TAG, "onNotFound <<");
	}
}
