/*
 * Copyright (C) 2012 Mykola Kondratenko
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.mykola.lexinproject.ui;

import java.util.ArrayList;

import android.content.Intent;
import android.os.Bundle;
import android.support.v4.app.Fragment;
import android.support.v4.app.ShareCompat;
import android.text.Html;
import android.view.WindowManager;
import android.widget.ArrayAdapter;

import com.actionbarsherlock.app.ActionBar;
import com.actionbarsherlock.app.SherlockFragmentActivity;
import com.actionbarsherlock.view.Menu;
import com.actionbarsherlock.view.MenuInflater;
import com.actionbarsherlock.view.MenuItem;
import com.actionbarsherlock.view.Window;
import com.mykola.lexinproject.R;
import com.mykola.lexinproject.data.Translation;
import com.mykola.lexinproject.providers.TranslationManager;
import com.mykola.lexinproject.utils.LogUtils;

public class LexinActivity extends SherlockFragmentActivity implements
		ActionBar.OnNavigationListener, IUIListener {
	private static final String STRING_TEXT_PLAIN = "text/plain";
	private static final String TAG = "LexinActivity";
	private static final String STATE_SELECTED_NAVIGATION_ITEM = "selected_navigation_item";
	private Translation mSharedData;

	public static IUIListener sDummyListener = new IUIListener() {
		@Override
		public void startProgress() {
		}

		@Override
		public void stopProgress() {
		}

		@Override
		public void showOnView(String content) {
		}

		@Override
		public void onFoundVariants(ArrayList<String> words) {
		}

		@Override
		public void onFound(Translation translation) {
		}

		@Override
		public void onFound(ArrayList<Translation> translations) {
		}

		@Override
		public void onFoundWords(ArrayList<String> words) {
		}

		@Override
		public void onNotFound(String message) {
		}
	};

	@Override
	public void onCreate(Bundle savedInstanceState) {
		LogUtils.LOGD(TAG, "onCreate >> " + savedInstanceState);
		requestWindowFeature(Window.FEATURE_INDETERMINATE_PROGRESS);
		getWindow().setSoftInputMode(
				WindowManager.LayoutParams.SOFT_INPUT_STATE_HIDDEN);
		super.onCreate(savedInstanceState);
		setContentView(R.layout.activity_main);

		// Set up the action bar.
		final ActionBar actionBar = getSupportActionBar();
		actionBar.setDisplayShowTitleEnabled(false);
		actionBar.setDisplayHomeAsUpEnabled(false);
		actionBar.setNavigationMode(ActionBar.NAVIGATION_MODE_LIST);
		// Set up the dropdown list navigation in the action bar.
		actionBar.setListNavigationCallbacks(
				new ArrayAdapter<String>(actionBar.getThemedContext(),
						android.R.layout.simple_list_item_1,
						android.R.id.text1, new String[] {
								getString(R.string.translate),
								getString(R.string.history),
								getString(R.string.attachments) }), this);
		if (savedInstanceState == null) {
			LogUtils.LOGD(TAG, "savedInstanceState not available");
			Bundle args = new Bundle();
			args.putInt(PreviewFragment.ARG_SECTION_NUMBER, 0 + 1);
			PreviewFragment newFragment = new PreviewFragment();
			newFragment.setArguments(args);
			getSupportFragmentManager().beginTransaction()
					.add(R.id.container, newFragment).commit();
		} else {
			if (savedInstanceState.containsKey(STATE_SELECTED_NAVIGATION_ITEM)) {
				LogUtils.LOGD(TAG, "savedInstanceState available");
				getSupportActionBar().setSelectedNavigationItem(
						savedInstanceState
								.getInt(STATE_SELECTED_NAVIGATION_ITEM));
			}
			if (savedInstanceState.containsKey(PreviewFragment.SHARED_DATA)) {
				mSharedData = savedInstanceState
						.getParcelable(PreviewFragment.SHARED_DATA);
			}
		}
		LogUtils.LOGD(TAG, "onCreate <<");
	}

	@Override
	protected void onPostCreate(Bundle savedInstanceState) {
		super.onPostCreate(savedInstanceState);
		setSupportProgressBarIndeterminateVisibility(false);
	}

	@Override
	public void onRestoreInstanceState(Bundle savedInstanceState) {
		LogUtils.LOGD(TAG, "onRestoreInstanceState >>");
		if (savedInstanceState.containsKey(STATE_SELECTED_NAVIGATION_ITEM)) {
			getSupportActionBar().setSelectedNavigationItem(
					savedInstanceState.getInt(STATE_SELECTED_NAVIGATION_ITEM));
			mSharedData = savedInstanceState
					.getParcelable(PreviewFragment.SHARED_DATA);
		}
		super.onRestoreInstanceState(savedInstanceState);
		LogUtils.LOGD(TAG, "onRestoreInstanceState <<");
	}

	@Override
	public void onSaveInstanceState(Bundle outState) {
		LogUtils.LOGD(TAG, "onSaveInstanceState >>");
		outState.putInt(STATE_SELECTED_NAVIGATION_ITEM, getSupportActionBar()
				.getSelectedNavigationIndex());
		outState.putParcelable(PreviewFragment.SHARED_DATA, getSharedData());
		super.onSaveInstanceState(outState);
		LogUtils.LOGD(TAG, "onSaveInstanceState <<");
	}

	@Override
	public boolean onCreateOptionsMenu(Menu menu) {
		LogUtils.LOGD(TAG, "onCreateOptionsMenu");
		MenuInflater inflater = getSupportMenuInflater();
		inflater.inflate(R.menu.mainmenu, menu);
		return true;
	}

	@Override
	public boolean onNavigationItemSelected(int position, long id) {
		LogUtils.LOGD(TAG, "onNavigationItemSelected " + position);
		switch (position) {
		case 0: {
			Fragment newFragment = getSupportFragmentManager()
					.findFragmentById(R.id.container);
			if (newFragment instanceof PreviewFragment) {
				LogUtils.LOGD(TAG, "PreviewFragment exists");
			} else {
				newFragment = new PreviewFragment();
				Bundle args = new Bundle();
				args.putInt(PreviewFragment.ARG_SECTION_NUMBER, position + 1);
				args.putParcelable(PreviewFragment.SHARED_DATA, getSharedData());
				newFragment.setArguments(args);
				getSupportFragmentManager().beginTransaction()
						.replace(R.id.container, newFragment).commit();
			}
		}
			break;
		case 1: {
			Fragment newFragment = getSupportFragmentManager()
					.findFragmentById(R.id.container);
			if (newFragment instanceof HistoryFragment) {
				LogUtils.LOGD(TAG, "HistroyFragment exists");
			} else {
				Bundle args = new Bundle();
				args.putInt(PreviewFragment.ARG_SECTION_NUMBER, position + 1);
				HistoryFragment attachFragment = new HistoryFragment();
				attachFragment.setArguments(args);
				getSupportFragmentManager().beginTransaction()
						.replace(R.id.container, attachFragment).commit();
			}
		}
			break;
		case 2: {
			Fragment newFragment = getSupportFragmentManager()
					.findFragmentById(R.id.container);
			if (newFragment instanceof AttachmentsFragment) {
				LogUtils.LOGD(TAG, "AttachmentsFragment exists");
			} else {
				AttachmentsFragment attachFragment = new AttachmentsFragment();
				Bundle args = new Bundle();
				args.putInt(PreviewFragment.ARG_SECTION_NUMBER, position + 1);
				args.putParcelable(PreviewFragment.SHARED_DATA, getSharedData());
				attachFragment.setArguments(args);
				getSupportFragmentManager().beginTransaction()
						.replace(R.id.container, attachFragment).commit();
			}
		}
			break;
		default:
			LogUtils.LOGE(TAG, "Unexpected position " + position);
			break;
		}

		return true;
	}

	@Override
	public boolean onOptionsItemSelected(MenuItem item) {
		LogUtils.LOGD(TAG, "onOptionsItemSelected #" + item.getItemId());
		switch (item.getItemId()) {
		case android.R.id.home:
			startActivity(new Intent(LexinActivity.this, LexinActivity.class)
					.addFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP));
			return true;
		case R.id.share: {
			doShare();
			return true;
		}
		default:
			return super.onOptionsItemSelected(item);
		}
	}

	private void doShare() {
		if (getSharedData() != null) {
			final StringBuilder subject = new StringBuilder();
			subject.append(getSharedData().getWord());
			subject.append(" ");
			subject.append(TranslationManager.id2Type(getSharedData().getType()));
			subject.append(" ");
			subject.append(getSharedData().getLanguage().getName());
			ShareCompat.IntentBuilder builder = ShareCompat.IntentBuilder
					.from(this)
					.setType(STRING_TEXT_PLAIN)
					.setSubject(String.valueOf(subject))
					.setText(
							String.valueOf(Html.fromHtml(getSharedData()
									.getTranslation())));
			startActivity(Intent.createChooser(builder.getIntent(),
					getString(R.string.title_share)));
		}
	}

	private Translation getSharedData() {
		return mSharedData;
	}

	@Override
	public void startProgress() {
		LogUtils.LOGE(TAG, "startProgress");
	}

	@Override
	public void stopProgress() {
		LogUtils.LOGE(TAG, "stopProgress");
	}

	@Override
	public void showOnView(String content) {
		LogUtils.LOGE(TAG, "showOnWebView");
	}

	@Override
	public void onFoundVariants(ArrayList<String> words) {
		LogUtils.LOGD(TAG, "onFoundVariants");
		mSharedData = null;
	}

	@Override
	public void onFound(Translation translation) {
		LogUtils.LOGD(TAG, "onFound");
		mSharedData = translation;
	}

	@Override
	public void onFound(ArrayList<Translation> translations) {
		LogUtils.LOGE(TAG, "onFound");
	}

	@Override
	public void onFoundWords(ArrayList<String> words) {
		LogUtils.LOGE(TAG, "onFoundWords");
	}

	@Override
	public void onNotFound(String message) {
		LogUtils.LOGE(TAG, "onNotFound");
		mSharedData = null;
	}
}
