package com.noprestige.kanaquiz.questions;

import com.noprestige.kanaquiz.R;
import com.noprestige.kanaquiz.logs.LogDao;
import com.noprestige.kanaquiz.options.OptionsControl;

import java.util.Arrays;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;

public class QuestionBank extends WeightedList<Question>
{
    private Question currentQuestion;
    private Set<String> fullKanaAnswerList = new TreeSet<>(new GojuonOrder());
    private Map<String, WeightedList<String>> weightedAnswerListCache;

    private Set<String> basicAnswerList = new TreeSet<>(new GojuonOrder());
    private Set<String> diacriticAnswerList = new TreeSet<>(new GojuonOrder());
    private Set<String> digraphAnswerList = new TreeSet<>(new GojuonOrder());
    private Set<String> diacriticDigraphAnswerList = new TreeSet<>(new GojuonOrder());

    private Set<String> wordAnswerList = new TreeSet<>();

    private static final int MAX_MULTIPLE_CHOICE_ANSWERS = 6;

    private QuestionRecord previousQuestions;
    private int maxKanaAnswerWeight = -1;
    private int maxWordAnswerWeight = -1;

    public boolean isCurrentQuestionVocab()
    {
        return (currentQuestion.getClass().equals(WordQuestion.class));
    }

    public void newQuestion() throws NoQuestionsException
    {
        if (count() > 0)
        {
            if (previousQuestions == null)
                previousQuestions =
                        new QuestionRecord(Math.min(count(), OptionsControl.getInt(R.string.prefid_repetition)));
            do
                currentQuestion = getRandom();
            while (!previousQuestions.add(currentQuestion));
        }
        else
            throw new NoQuestionsException();
    }

    public String getCurrentQuestionText()
    {
        return currentQuestion.getQuestionText();
    }

    public String getCurrentQuestionKey()
    {
        return currentQuestion.getDatabaseKey();
    }

    public boolean checkCurrentAnswer(String response)
    {
        return currentQuestion.checkAnswer(response);
    }

    public String fetchCorrectAnswer()
    {
        return currentQuestion.fetchCorrectAnswer();
    }

    public boolean addQuestions(Question[] questions)
    {
        weightedAnswerListCache = null;
        previousQuestions = null;
        maxKanaAnswerWeight = -1;
        maxWordAnswerWeight = -1;
        if (questions != null)
        {
            boolean returnValue = true;
            for (Question question : questions)
            {
                // Fetches the percentage of times the user got a kana right,
                Float percentage = LogDao.getKanaPercentage(question.getDatabaseKey());
                if (percentage == null)
                    percentage = 0.1f;
                // The 1f is to invert the value so we get the number of times they got it wrong,
                // Times 100f to get the percentage.
                int weight = (int) Math.ceil((1f - percentage) * 100f);
                // Setting weight to never get lower than 2,
                // so any kana the user got perfect will still appear in the quiz.
                if (weight < 2)
                    weight = 2;
                // if any one of the additions fail, the method returns false
                returnValue = add(weight, question) && returnValue;
                if (question.getClass().equals(WordQuestion.class))
                    returnValue = wordAnswerList.add(question.fetchCorrectAnswer()) && returnValue;
                else if (question.getClass().equals(KanaQuestion.class))
                {
                    returnValue = fullKanaAnswerList.add(question.fetchCorrectAnswer()) && returnValue;
                    // Storing answers in specialized answer lists for more specialized answer selection
                    returnValue =
                            getSpecialList((KanaQuestion) question).add(question.fetchCorrectAnswer()) && returnValue;
                }
            }
            return returnValue;
        }
        return false;
    }

    private Set<String> getSpecialList(KanaQuestion question)
    {
        if (question.isDiacritic() && question.isDigraph())
            return diacriticDigraphAnswerList;
        else if (question.isDiacritic())
            return diacriticAnswerList;
        else if (question.isDigraph())
            return digraphAnswerList;
        else
            return basicAnswerList;
    }

    public boolean addQuestions(QuestionBank questions)
    {
        weightedAnswerListCache = null;
        previousQuestions = null;
        maxKanaAnswerWeight = -1;
        maxWordAnswerWeight = -1;
        fullKanaAnswerList.addAll(questions.fullKanaAnswerList);
        basicAnswerList.addAll(questions.basicAnswerList);
        diacriticAnswerList.addAll(questions.diacriticAnswerList);
        digraphAnswerList.addAll(questions.digraphAnswerList);
        diacriticDigraphAnswerList.addAll(questions.diacriticDigraphAnswerList);
        wordAnswerList.addAll(questions.wordAnswerList);
        return merge(questions);
    }

    private int getMaxKanaAnswerWeight()
    {
        if (maxKanaAnswerWeight < 0)
            maxKanaAnswerWeight = getMaxAnswerWeight(fullKanaAnswerList);
        return maxKanaAnswerWeight;
    }

    private int getMaxWordAnswerWeight()
    {
        if (maxWordAnswerWeight < 0)
            maxWordAnswerWeight = getMaxAnswerWeight(wordAnswerList);
        return maxWordAnswerWeight;
    }

    private int getMaxAnswerWeight(Set<String> list)
    {
        // Max value is to prevent integer overflow in the weighted answer list. Number is chosen so if every
        // answer had this count (which is actually impossible because of the range restriction, but better to
        // err on the side of caution) the resultant calculations of 2^count would not add up to an integer
        // overflow.
        return (int) Math.floor(Math.log(Integer.MAX_VALUE / (list.size() - 1)) / Math.log(2));
    }

    public String[] getPossibleAnswers()
    {
        return (isCurrentQuestionVocab()) ? getPossibleWordAnswers(MAX_MULTIPLE_CHOICE_ANSWERS) :
                getPossibleKanaAnswers(MAX_MULTIPLE_CHOICE_ANSWERS);
    }


    public String[] getPossibleWordAnswers(int maxChoices)
    {
        if (wordAnswerList.size() <= maxChoices)
            return wordAnswerList.toArray(new String[0]);
        else
        {
            if (weightedAnswerListCache == null)
                weightedAnswerListCache = new TreeMap<>();
            if (!weightedAnswerListCache.containsKey(currentQuestion.getDatabaseKey()))
            {
                Map<String, Integer> answerCounts = new TreeMap<>();
                for (String answer : wordAnswerList)
                {
                    if (!answer.equals(fetchCorrectAnswer()))
                    {
                        //fetch all data
                        int count = LogDao.getIncorrectAnswerCount(currentQuestion.getDatabaseKey(), answer);
                        answerCounts.put(answer, count);
                    }
                }

                WeightedList<String> weightedAnswerList =
                        generateWeightedAnswerList(answerCounts, getMaxWordAnswerWeight());

                weightedAnswerListCache.put(currentQuestion.getDatabaseKey(), weightedAnswerList);
            }

            String[] possibleAnswerStrings =
                    weightedAnswerListCache.get(currentQuestion.getDatabaseKey()).getRandom(new String[maxChoices - 1]);

            possibleAnswerStrings = Arrays.copyOf(possibleAnswerStrings, maxChoices);
            possibleAnswerStrings[maxChoices - 1] = fetchCorrectAnswer();

            Arrays.sort(possibleAnswerStrings);

            return possibleAnswerStrings;
        }
    }

    public String[] getPossibleKanaAnswers(int maxChoices)
    {
        if (fullKanaAnswerList.size() <= maxChoices)
            return fullKanaAnswerList.toArray(new String[0]);
        else
        {
            if (weightedAnswerListCache == null)
                weightedAnswerListCache = new TreeMap<>();
            if (!weightedAnswerListCache.containsKey(currentQuestion.getDatabaseKey()))
            {
                Map<String, Integer> answerCounts = new TreeMap<>();
                for (String answer : fullKanaAnswerList)
                {
                    if (!answer.equals(fetchCorrectAnswer()))
                    {
                        //fetch all data
                        int count = LogDao.getIncorrectAnswerCount(currentQuestion.getDatabaseKey(), answer);
                        if (getSpecialList((KanaQuestion) currentQuestion).contains(currentQuestion.getDatabaseKey()))
                            count += 2;
                        answerCounts.put(answer, count);
                    }
                }

                WeightedList<String> weightedAnswerList =
                        generateWeightedAnswerList(answerCounts, getMaxKanaAnswerWeight());

                weightedAnswerListCache.put(currentQuestion.getDatabaseKey(), weightedAnswerList);
            }

            String[] possibleAnswerStrings =
                    weightedAnswerListCache.get(currentQuestion.getDatabaseKey()).getRandom(new String[maxChoices - 1]);

            possibleAnswerStrings = Arrays.copyOf(possibleAnswerStrings, maxChoices);
            possibleAnswerStrings[maxChoices - 1] = fetchCorrectAnswer();

            GojuonOrder.sort(possibleAnswerStrings);

            return possibleAnswerStrings;
        }
    }

    private static WeightedList<String> generateWeightedAnswerList(Map<String, Integer> answerCounts,
            int maxAnswerWeight)
    {
        int maxCount = 0;
        int minCount = Integer.MAX_VALUE;

        Map<String, Float> newAnswerCount = new TreeMap<>();

        for (String answer : answerCounts.keySet())
        {
            int count = answerCounts.get(answer);
            maxCount = Math.max(maxCount, count);
            minCount = Math.min(minCount, count);
            newAnswerCount.put(answer, (float) count);
        }

        // Because of the inherent properties of the power function used later, subtracting the same amount
        // from each count so the lowest count is zero will not change the relative weight of each item. And
        // doing so will also save the space I need to prevent overflow issues.
        if (minCount > 0)
        {
            maxCount -= minCount;
            for (String answer : newAnswerCount.keySet())
            {
                float newCount = newAnswerCount.remove(answer) - minCount;
                newAnswerCount.put(answer, newCount);
            }
        }

        if (maxCount > maxAnswerWeight)
        {
            float controlFactor = maxAnswerWeight / maxCount;
            for (String answer : newAnswerCount.keySet())
            {
                float newCount = newAnswerCount.remove(answer) * controlFactor;
                newAnswerCount.put(answer, newCount);
            }
        }

        WeightedList<String> weightedAnswerList = new WeightedList<>();
        for (String answer : newAnswerCount.keySet())
            weightedAnswerList.add(Math.pow(2, newAnswerCount.get(answer)), answer);

        return weightedAnswerList;
    }
}
