/*
 * Copyright (C) 2008 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.psrivastava.deviceframegenerator;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.SharedPreferences.Editor;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.net.Uri;
import android.os.AsyncTask;
import android.os.Bundle;
import android.preference.PreferenceManager;
import android.support.v4.app.FragmentManager;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.ViewGroup;
import android.widget.AdapterView;
import android.widget.AdapterView.OnItemClickListener;
import android.widget.BaseAdapter;
import android.widget.CheckBox;
import android.widget.CompoundButton;
import android.widget.CompoundButton.OnCheckedChangeListener;
import android.widget.Gallery;
import android.widget.ImageView;
import android.widget.TextView;

import com.actionbarsherlock.app.SherlockFragmentActivity;
import com.actionbarsherlock.view.Menu;
import com.actionbarsherlock.view.MenuItem;
import com.actionbarsherlock.view.Window;
import com.psrivastava.deviceframegenerator.devices.Device;

import de.neofonie.mobile.app.android.widget.crouton.Crouton;
import de.neofonie.mobile.app.android.widget.crouton.Style;

public class MainActivity extends SherlockFragmentActivity implements
		AdapterView.OnItemSelectedListener, OnItemClickListener,
		OnClickListener, OnCheckedChangeListener {

	private static final String LOGTAG = Util.makeLogTag(MainActivity.class);

	private static final ArrayList<Integer> mThumbs = Util.getAllThubnails();

	private static final String URL_PROJECT_PAGE = "http://f2prateek.github.com/Device-Frame-Generator/";
	private static final String URL_ISSUES = "https://github.com/f2prateek/Device-Frame-Generator/issues";
	private static final String URL_DONATION = "https://www.paypal.com/cgi-bin/webscr?cmd=_donations&business=2BNJ6472KN66L&lc=CA&currency_code=CAD&bn=PP%2dDonationsBF%3abtn_donateCC_LG%2egif%3aNonHosted";
	private static final String URL_ASSET_STUDIO = "http://android-ui-utils.googlecode.com/hg/asset-studio/dist/index.html";
	private static final String URL_ATTRIBUTION_INFO = "http://android-ui-utils.googlecode.com/hg/asset-studio/dist/attribution.html";

	private Gallery mGallery;

	private BitmapFactory.Options mOptions;

	private static final int RESULT_SELECT_PICTURE = 21;

	private SharedPreferences mPrefs;
	private ArrayList<Device> mDeviceList;

	TextView mDeviceTitleText;
	TextView mDeviceSizeText;
	TextView mDeviceDimensionsText;

	/**
	 * Remembers last click to detect double click
	 */
	private int lastClick = 0;

	@Override
	public void onCreate(Bundle icicle) {
		super.onCreate(icicle);

		// This has to be called before setContentView and you must use the
		// class in com.actionbarsherlock.view and NOT android.view
		requestWindowFeature(Window.FEATURE_PROGRESS);
		requestWindowFeature(Window.FEATURE_INDETERMINATE_PROGRESS);

		setContentView(R.layout.activity_main);

		if (mPrefs == null) {
			mPrefs = PreferenceManager.getDefaultSharedPreferences(this);
		}

		// TODO:For pre-jellybean devices, spinner is shown onCreate, temporary
		// fix while I investigate more
		setSupportProgressBarIndeterminateVisibility(false);

		mOptions = new BitmapFactory.Options();
		mOptions.inDither = false;
		mOptions.inPreferredConfig = Bitmap.Config.ARGB_8888;
		mDeviceList = Util.getAllDevices();

		mGallery = (Gallery) findViewById(R.id.gallery_thumbnails);
		mGallery.setAdapter(new ThumbnailAdapter(this));
		mGallery.setOnItemSelectedListener(this);
		mGallery.setCallbackDuringFling(false);
		mGallery.setOnItemClickListener(this);
		mGallery.setSelection(mPrefs
				.getInt(AppConstants.KEY_DEVICE_POSITION, 0));

		CheckBox glare = (CheckBox) findViewById(R.id.glare);
		glare.setChecked(mPrefs.getBoolean(AppConstants.KEY_WITH_GLARE, true));
		glare.setOnCheckedChangeListener(this);

		CheckBox shadow = (CheckBox) findViewById(R.id.shadow);
		shadow.setChecked(mPrefs.getBoolean(AppConstants.KEY_WITH_SHADOW, true));
		shadow.setOnCheckedChangeListener(this);

		mDeviceTitleText = (TextView) findViewById(R.id.device_title);
		mDeviceTitleText.setOnClickListener(this);
		mDeviceSizeText = (TextView) findViewById(R.id.device_size);
		mDeviceDimensionsText = (TextView) findViewById(R.id.device_resolution);

		if (!Util.checkStorage()) {
			Crouton.makeText(this, R.string.error_storage_unavailable,
					Style.ALERT).show();
		}

		if (mPrefs.getBoolean(AppConstants.KEY_RUNNING_FIRST_TIME, true)) {
			mPrefs.edit()
					.putBoolean(AppConstants.KEY_RUNNING_FIRST_TIME, false)
					.commit();
			Crouton.makeText(this, R.string.first_time_information, Style.INFO)
					.show();
		}

		// Get intent, action and MIME type
		Intent intent = getIntent();
		if (intent != null) {
			String action = intent.getAction();
			String type = intent.getType();

			if (Intent.ACTION_SEND.equals(action) && type != null) {
				if (type.startsWith("image/")) {
					// Handle single image being sent
					handleSendImage(intent);
				}
			} else if (Intent.ACTION_SEND_MULTIPLE.equals(action)
					&& type != null) {
				if (type.startsWith("image/")) {
					// Handle multiple images being sent
					handleSendMultipleImages(intent);
				}
			}
		}

	}

	/**
	 * handles processing images when only one is sent
	 * 
	 * @param intent
	 */
	void handleSendImage(Intent intent) {

		Uri imageUri = (Uri) intent.getParcelableExtra(Intent.EXTRA_STREAM);
		if (imageUri != null) {
			String selectedImagePath = Util.getPath(this, imageUri);
			new FrameGenerator(this, true).execute(selectedImagePath);
		}
	}

	/**
	 * handles processing multiple images
	 * 
	 * @param intent
	 */
	void handleSendMultipleImages(Intent intent) {
		ArrayList<Uri> imageUris = intent
				.getParcelableArrayListExtra(Intent.EXTRA_STREAM);

		ArrayList<String> imagePaths = new ArrayList<String>();

		for (int i = 0; i < imageUris.size(); i++) {
			imagePaths.add(Util.getPath(this, imageUris.get(i)));
		}

		new FrameGeneratorMultiple(this).execute(imagePaths);

	}

	private class FrameGenerator extends AsyncTask<String, Integer, String> {
		Boolean fromShare;
		private Activity mActivity;

		FrameGenerator(Activity mActivity, Boolean fromShare) {
			this.fromShare = fromShare;
			this.mActivity = mActivity;
		}

		@Override
		protected void onPreExecute() {
			setSupportProgressBarIndeterminateVisibility(true);
			super.onPreExecute();
		}

		@Override
		protected String doInBackground(String... params) {
			if (params.length != 1) {
				return null;
			}

			String path = drawImage(params[0], fromShare);

			return path;
		}

		@Override
		protected void onPostExecute(String result) {
			setSupportProgressBarIndeterminateVisibility(false);

			if (result != null) {
				Log.i(LOGTAG, "result was not null - the path is " + result);

				galleryAddPic(result);

				viewImage(result);

			} else {
				Crouton.makeText(mActivity, R.string.error, Style.ALERT).show();
			}

			super.onPostExecute(result);
		}
	}

	private class FrameGeneratorMultiple extends
			AsyncTask<ArrayList<String>, Integer, String> {

		private Activity mActivity;
		private int successCount = 0;
		private int maxCount = 0;

		FrameGeneratorMultiple(Activity activity) {
			mActivity = activity;
		}

		@Override
		protected void onPreExecute() {
			setSupportProgress(0);
			setSupportProgressBarIndeterminateVisibility(true);
			super.onPreExecute();
		}

		@Override
		protected String doInBackground(ArrayList<String>... params) {
			if (params.length != 1) {
				return null;
			}

			ArrayList<String> imagePaths = params[0];
			maxCount = imagePaths.size();

			for (int i = 0; i < imagePaths.size(); i++) {
				String result = drawImage(imagePaths.get(i), true);

				if (result != null) {
					Log.i(LOGTAG, "result was not null - the path is " + result);
					galleryAddPic(result);
					successCount++;
					publishProgress((Window.PROGRESS_END - Window.PROGRESS_START)
							/ imagePaths.size() * (i + 1));
				}
			}

			return null;
		}

		@Override
		protected void onProgressUpdate(Integer... progress) {
			if (progress.length != 1) {
				Log.e(LOGTAG,
						"Progress should have only one parameter, but has "
								+ progress.length);
				return;
			}
			setSupportProgress(progress[0]);
		}

		@Override
		protected void onPostExecute(String result) {

			setSupportProgressBarIndeterminateVisibility(false);
			setSupportProgress(0);

			if (successCount != maxCount) {
				Crouton.makeText(
						mActivity,
						String.format(((Context) mActivity).getResources()
								.getString(R.string.images_not_generated),
								maxCount - successCount), Style.CONFIRM).show();
			}

			Crouton.makeText(
					mActivity,
					String.format(((Context) mActivity).getResources()
							.getString(R.string.images_generated), successCount),
					Style.CONFIRM).show();

			super.onPostExecute(result);
		}
	}

	private void galleryAddPic(String path) {
		Intent mediaScanIntent = new Intent(
				Intent.ACTION_MEDIA_SCANNER_SCAN_FILE);
		File f = new File(path);
		Uri contentUri = Uri.fromFile(f);
		mediaScanIntent.setData(contentUri);
		this.sendBroadcast(mediaScanIntent);
	}

	public void onItemSelected(AdapterView parent, View v, int position, long id) {

		Device device = mDeviceList.get(position);

		mDeviceTitleText.setText(device.getTitle());

		mDeviceSizeText.setText(device.getPhysicalSize() + "\" @ "
				+ device.getDensity() + "dpi");

		mDeviceDimensionsText.setText(device.getPortSize()[0] + "x"
				+ device.getPortSize()[1]);

	}

	public void onItemClick(AdapterView<?> parent, View view, int position,
			long id) {

		if (lastClick == position) {
			Editor edit = mPrefs.edit();
			edit.putInt(AppConstants.KEY_DEVICE_POSITION, position);
			edit.commit();

			Crouton.makeText(
					this,
					String.format(
							getResources().getString(R.string.device_saved),
							mDeviceList.get(position).getTitle()),
					Style.CONFIRM).show();

			// set to any number so that lastClick is reset
			lastClick = 34782730;
		} else {
			lastClick = position;
		}

	}

	public void onNothingSelected(AdapterView parent) {
	}

	@Override
	public boolean onCreateOptionsMenu(Menu menu) {
		getSupportMenuInflater().inflate(R.menu.activity_main, menu);

		return true;
	}

	@Override
	public boolean onOptionsItemSelected(MenuItem item) {
		switch (item.getItemId()) {
		case R.id.menu_generate:
			Intent intent = new Intent();
			intent.setType("image/*");
			intent.setAction(Intent.ACTION_GET_CONTENT);
			startActivityForResult(
					Intent.createChooser(intent, "Select Picture"),
					RESULT_SELECT_PICTURE);
			return true;
		case R.id.menu_about:
			FragmentManager fm = getSupportFragmentManager();
			AboutDialogFragment aboutDialog = new AboutDialogFragment();
			aboutDialog.show(fm, "fragment_edit_name");
			return true;
		case R.id.menu_help:
			goToWebPage(URL_PROJECT_PAGE);
			return true;
		}

		return super.onOptionsItemSelected(item);
	}

	public void onClick(View v) {
		switch (v.getId()) {
		case R.id.device_title:
			String url = mDeviceList.get(mGallery.getSelectedItemPosition())
					.getUrl();
			goToWebPage(url);
			break;
		case R.id.textView_attribution:
			goToWebPage(URL_ATTRIBUTION_INFO);
			break;
		case R.id.textView_credits:
			goToWebPage(URL_ASSET_STUDIO);
			break;
		case R.id.textView_donate:
			goToWebPage(URL_DONATION);
			break;
		case R.id.textView_issues:
			goToWebPage(URL_ISSUES);
			break;
		}
	}

	private void goToWebPage(String url) {
		Intent i = new Intent(Intent.ACTION_VIEW);
		i.setData(Uri.parse(url));
		startActivity(i);
	}

	@Override
	protected void onActivityResult(int requestCode, int resultCode, Intent data) {

		if (resultCode == RESULT_OK) {
			if (requestCode == RESULT_SELECT_PICTURE) {
				Uri selectedImageUri = data.getData();
				String selectedImagePath = Util.getPath(this, selectedImageUri);
				new FrameGenerator(this, false).execute(selectedImagePath);
			}
		}

		super.onActivityResult(requestCode, resultCode, data);

	}

	/**
	 * draws images and saves them to SD card using the Util class
	 * 
	 * @param selectedImagePath
	 * @param fromShare
	 *            : determines whethere to use globally saved device (from
	 *            share) or if to use currently selected device(not from share)
	 * @return path to saved file
	 */
	private String drawImage(String selectedImagePath, boolean fromShare) {

		Device device = fromShare ? mDeviceList.get(mPrefs.getInt(
				AppConstants.KEY_DEVICE_POSITION, 0)) : mDeviceList
				.get(mGallery.getSelectedItemPosition());

		Bitmap screenshot = null;

		try {
			screenshot = Util.convertToMutable(BitmapFactory
					.decodeFile(selectedImagePath));
		} catch (IOException e) {
			e.printStackTrace();
			Crouton.makeText(this, R.string.error_unable_to_save, Style.ALERT)
					.show();
			return null;
		}

		String pathToFile = null;

		try {
			pathToFile = Util.combine(this, device, screenshot,
					mPrefs.getBoolean(AppConstants.KEY_WITH_SHADOW, true),
					mPrefs.getBoolean(AppConstants.KEY_WITH_GLARE, true));
		} catch (IOException e) {
			e.printStackTrace();
			Crouton.makeText(this, R.string.error_unable_to_save, Style.ALERT)
					.show();
			return null;
		} catch (UnmatchedDimensionsException e) {
			e.printStackTrace();
			Crouton.makeText(this, R.string.error_dimensions_not_matched,
					Style.ALERT).show();
			return null;
		}

		screenshot.recycle();

		return pathToFile;
	}

	/**
	 * View the image with given path
	 * 
	 * @param pathToFile
	 */
	private void viewImage(String pathToFile) {

		if (pathToFile == null) {
			return;
		}
		Intent intent = new Intent();
		intent.setAction(Intent.ACTION_VIEW);
		intent.setDataAndType(Uri.parse("file://" + pathToFile), "image/*");
		startActivity(intent);
	}

	public void onCheckedChanged(CompoundButton buttonView, boolean isChecked) {
		Editor edit = mPrefs.edit();

		switch (buttonView.getId()) {
		case R.id.shadow:
			edit.putBoolean(AppConstants.KEY_WITH_SHADOW, isChecked);
			if (isChecked) {
				Crouton.makeText(this, R.string.shadow_enabled, Style.CONFIRM)
						.show();
			} else {
				Crouton.makeText(this, R.string.shadow_disabled, Style.ALERT)
						.show();
			}
			break;
		case R.id.glare:
			edit.putBoolean(AppConstants.KEY_WITH_GLARE, isChecked);
			if (isChecked) {
				Crouton.makeText(this, R.string.glare_enabled, Style.CONFIRM)
						.show();
			} else {
				Crouton.makeText(this, R.string.glare_disabled, Style.ALERT)
						.show();
			}
			break;
		}

		edit.commit();

	}

	private class ThumbnailAdapter extends BaseAdapter {
		private LayoutInflater mLayoutInflater;

		ThumbnailAdapter(MainActivity context) {
			mLayoutInflater = context.getLayoutInflater();
		}

		public int getCount() {
			return mThumbs.size();
		}

		public Object getItem(int position) {
			return position;
		}

		public long getItemId(int position) {
			return position;
		}

		public View getView(int position, View convertView, ViewGroup parent) {
			ImageView image;

			if (convertView == null) {
				image = (ImageView) mLayoutInflater.inflate(
						R.layout.device_item, parent, false);
			} else {
				image = (ImageView) convertView;
			}

			image.setImageResource(mThumbs.get(position));
			image.getDrawable().setDither(true);
			return image;
		}
	}

}
