package com.psrivastava.deviceframegenerator;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileChannel.MapMode;
import java.util.ArrayList;
import java.util.Calendar;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.graphics.Bitmap;
import android.graphics.Bitmap.CompressFormat;
import android.graphics.Bitmap.Config;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.net.Uri;
import android.os.Environment;
import android.provider.MediaStore;
import android.provider.MediaStore.MediaColumns;
import android.support.v4.content.CursorLoader;
import android.util.Log;

import com.psrivastava.deviceframegenerator.devices.Device;
import com.psrivastava.deviceframegenerator.devices.GalaxyNexus;
import com.psrivastava.deviceframegenerator.devices.HTCOneX;
import com.psrivastava.deviceframegenerator.devices.Nexus7;
import com.psrivastava.deviceframegenerator.devices.NexusS;
import com.psrivastava.deviceframegenerator.devices.SamsungGalaxyNote;
import com.psrivastava.deviceframegenerator.devices.SamsungGalaxyS3;
import com.psrivastava.deviceframegenerator.devices.SamsungGalaxyTab27inch;
import com.psrivastava.deviceframegenerator.devices.Xoom;

public class Util {

	// TODO: come up with a shorter name for the app
	private static final String LOG_PREFIX = "dfg_";
	private static final int LOG_PREFIX_LENGTH = LOG_PREFIX.length();
	private static final int MAX_LOG_TAG_LENGTH = 23;

	private static final String LOGTAG = makeLogTag(Util.class);

	/**
	 * Folder for storage finished images.
	 */
	public static final String APP_FOLDER = "/Device-Frame-Generator/";

	/**
	 * Storage directory, obtained by combining {@link #APP_FOLDER} and the
	 * environment path
	 */
	public static final String STORAGE_DIRECTORY = Environment
			.getExternalStoragePublicDirectory(Environment.DIRECTORY_PICTURES)
			.toString()
			+ APP_FOLDER;

	/**
	 * TODO: for future feature, show a gallery within application from this
	 * folder allow user to generate images without leaving app (eliminate
	 * onActivityResult) use Contextual action bar for batch processing form
	 * within app
	 */
	public static final String SCREENSHOT_FOLDER = "/Screenshots/";

	public static final ArrayList<Device> mDeviceList = getAllDevices();

	/**
	 * Return name of the class to use for logging
	 * 
	 * @param cls
	 * @return
	 */
	public static String makeLogTag(Class cls) {
		return makeLogTag(cls.getSimpleName());
	}

	public static String makeLogTag(String str) {
		if (str.length() > MAX_LOG_TAG_LENGTH - LOG_PREFIX_LENGTH) {
			return LOG_PREFIX
					+ str.substring(0, MAX_LOG_TAG_LENGTH - LOG_PREFIX_LENGTH
							- 1);
		}

		return LOG_PREFIX + str;
	}

	/**
	 * Checks if storage is available
	 * 
	 * @return true if storage is available
	 */
	public static boolean checkStorage() {
		boolean mExternalStorageAvailable = false;
		boolean mExternalStorageWriteable = false;
		String state = Environment.getExternalStorageState();

		if (Environment.MEDIA_MOUNTED.equals(state)) {
			// We can read and write the media
			mExternalStorageAvailable = mExternalStorageWriteable = true;
		} else if (Environment.MEDIA_MOUNTED_READ_ONLY.equals(state)) {
			// We can only read the media
			mExternalStorageAvailable = true;
			mExternalStorageWriteable = false;
		} else {
			// Something else is wrong. It may be one of many other states, but
			// all we need
			// to know is we can neither read nor write
			mExternalStorageAvailable = mExternalStorageWriteable = false;
		}

		Log.i(LOGTAG, mExternalStorageAvailable ? "Storage available"
				: "Storage Unavailable");
		Log.i(LOGTAG, mExternalStorageWriteable ? "Storage writeable"
				: "Storage not writeable");

		return (mExternalStorageAvailable && mExternalStorageWriteable);
	}

	/**
	 * 
	 * frames the given screenshot using the parameters
	 * 
	 * @param context
	 * @param device
	 *            : the device for framing
	 * @param orientation
	 *            : orientation of screenshot
	 * @param screenshot
	 *            : screenshot to be framed
	 * @param opts
	 *            : BitmapFactory.Options to make Bitmaps mutable
	 * @param withShadow
	 *            : If should be generated with shadow
	 * @param withGlare
	 *            : if should be generated with glare
	 * 
	 * @return the path to the saved image
	 * @throws IOException
	 * @throws UnmatchedDimensionsException
	 */
	public static String combine(Context context, Device device,
			Bitmap screenshot, Boolean withShadow, Boolean withGlare)
			throws IOException, UnmatchedDimensionsException {

		String orientation = Util.checkDimensions(device, screenshot);

		int[] offset = orientation.compareTo("port") == 0 ? device
				.getPortOffset() : device.getLandOffset();

		String shadowString = device.getId() + "_" + orientation + "_shadow";
		String backString = device.getId() + "_" + orientation + "_back";
		String foreString = device.getId() + "_" + orientation + "_fore";

		File folder = new File(STORAGE_DIRECTORY);
		folder.mkdirs();

		Bitmap shadow = convertToMutable(BitmapFactory.decodeResource(
				context.getResources(),
				context.getResources().getIdentifier(shadowString, "drawable",
						context.getPackageName())));
		Bitmap back = convertToMutable(BitmapFactory.decodeResource(
				context.getResources(),
				context.getResources().getIdentifier(backString, "drawable",
						context.getPackageName())));
		Bitmap fore = convertToMutable(BitmapFactory.decodeResource(
				context.getResources(),
				context.getResources().getIdentifier(foreString, "drawable",
						context.getPackageName())));

		Canvas comboImage = new Canvas(back);
		if (withShadow) {
			comboImage.drawBitmap(shadow, 0f, 0f, null);
		}

		comboImage.drawBitmap(screenshot, offset[0], offset[1], null);
		if (withGlare) {
			comboImage.drawBitmap(fore, 0f, 0f, null);
		}

		// To write the file out to the SDCard:
		OutputStream os = null;
		File f = new File(STORAGE_DIRECTORY);
		File[] files = f.listFiles();

		Calendar c = Calendar.getInstance();
		String fileName = device.getId() + "_" + c.get(Calendar.YEAR) + "-"
				+ (c.get(Calendar.MONTH) + 1) + "-"
				+ c.get(Calendar.DAY_OF_MONTH) + "-" + files.length + ".png";
		Log.i(LOGTAG, "Image was saved as " + STORAGE_DIRECTORY + fileName);

		os = new FileOutputStream(STORAGE_DIRECTORY + fileName);
		back.compress(CompressFormat.PNG, 50, os);

		ContentValues values = new ContentValues();
		values.put(MediaColumns.TITLE, STORAGE_DIRECTORY + fileName);
		values.put(MediaColumns.DATE_ADDED, System.currentTimeMillis());
		values.put(MediaColumns.MIME_TYPE, "image/png");

		shadow.recycle();
		fore.recycle();
		back.recycle();

		return STORAGE_DIRECTORY + fileName;

	}

	/**
	 * checks if screenshot matches size of device
	 * 
	 * @param device
	 * @param screenshot
	 * @return null if no match, port if matched to portrait and land if matched
	 *         to landscape
	 * @throws UnmatchedDimensionsException
	 */
	public static String checkDimensions(Device device, Bitmap screenshot)
			throws UnmatchedDimensionsException {

		Log.i(LOGTAG, "Screenshot height is " + screenshot.getHeight()
				+ " and width is " + screenshot.getWidth());

		if (screenshot.getHeight() == device.getPortSize()[1]
				&& screenshot.getWidth() == device.getPortSize()[0]) {
			return "port";
		} else if (screenshot.getHeight() == device.getPortSize()[0]
				&& screenshot.getWidth() == device.getPortSize()[1]) {
			return "land";
		}

		throw new UnmatchedDimensionsException();

	}

	/**
	 * gets path of selected image
	 * 
	 * @param context
	 * @param uri
	 * @return path of selected image
	 */
	public static String getPath(Context context, Uri uri) {
		String[] projection = { MediaStore.Images.Media.DATA };

		CursorLoader cursorLoader = new CursorLoader(context, uri, projection,
				null, null, null);
		Cursor cursor = cursorLoader.loadInBackground();

		int column_index = cursor
				.getColumnIndexOrThrow(MediaStore.Images.Media.DATA);
		cursor.moveToFirst();
		return cursor.getString(column_index);
	}

	/**
	 * Converts a immutable bitmap to a mutable bitmap. This operation doesn't
	 * allocates more memory that there is already allocated.
	 * 
	 * @param imgIn
	 *            - Source image. It will be released, and should not be used
	 *            more
	 * @return a copy of imgIn, but mutable.
	 * @throws IOException
	 */
	public static Bitmap convertToMutable(Bitmap imgIn) throws IOException {
		// this is the file going to use temporally to save the bytes.
		// This file will not be a image, it will store the raw image data.
		File file = new File(Environment.getExternalStorageDirectory()
				+ File.separator + "temp.tmp");

		// Open an RandomAccessFile
		// Make sure you have added uses-permission
		// android:name="android.permission.WRITE_EXTERNAL_STORAGE"
		// into AndroidManifest.xml file
		RandomAccessFile randomAccessFile = new RandomAccessFile(file, "rw");

		// get the width and height of the source bitmap.
		int width = imgIn.getWidth();
		int height = imgIn.getHeight();
		Config type = imgIn.getConfig();

		// Copy the byte to the file
		// Assume source bitmap loaded using options.inPreferredConfig =
		// Config.ARGB_8888;
		FileChannel channel = randomAccessFile.getChannel();
		MappedByteBuffer map = channel.map(MapMode.READ_WRITE, 0,
				imgIn.getRowBytes() * height);
		imgIn.copyPixelsToBuffer(map);
		// recycle the source bitmap, this will be no longer used.
		imgIn.recycle();
		System.gc();// try to force the bytes from the imgIn to be released

		// Create a new bitmap to load the bitmap again. Probably the memory
		// will be available.
		imgIn = Bitmap.createBitmap(width, height, type);
		map.position(0);
		// load it back from temporary
		imgIn.copyPixelsFromBuffer(map);
		// close the temporary file and channel , then delete that also
		channel.close();
		randomAccessFile.close();

		// delete the temp file
		file.delete();

		return imgIn;
	}

	public static ArrayList<Device> getAllDevices() {
		ArrayList<Device> deviceList = new ArrayList<Device>();

		deviceList.add(new Device(GalaxyNexus.getId(), GalaxyNexus.getTitle(),
				GalaxyNexus.getUrl(), GalaxyNexus.getPhysicalsize(),
				GalaxyNexus.getDensity(), GalaxyNexus.getLandoffset(),
				GalaxyNexus.getPortoffset(), GalaxyNexus.getPortsize(),
				GalaxyNexus.getThumbnail()));

		deviceList.add(new Device(Nexus7.getId(), Nexus7.getTitle(), Nexus7
				.getUrl(), Nexus7.getPhysicalsize(), Nexus7.getDensity(),
				Nexus7.getLandoffset(), Nexus7.getPortoffset(), Nexus7
						.getPortsize(), Nexus7.getThumbnail()));

		deviceList.add(new Device(NexusS.getId(), NexusS.getTitle(), NexusS
				.getUrl(), NexusS.getPhysicalsize(), NexusS.getDensity(),
				NexusS.getLandoffset(), NexusS.getPortoffset(), NexusS
						.getPortsize(), NexusS.getThumbnail()));

		deviceList.add(new Device(SamsungGalaxyNote.getId(), SamsungGalaxyNote
				.getTitle(), SamsungGalaxyNote.getUrl(), SamsungGalaxyNote
				.getPhysicalsize(), SamsungGalaxyNote.getDensity(),
				SamsungGalaxyNote.getLandoffset(), SamsungGalaxyNote
						.getPortoffset(), SamsungGalaxyNote.getPortsize(),
				SamsungGalaxyNote.getThumbnail()));

		deviceList.add(new Device(SamsungGalaxyS3.getId(), SamsungGalaxyS3
				.getTitle(), SamsungGalaxyS3.getUrl(), SamsungGalaxyS3
				.getPhysicalsize(), SamsungGalaxyS3.getDensity(),
				SamsungGalaxyS3.getLandoffset(), SamsungGalaxyS3
						.getPortoffset(), SamsungGalaxyS3.getPortsize(),
				SamsungGalaxyS3.getThumbnail()));

		deviceList.add(new Device(SamsungGalaxyTab27inch.getId(),
				SamsungGalaxyTab27inch.getTitle(), SamsungGalaxyTab27inch
						.getUrl(), SamsungGalaxyTab27inch.getPhysicalsize(),
				SamsungGalaxyTab27inch.getDensity(), SamsungGalaxyTab27inch
						.getLandoffset(), SamsungGalaxyTab27inch
						.getPortoffset(), SamsungGalaxyTab27inch.getPortsize(),
				SamsungGalaxyTab27inch.getThumbnail()));

		deviceList.add(new Device(Xoom.getId(), Xoom.getTitle(), Xoom.getUrl(),
				Xoom.getPhysicalsize(), Xoom.getDensity(),
				Xoom.getLandoffset(), Xoom.getPortoffset(), Xoom.getPortsize(),
				Xoom.getThumbnail()));

		deviceList.add(new Device(HTCOneX.getId(), HTCOneX.getTitle(), HTCOneX
				.getUrl(), HTCOneX.getPhysicalsize(), HTCOneX.getDensity(),
				HTCOneX.getLandoffset(), HTCOneX.getPortoffset(), HTCOneX
						.getPortsize(), HTCOneX.getThumbnail()));

		/*
		 * Galaxy S3 in Pebble Blue, not complete yet
		 * 
		 * deviceList.add(new Device(SamsungGalaxyS3Blue.getId(),
		 * SamsungGalaxyS3Blue.getTitle(), SamsungGalaxyS3Blue.getUrl(),
		 * SamsungGalaxyS3Blue.getPhysicalsize(), SamsungGalaxyS3Blue
		 * .getDensity(), SamsungGalaxyS3Blue.getLandoffset(),
		 * SamsungGalaxyS3Blue.getPortoffset(), SamsungGalaxyS3Blue
		 * .getPortsize(), SamsungGalaxyS3Blue.getThumbnail()));
		 */
		return deviceList;
	}

	public static ArrayList<Integer> getAllThubnails() {
		ArrayList<Integer> thumbnails = new ArrayList<Integer>();

		for (Device device : mDeviceList) {
			thumbnails.add(device.getThumbnail());
		}

		return thumbnails;
	}
}
