package com.seafile.seadroid.data;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.seafile.seadroid.SeadroidApplication;
import com.seafile.seadroid.SeafConnection;
import com.seafile.seadroid.SeafException;
import com.seafile.seadroid.Utils;
import com.seafile.seadroid.account.Account;

import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.Environment;
import android.util.Log;

public class DataManager {

    public static String getExternalRootDirectory() {
        if (Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED)) {
            File extDir = new File(Environment.getExternalStorageDirectory().getAbsolutePath() + "/Seafile/");
            if (extDir.mkdirs() || extDir.exists()) {
                return extDir.getAbsolutePath();
            } else {
                throw new RuntimeException("Couldn't create external directory");
            }
        } else {
            throw new RuntimeException("External Storage is currently not available");
        }
    }

    public static String getExternalTempDirectory() {
        String root = getExternalRootDirectory();
        File tmpDir = new File(root + "/" + "temp");
        if (tmpDir.exists())
            return tmpDir.getAbsolutePath();
        else {
            if (tmpDir.mkdirs() == false)
                throw new RuntimeException("Couldn't create external temp directory");
            else
                return tmpDir.getAbsolutePath();
        }
    }

    public static String getThumbDirectory() {
        String root = SeadroidApplication.getAppContext().getFilesDir().getAbsolutePath();
        File tmpDir = new File(root + "/" + "thumb");
        if (tmpDir.exists())
            return tmpDir.getAbsolutePath();
        else {
            if (tmpDir.mkdirs() == false)
                throw new RuntimeException("Couldn't create thumb directory");
            else
                return tmpDir.getAbsolutePath();
        }
    }

    public static String getExternalCacheDirectory() {
        String root = getExternalRootDirectory();
        File tmpDir = new File(root + "/" + "cache");
        if (tmpDir.exists())
            return tmpDir.getAbsolutePath();
        else {
            if (tmpDir.mkdirs() == false)
                throw new RuntimeException("Couldn't create external temp directory");
            else
                return tmpDir.getAbsolutePath();
        }
    }

    static public String constructFileName(String path, String oid) {
        String filename = path.substring(path.lastIndexOf("/") + 1);
        if (filename.contains(".")) {
            String purename = filename.substring(0, filename.lastIndexOf('.'));
            String suffix = filename.substring(filename.lastIndexOf('.') + 1);
            return purename + "-" + oid.substring(0, 8) + "." + suffix;
        } else {
            return filename + "-" + oid.substring(0, 8);
        }

    }

    static public File getFileForFileCache(String path, String oid) {
        String p = getExternalRootDirectory() + "/" + constructFileName(path, oid);
        return new File(p);
    }

    static public File getTempFile(String path, String oid) {
        String p = getExternalTempDirectory() + "/" + constructFileName(path, oid);
        return new File(p);
    }

    static public File getThumbFile(String path, String oid) {
        String filename = path.substring(path.lastIndexOf("/") + 1);
        String purename = filename.substring(0, filename.lastIndexOf('.'));
        String p = getThumbDirectory() + "/" + purename + "-"
                + oid.substring(0, 8) + ".png";
        return new File(p);
    }

    // Obtain a cache file for storing a directory with oid
    static public File getFileForDirentsCache(String oid) {
        return new File(getExternalCacheDirectory() + "/" + oid);
    }

    static public final int MAX_GEN_CACHE_THUMB = 1000000;  // Only generate thumb cache for files less than 1MB
    static public final int MAX_DIRECT_SHOW_THUMB = 100000;  // directly show thumb

    static public void calculateThumbnail(String fileName, String fileID) {
        try {
            final int THUMBNAIL_SIZE = 72;

            File file = getFileForFileCache(fileName, fileID);
            if (!file.exists())
                return;
            if (file.length() > MAX_GEN_CACHE_THUMB)
                return;

            Bitmap imageBitmap = BitmapFactory.decodeStream(new FileInputStream(file));
            imageBitmap = Bitmap.createScaledBitmap(imageBitmap, THUMBNAIL_SIZE,
                    THUMBNAIL_SIZE, false);
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            imageBitmap.compress(Bitmap.CompressFormat.PNG, 100, baos);
            byte[] byteArray = baos.toByteArray();
            File thumb = getThumbFile(fileName, fileID);
            FileOutputStream out = new FileOutputStream(thumb);
            out.write(byteArray);
            out.close();
        } catch (Exception ex) {

        }
    }

    static public Bitmap getThumbnail(String fileName, String fileID) {
        try {
            final int THUMBNAIL_SIZE = 72;

            File file = getFileForFileCache(fileName, fileID);
            if (!file.exists())
                return null;

            Bitmap imageBitmap = BitmapFactory.decodeStream(new FileInputStream(file));
            imageBitmap = Bitmap.createScaledBitmap(imageBitmap, THUMBNAIL_SIZE,
                    THUMBNAIL_SIZE, false);
            return imageBitmap;
        } catch (Exception ex) {
            return null;
        }
    }


    private static final String DEBUG_TAG = "DataManager";

    private SeafConnection sc;
    private Account account;
    private CachedFileDbHelper cdbHelper;

    HashMap<String, String> pathObjectIDMap = new HashMap<String, String>();
    List<SeafRepo> reposCache = null;

    public DataManager(Account act) {
        account = act;
        sc = new SeafConnection(act);
        cdbHelper = new CachedFileDbHelper(SeadroidApplication.getAppContext());
    }

    public Account getAccount() {
        return sc.getAccount();
    }

    private File getFileForReposCache() {
        String filename = "repos-" + (account.server + account.email).hashCode() + ".dat";
        return new File(getExternalCacheDirectory() + "/" +
                filename);
    }

    private List<SeafRepo> parseRepos(String json) {
        try {
            // may throw ClassCastException
            JSONArray array = Utils.parseJsonArray(json);
            if (array.length() == 0)
                return null;
            ArrayList<SeafRepo> repos = new ArrayList<SeafRepo>();
            for (int i = 0; i < array.length(); i++) {
                JSONObject obj = array.getJSONObject(i);
                SeafRepo repo = SeafRepo.fromJson(obj);
                if (repo != null)
                    repos.add(repo);
            }
            return repos;
        } catch (JSONException e) {
            Log.d(DEBUG_TAG, "repos: parse json error");
            return null;
        } catch (Exception e) {
            // other exception, for example ClassCastException
            return null;
        }
    }

    public List<SeafRepo> getCachedRepos() {
        return reposCache;
    }

    public SeafRepo getCachedRepo(int position) {
        return reposCache.get(position);
    }

    public SeafRepo getCachedRepoByID(String id) {
        List<SeafRepo> cachedRepos = getCachedRepos();
        if (cachedRepos == null) {
            return null;
        }

        for (SeafRepo repo: cachedRepos) {
            if (repo.getID().equals(id)) {
                return repo;
            }
        }

        return null;
    }

    public List<SeafRepo> getReposFromCache() {
        if (reposCache != null)
            return reposCache;

        File cache = getFileForReposCache();
        if (cache.exists()) {
            String json = Utils.readFile(cache);
            reposCache = parseRepos(json);
            return reposCache;
        }
        return null;
    }

    public List<SeafRepo> getRepos() throws SeafException {
        if (!Utils.isNetworkOn()) {
            if (reposCache != null)
                return reposCache;

            File cache = getFileForReposCache();
            if (cache.exists()) {
                String json = Utils.readFile(cache);
                reposCache = parseRepos(json);
                return reposCache;
            }
        }

        String json = sc.getRepos();
        if (json == null)
            return null;
        reposCache = parseRepos(json);

        try {
            File cache = getFileForReposCache();
            Utils.writeFile(cache, json);
        } catch (IOException e) {
            // ignore
        }

        return reposCache;
    }

    public interface ProgressMonitor {
        public void onProgressNotify(long total);
        boolean isCancelled();
    }

    public File getFile(String repoID, String path, String oid, ProgressMonitor monitor)
            throws SeafException {
        String p = getExternalRootDirectory() + "/" + constructFileName(path, oid);
        File f = new File(p);
        if (f.exists())
            return f;
        f = sc.getFile(repoID, path, oid, monitor);
        if (f != null)
            addCachedFile(repoID, path, oid, f);
        return f;
    }

    private List<SeafDirent> parseDirents(String json) {
        try {
            JSONArray array = Utils.parseJsonArray(json);
            if (array == null)
                return null;

            ArrayList<SeafDirent> dirents = new ArrayList<SeafDirent>();
            for (int i = 0; i < array.length(); i++) {
                JSONObject obj = array.getJSONObject(i);
                SeafDirent de = SeafDirent.fromJson(obj);
                if (de != null)
                    dirents.add(de);
            }
            return dirents;
        } catch (JSONException e) {
            return null;
        }
    }

    public List<SeafDirent> getDirents(String repoID,
            String path, String objectID) throws SeafException {
        //Log.d(DEBUG_TAG, "getDirents " + repoID + ":" + path + ", " + objectID);

        if (objectID != null) {
            // put the mapping to cache for later usage.
            pathObjectIDMap.put(repoID + path, objectID);
        } else {
            objectID = pathObjectIDMap.get(repoID + path);
        }

        if (objectID != null) {
            File cache = getFileForDirentsCache(objectID);
            if (cache.exists()) {
                String json = Utils.readFile(cache);
                return parseDirents(json);
            }
        }

        String json = sc.getDirents(repoID, path);
        if (json == null)
            return null;
        List<SeafDirent> dirents = parseDirents(json);

        if (objectID != null) {
            try {
                File cache = getFileForDirentsCache(objectID);
                Utils.writeFile(cache, json);
            } catch (IOException e) {
                // ignore
            }
        }

        return dirents;
    }


    public List<SeafCachedFile> getCachedFiles() {
        return cdbHelper.getItems(account);
    }

    public void addCachedFile(String repo, String path, String fileID, File file) {
        SeafCachedFile item = new SeafCachedFile();
        item.repo = repo;
        item.path = path;
        item.fileID = fileID;
        item.ctime = file.lastModified();
        item.accountSignature = account.getSignature();
        cdbHelper.saveItem(item);
    }

    public void removeCachedFile(SeafCachedFile cf) {
        cf.file.delete();
        cdbHelper.deleteItem(cf);
    }

    public void setPassword(String repoID, String passwd) {
        try {
            sc.setPassword(repoID, passwd);
        } catch (SeafException e) {
            // ignore
        }
    }

    public void uploadFile(String repoID, String dir, String filePath,
            ProgressMonitor monitor) throws SeafException {
        sc.uploadFile(repoID, dir, filePath, monitor);
    }

    /** Remove cached dirents from dir to the root.
     */
    public void invalidateCache(String repoID, String dir) {
        if (repoID == null || dir == null)
            return;

        String d = dir;
        while (true) {
            String objectID = pathObjectIDMap.get(repoID + d);
            if (objectID != null) {
                File cache = getFileForDirentsCache(objectID);
                if (cache.exists())
                    cache.delete();
            }
            pathObjectIDMap.remove(repoID + d);
            if (d.equals("/"))
                break;
            d = Utils.getParentPath(d);
        }
    }

}
