package com.sgr_b2.compass.activities;

import android.app.Service;
import android.content.ContentValues;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.SharedPreferences;
import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.location.Location;
import android.location.LocationListener;
import android.net.Uri;
import android.os.Bundle;
import android.os.IBinder;
import android.os.Vibrator;
import android.preference.PreferenceManager;
import android.telephony.SmsManager;

import com.sgr_b2.compass.Config;
import com.sgr_b2.compass.R;
import com.sgr_b2.compass.db.POI;
import com.sgr_b2.compass.io.SMSText;
import com.sgr_b2.compass.jni.CalleeLookup;
import com.sgr_b2.compass.jni.DistressDetector;
import com.sgr_b2.compass.jni.DistressDynamo;
import com.sgr_b2.compass.phone.CallsReader;
import com.sgr_b2.compass.phone.CancelDistressReceiver;
import com.sgr_b2.compass.phone.DeviceWrapper;
import com.sgr_b2.compass.phone.PowerButtonBroadcastReceiver;


public class DistressCallService
	extends Service
	implements SensorEventListener, LocationListener,
	SharedPreferences.OnSharedPreferenceChangeListener {

	public static String ACTION_CANCEL = "com.sgr_b2.compass.activities.DistressCallService.CANCEL_DISTRESS";

	public static long[] VIBRATE_ARMED = { 0, 200, 0 }; // msecs
	public static long[] VIBRATE_DISTRESS = { 0, 1000, 0 }; // msecs
	public static long[] VIBRATE_SENT = { 0, 200, 500, 200, 500, 200, 0 }; // msecs

	private static int STATE_DISARMED = 0;
	private static int STATE_ARMED = 1;
	private static int STATE_CALLING = 2;

	private int state = STATE_DISARMED;
	private DistressDynamo dynamo = new DistressDynamo(); // no need to dispose() since service is always running

	private DeviceWrapper device_wrapper = null;
	private DistressDetector detector = new DistressDetector();
	private PowerButtonBroadcastReceiver power_button_listener = new PowerButtonBroadcastReceiver(this, this.detector);
	private CancelDistressReceiver cancel_listener = new CancelDistressReceiver(this);

	public static boolean indistress = false; // dammit, i don't need BROADCAST_STICKY permission for my app

	@Override
	public void onCreate() {
		super.onCreate();

		this.device_wrapper = new DeviceWrapper(getBaseContext());
		DistressCallService.indistress = false;

		setupPowerButton();

		final SharedPreferences shared_pref = PreferenceManager.getDefaultSharedPreferences(this);
		shared_pref.registerOnSharedPreferenceChangeListener(this);
	}

	@Override
	public void onDestroy() {
		final SharedPreferences shared_pref = PreferenceManager.getDefaultSharedPreferences(this);
		shared_pref.unregisterOnSharedPreferenceChangeListener(this);

		releasePowerButton();
		releaseSensors();
		releaseGPS();

		super.onDestroy();
	}

	@Override
	public IBinder onBind(Intent intent) {
		return null;
	}

	public synchronized boolean arm() {
		if (this.state != STATE_DISARMED) {
			return false;
		}

		if (!setupSensors()) {
			return false;
		}

		this.state = STATE_ARMED;
		indicateArmedStatus();

		return true;
	}

	public synchronized boolean disarm() {
		if (this.state != STATE_ARMED) {
			return false;
		}

		this.state = STATE_DISARMED;
		releaseSensors();

		return true;
	}

	public synchronized boolean call() {
		if (!setupGPS()) {
			disarm();
			return false;
		}

		releaseSensors();

		this.state = STATE_CALLING;
		indicateDistressStatus();

		DistressCallService.indistress = true;

		return true;
	}

	public synchronized boolean hangup() {
		if (this.state != STATE_CALLING) {
			return false;
		}

		this.state = STATE_DISARMED;
		releaseGPS();

		this.dynamo.reset();

		DistressCallService.indistress = false;

		return true;
	}

	private void setupPowerButton() {
		getBaseContext().registerReceiver(this.power_button_listener, new IntentFilter(Intent.ACTION_SCREEN_ON));
		getBaseContext().registerReceiver(this.power_button_listener, new IntentFilter(Intent.ACTION_SCREEN_OFF));
		getBaseContext().registerReceiver(this.cancel_listener, new IntentFilter(DistressCallService.ACTION_CANCEL));
	}

	private void releasePowerButton() {
		getBaseContext().unregisterReceiver(this.power_button_listener);
		getBaseContext().unregisterReceiver(this.cancel_listener);
	}

	private boolean setupSensors() {
		return this.device_wrapper.setupAccelerometer(this);
	}

	private void releaseSensors() {
		this.device_wrapper.releaseAccelerometer(this);
	}

	private boolean setupGPS() {
		return this.device_wrapper.setupGPS(this);
	}

	private void releaseGPS() {
		this.device_wrapper.releaseGPS(this);
	}

	private synchronized void indicateArmedStatus() {
		Vibrator vibrator = (Vibrator)getBaseContext().getSystemService(Context.VIBRATOR_SERVICE);
		vibrator.vibrate(VIBRATE_ARMED, -1);
	}

	private synchronized void indicateDistressStatus() {
		Vibrator vibrator = (Vibrator)getBaseContext().getSystemService(Context.VIBRATOR_SERVICE);
		vibrator.vibrate(VIBRATE_DISTRESS, -1);
	}

	private synchronized void indicateLocationSent() {
		Vibrator vibrator = (Vibrator)getBaseContext().getSystemService(Context.VIBRATOR_SERVICE);
		vibrator.vibrate(VIBRATE_SENT, -1);
	}

	private String sosText(final Location location) {
		POI poi = new POI(location, getResources().getString(R.string.sos_sms_text));

		return SMSText.format(this, poi,
			getResources().getString(R.string.sos_sms_append));
	}

	/**
	 * @return preset distress number if it's enabled, set and valid. return null otherwise
	 */
	private String distressNumberValid() {
		final SharedPreferences shared_pref = PreferenceManager.getDefaultSharedPreferences(this);
		final String distress_number = shared_pref.getString(SettingsActivity.KEY_PREF_DISTRESS_NUMBER, "");
		final boolean use_distress_number = shared_pref.getBoolean(SettingsActivity.KEY_PREF_DISTRESS_USE_NUMBER, false);

		return (use_distress_number
				&& distress_number != null
				&& distress_number.length() > 0) ? distress_number : null;
	}

	private void sendSMS(final String number, final String text) {
		// send sms
		SmsManager sms_manager = SmsManager.getDefault();
		sms_manager.sendTextMessage(number, null, text, null, null);

		// make a record in sent
		ContentValues values = new ContentValues();
		values.put("address", number);
		values.put("body", text);
		getContentResolver().insert(Uri.parse("content://sms/sent"), values);
	}

	private synchronized boolean sendLocation(final Location location) {
		final String distress_number = distressNumberValid();

		final SharedPreferences shared_pref = PreferenceManager.getDefaultSharedPreferences(this);
		final boolean use_preset_number_only = shared_pref.getBoolean(SettingsActivity.KEY_PREF_DISTRESS_USE_NUMBER_ONLY, false);

		if (distress_number == null && use_preset_number_only) {
			return false;
		}

		CalleeLookup lookup = null;

		if (!use_preset_number_only) {
			lookup = new CalleeLookup();
			CallsReader reader = new CallsReader(getBaseContext());

			for (CallsReader.Call call : reader.getCalls()) {
				lookup.pushCall(call.number, call.cached_name, call.date, call.duration);
			}
		}

		final String best_number = this.dynamo.getNumber(distress_number, lookup);

		if (best_number == null) {
			return false;
		}

		sendSMS(best_number, sosText(location));

		// rotate numbers, next distress call will be to another number if possible
		this.dynamo.next();

		if (lookup != null) {
			lookup.dispose();
		}

		return true;
	}

	@Override
	public synchronized void onAccuracyChanged(Sensor sensor, int accuracy) {

	}

	@Override
	public synchronized void onSensorChanged(SensorEvent event) {
		if (event.sensor.getType() != Sensor.TYPE_ACCELEROMETER) {
			return;
		}

		long now = System.currentTimeMillis();

		this.detector.updateAcceleration(event.values, now);

		if (this.detector.panic(now)
			&& state != STATE_CALLING) {

			this.detector.reset();
			call();
		}
	}

	@Override
	public void onLocationChanged(Location location) {
		synchronized (this) {
			if (location != null) {
				if (!location.hasAccuracy()) {
					return;
				}

				if (location.getAccuracy() > Config.MONITOR_ACCURACY_THRESHOLD) {
					return;
				}

				if (sendLocation(location)) {
					indicateLocationSent();
					hangup();
				}
			}
		}
	}

	@Override
	public void onProviderDisabled(String provider) {

	}

	@Override
	public void onProviderEnabled(String provider) {

	}

	@Override
	public void onStatusChanged(String provider, int status, Bundle extras) {

	}

	@Override
	public void onSharedPreferenceChanged(SharedPreferences sharedPreferences, String key) {
		if (key.equals(SettingsActivity.KEY_PREF_DISTRESS_NUMBER)
			|| key.equals(SettingsActivity.KEY_PREF_DISTRESS_USE_NUMBER)) {
			// reset dynamo if preset distress number has changed
			this.dynamo.reset();
		}
	}
}
