"use strict";

/*!
 * jQuery Internationalization library - Message Store
 *
 * Copyright (C) 2012 Santhosh Thottingal
 *
 * jquery.i18n is dual licensed GPLv2 or later and MIT. You don't have to do anything special to
 * choose one license or the other and you don't have to notify anyone which license you are using.
 * You are free to use UniversalLanguageSelector in commercial projects as long as the copyright
 * header is left intact. See files GPL-LICENSE and MIT-LICENSE for details.
 *
 * @licence GNU General Public Licence 2.0 or later
 * @licence MIT License
 */
(function ($) {
  'use strict';

  var MessageStore = function MessageStore() {
    this.messages = {};
    this.sources = {};
  };

  function jsonMessageLoader(url) {
    var deferred = $.Deferred();
    $.getJSON(url).done(deferred.resolve).fail(function (jqxhr, settings, exception) {
      $.i18n.log('Error in loading messages from ' + url + ' Exception: ' + exception); // Ignore 404 exception, because we are handling fallabacks explicitly

      deferred.resolve();
    });
    return deferred.promise();
  }
  /**
   * See https://github.com/wikimedia/jquery.i18n/wiki/Specification#wiki-Message_File_Loading
   */


  MessageStore.prototype = {
    /**
     * General message loading API This can take a URL string for
     * the json formatted messages.
     * <code>load('path/to/all_localizations.json');</code>
     *
     * This can also load a localization file for a locale <code>
     * load( 'path/to/de-messages.json', 'de' );
     * </code>
     * A data object containing message key- message translation mappings
     * can also be passed Eg:
     * <code>
     * load( { 'hello' : 'Hello' }, optionalLocale );
     * </code> If the data argument is
     * null/undefined/false,
     * all cached messages for the i18n instance will get reset.
     *
     * @param {string|Object} source
     * @param {string} locale Language tag
     * @return {jQuery.Promise}
     */
    load: function load(source, locale) {
      var key = null,
          deferred = null,
          deferreds = [],
          messageStore = this;

      if (typeof source === 'string') {
        // This is a URL to the messages file.
        $.i18n.log('Loading messages from: ' + source);
        deferred = jsonMessageLoader(source).done(function (localization) {
          messageStore.set(locale, localization);
        });
        return deferred.promise();
      }

      if (locale) {
        // source is an key-value pair of messages for given locale
        messageStore.set(locale, source);
        return $.Deferred().resolve();
      } else {
        // source is a key-value pair of locales and their source
        for (key in source) {
          if (Object.prototype.hasOwnProperty.call(source, key)) {
            locale = key; // No {locale} given, assume data is a group of languages,
            // call this function again for each language.

            deferreds.push(messageStore.load(source[key], locale));
          }
        }

        return $.when.apply($, deferreds);
      }
    },

    /**
     * Set messages to the given locale.
     * If locale exists, add messages to the locale.
     *
     * @param {string} locale
     * @param {Object} messages
     */
    set: function set(locale, messages) {
      if (!this.messages[locale]) {
        this.messages[locale] = messages;
      } else {
        this.messages[locale] = $.extend(this.messages[locale], messages);
      }
    },

    /**
     *
     * @param {string} locale
     * @param {string} messageKey
     * @return {boolean}
     */
    get: function get(locale, messageKey) {
      return this.messages[locale] && this.messages[locale][messageKey];
    }
  };
  $.extend($.i18n.messageStore, new MessageStore());
})(jQuery);