/*
 * 24-game, a simple arithmetic game.
 * Copyright 2014 Przemyslaw Rzepecki
 * Contact: przemekr@sdfeu.org
 * 
 * This file is part of 24-game.
 * 
 * 24-game is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 * 
 * 24-game is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * 24-game.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <math.h>
#include <algorithm> 
#include <stack>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <time.h>

#include "agg_basics.h"
#include "agg_rendering_buffer.h"
#include "agg_rasterizer_scanline_aa.h"
#include "agg_rasterizer_outline_aa.h"
#include "agg_scanline_u.h"
#include "agg_scanline_p.h"
#include "agg_renderer_scanline.h"
#include "agg_ellipse.h"
#include "agg_pixfmt_gray.h"
#include "agg_pixfmt_rgb.h"
#include "agg_pixfmt_amask_adaptor.h"
#include "agg_span_allocator.h"
#include "agg_span_gradient.h"
#include "agg_span_interpolator_linear.h"
#include "agg_span_image_filter_rgb.h"
#include "agg_span_image_filter_rgba.h"
#include "agg_span_image_filter_gray.h"
#include "agg_pixfmt_rgba.h"
#include "agg_image_accessors.h"
#include "agg_conv_transform.h"
#include "agg_conv_stroke.h"
#include "agg_color_rgba.h"
#include "agg_conv_clip_polyline.h"
#include "agg_gsv_text.h"

#include "agg_bezier_arc.h"
#include "ctrl/agg_bezier_ctrl.h"
#include "agg_pattern_filters_rgba.h"
#include "agg_renderer_outline_aa.h"
#include "agg_renderer_outline_image.h"
#include "agg_renderer_outline_aa.h"
#include "agg_renderer_outline_image.h"

#include "ctrl/agg_slider_ctrl.h"
#include "agg_button_ctrl.h"
#include "platform/agg_platform_support.h"
#include <stdint.h>
#include "util.h"
#include "card.h"
#include "solver.h"

#include <time.h>

#ifdef MOBILE
#define START_H  0
#define START_W  0
#define S 10
#define CTRL_TEXT_THICKNESS 3
#define WINDOW_FLAGS agg::window_fullscreen
#else
#define START_H  380
#define START_W  500
#define S 7
#define CTRL_TEXT_THICKNESS 1
#define WINDOW_FLAGS agg::window_resize
#endif

#if __ANDROID__
#include <android/log.h>
#define DEBUG_PRINT(...) do{ __android_log_print(ANDROID_LOG_INFO, __FILE__, __VA_ARGS__ ); } while (false)
#else
#define DEBUG_PRINT(...) do{fprintf(stderr, __VA_ARGS__ ); } while (false)
#endif

typedef agg::pixfmt_bgr24 pixfmt;
typedef agg::image_accessor_clip<pixfmt> img_source_type;
typedef agg::span_interpolator_linear<> interpolator_type;
typedef agg::span_image_filter_rgb_bilinear_clip<pixfmt, interpolator_type> span_gen_type;
typedef agg::pixfmt_bgr24_pre pixfmt_pre;
typedef agg::renderer_base<pixfmt_pre> renderer_base_pre;
typedef agg::rgba8 color_type;
typedef agg::line_image_pattern<agg::pattern_filter_bilinear_rgba8> pattern_type;
typedef agg::renderer_base<pixfmt> base_ren_type;
typedef agg::renderer_outline_image<base_ren_type, pattern_type> renderer_type;
typedef agg::rasterizer_outline_aa<renderer_type>                rasterizer_type;


template<class Pattern, 
   class Rasterizer, 
   class Renderer, 
   class PatternSource, 
class VertexSource>
void draw_curve(Pattern& patt, 
      Rasterizer& ras, 
      Renderer& ren, 
      PatternSource& src, 
      VertexSource& vs)
{
   patt.create(src);
   ren.scale_x(1.0);
   ren.start_x(1.0);
   ras.add_path(vs);
}

static agg::int8u brightness_to_alpha[256 * 3] = 
{
    255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 
    255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 
    255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 
    255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 
    255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 
    255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 
    255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 
    255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 
    255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 
    255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 254, 254, 254, 254, 254, 254, 
    255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 
    255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 
    255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 
    255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 
    255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 
    255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 
    255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 
    255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 
    255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 
    255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 
    255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 
    255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 
    255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 
    254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 253, 253, 
    253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 252, 
    252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 251, 251, 251, 251, 251, 
    251, 251, 251, 251, 250, 250, 250, 250, 250, 250, 250, 250, 249, 249, 249, 249, 
    249, 249, 249, 248, 248, 248, 248, 248, 248, 248, 247, 247, 247, 247, 247, 246, 
    246, 246, 246, 246, 246, 245, 245, 245, 245, 245, 244, 244, 244, 244, 243, 243, 
    243, 243, 243, 242, 242, 242, 242, 241, 241, 241, 241, 240, 240, 240, 239, 239, 
    239, 239, 238, 238, 238, 238, 237, 237, 237, 236, 236, 236, 235, 235, 235, 234, 
    234, 234, 233, 233, 233, 232, 232, 232, 231, 231, 230, 230, 230, 229, 229, 229, 
    228, 228, 227, 227, 227, 226, 226, 225, 225, 224, 224, 224, 223, 223, 222, 222, 
    221, 221, 220, 220, 219, 219, 219, 218, 218, 217, 217, 216, 216, 215, 214, 214, 
    213, 213, 212, 212, 211, 211, 210, 210, 209, 209, 208, 207, 207, 206, 206, 205, 
    204, 204, 203, 203, 202, 201, 201, 200, 200, 199, 198, 198, 197, 196, 196, 195, 
    194, 194, 193, 192, 192, 191, 190, 190, 189, 188, 188, 187, 186, 186, 185, 184, 
    183, 183, 182, 181, 180, 180, 179, 178, 177, 177, 176, 175, 174, 174, 173, 172, 
    171, 171, 170, 169, 168, 167, 166, 166, 165, 164, 163, 162, 162, 161, 160, 159, 
    158, 157, 156, 156, 155, 154, 153, 152, 151, 150, 149, 148, 148, 147, 146, 145, 
    144, 143, 142, 141, 140, 139, 138, 137, 136, 135, 134, 133, 132, 131, 130, 129, 
    128, 128, 127, 125, 124, 123, 122, 121, 120, 119, 118, 117, 116, 115, 114, 113, 
    112, 111, 110, 109, 108, 107, 106, 105, 104, 102, 101, 100,  99,  98,  97,  96,  
     95,  94,  93,  91,  90,  89,  88,  87,  86,  85,  84,  82,  81,  80,  79,  78, 
     77,  75,  74,  73,  72,  71,  70,  69,  67,  66,  65,  64,  63,  61,  60,  59, 
     58,  57,  56,  54,  53,  52,  51,  50,  48,  47,  46,  45,  44,  42,  41,  40, 
     39,  37,  36,  35,  34,  33,  31,  30,  29,  28,  27,  25,  24,  23,  22,  20, 
     19,  18,  17,  15,  14,  13,  12,  11,   9,   8,   7,   6,   4,   3,   2,   1
};
class pattern_src_brightness_to_alpha_rgba8
{
public:
    pattern_src_brightness_to_alpha_rgba8(agg::rendering_buffer& rb) : 
        m_rb(&rb), m_pf(*m_rb) {}

    unsigned width()  const { return m_pf.width();  }
    unsigned height() const { return m_pf.height(); }
    agg::rgba8 pixel(int x, int y) const
    {
        agg::rgba8 c = m_pf.pixel(x, y);
        c.a = brightness_to_alpha[c.r + c.g + c.b];
        return c;
    }

private:
    agg::rendering_buffer* m_rb;
    pixfmt m_pf;
};

enum flip_y_e { flip_y = true };
const agg::rgba transp(0, 0, 0, 0);
const agg::rgba lblue(0,0,0.9,1);
const agg::rgba lgreen(0,0.9,0,1);
const agg::rgba lgray(60,60,60);
const agg::rgba green(0,120,0);
const agg::rgba yellow(63,63,0);
const agg::rgba red(0.8,0,0,0.5);
const agg::rgba black(0,0,0);

class View
{
public:
   virtual void on_draw() {}
   virtual void on_resize(int sx, int sy) {}
   virtual void on_idle() {}
   virtual void on_mouse_move(int x, int y, unsigned flags) {}
   virtual void on_multi_gesture(float dTheta, float dDist, float x,
         float y, int numFinders) {}
   virtual void on_mouse_button_down(int x, int y, unsigned flags) {}
   virtual void on_mouse_button_up(int x, int y, unsigned flags) {}
   virtual void on_key(int x, int y, unsigned key, unsigned flags) {}
   virtual void on_ctrl_change() {}
   agg::ctrl_container m_ctrls;
   void add_ctrl(agg::ctrl& c) { m_ctrls.add(c);}
};

class App: public agg::platform_support
{
public:
   App(agg::pix_format_e format, bool flip_y) :
      agg::platform_support(format, flip_y),
      p1level(0), p2level(0), gameSec(0),
      hPoints(0), cPoints(0), scores(0),
      v1(0), v2(0), v3(0), v4(0), timerStarted(false)
   {
      bestSec[0] = 0;
      bestSec[1] = 0;
      bestSec[2] = 0;
   }

   virtual void changeView(const char* name) {};

   void gameStarted()
   {
      if (!timerStarted)
         start_timer();
      timerStarted = true;
   }

   void gameFinished()
   {
      gameSec = elapsed_time()/1000;
      if (!bestSec[0] || gameSec < bestSec[0])
      {
         bestSec[2] = bestSec[1];
         bestSec[1] = bestSec[0];
         bestSec[0] = gameSec;
      }
      else if (!bestSec[1] || gameSec < bestSec[1])
      {
         bestSec[2] = bestSec[1];
         bestSec[1] = gameSec;
      }
      else if (!bestSec[2] || gameSec < bestSec[2])
      {
         bestSec[2] = gameSec;
      }
      changeView("result");
   }

   View* view;
   int p1level;
   int p2level;
   int hPoints;
   int cPoints;
   int w, h;
   int scores;
   int gameSec;
   int bestSec[3];
   int v1, v2, v3, v4;
   bool timerStarted;


   virtual void on_ctrl_change()
   {
      view->on_ctrl_change();
   }

   virtual void on_resize(int nx, int ny)
   {
      w = nx; h = ny;
      view->on_resize(nx, ny);
   }

   virtual void on_idle()
   {
      view->on_idle();
   }

   virtual void on_mouse_button_up(int x, int y, unsigned flags)
   {
      view->on_mouse_button_up(x, y, flags);
   }

   virtual void on_mouse_button_down(int x, int y, unsigned flags)
   {
      view->on_mouse_button_down(x, y, flags);
   }

   virtual void on_mouse_move(int x, int y, unsigned flags)
   {
      view->on_mouse_move(x, y, flags);
   }

   virtual void on_multi_gesture(float x, float y,
         float dTheta, float dDist, int numFingers)
   {
      view->on_multi_gesture(x, y, dTheta, dDist, numFingers);
   }

   virtual void on_draw()
   {
      view->on_draw();
   }

};

class GameView : public View
{
public:
   GameView(App& application): app(application),
      menu(80,  20, 180, 40,   "MENU", !flip_y),
      next(220, 20, 320, 40,   "next", !flip_y),
      bracket(340, 20, 370, 40,   "()", !flip_y),

      card1(40,  80,  100, 200,  "PL", !flip_y),
      oper1(110, 120, 140, 160,  " ", !flip_y),
      card2(150, 80,  210, 200,  "AY", !flip_y),
      oper2(220, 120, 250, 160,  " ", !flip_y),
      card3(260, 80,  320, 200,  "2 ", !flip_y),
      oper3(330, 120, 360, 160,  " ", !flip_y),
      card4(370, 80,  430, 200,  "4 ", !flip_y),
      animation(0), bracketPos(0), selCard(0)
   {
      menu.background_color(red);
      next.background_color(red);
      bracket.background_color(red);
      card1.background_color(lblue);
      card2.background_color(lblue);
      card3.background_color(lblue);
      card4.background_color(lblue);
      card1.text_thickness(4);
      card2.text_thickness(4);
      card3.text_thickness(4);
      card4.text_thickness(4);
      oper1.background_color(lgreen);
      oper2.background_color(lgreen);
      oper3.background_color(lgreen);
      add_ctrl(next);
      add_ctrl(menu);
      add_ctrl(card1);
      add_ctrl(card2);
      add_ctrl(card3);
      add_ctrl(card4);
      add_ctrl(oper1);
      add_ctrl(oper2);
      add_ctrl(oper3);
      add_ctrl(bracket);

      m_curve1.curve(170, 424, 13, 87, 488, 423, 26, 333);
      add_ctrl(m_curve1);
      m_curve1.no_transform();
      
      app.wait_mode(true);
   }

   const char * nextOper(const char * curr)
   {
      if (strcmp(curr, "+") == 0)
         return "-";
      if (strcmp(curr, "-") == 0)
         return "*";
      if (strcmp(curr, "*") == 0)
         return "/";
      return "+";
   }

   double calc(double x, const char * curr, double y)
   {
      if (strcmp(curr, "") == 0)
         return 0;
      if (strcmp(curr, "+") == 0)
         return x+y;
      if (strcmp(curr, "-") == 0)
         return x-y;
      if (strcmp(curr, "*") == 0)
         return x*y;
      if (strcmp(curr, "/") == 0)
         return x/y;

      return 0;
   }

   virtual void on_ctrl_change()
   {
      if (menu.status())
      {
         menu.status(false);
         app.changeView("menu");
      }

      if (bracket.status())
      {
         bracket.status(false);
         ++bracketPos %= 3;
      }

      if (next.status())
      {
         app.gameStarted();
         char buf[20];
         next.status(false);
         bool possible;

         if (getResult(24, possible, NULL))
            app.scores += 3;
         else if (possible)
            app.scores--;

         if (app.scores >= 10)
         {
            app.gameFinished();
         }
         
         oper1.label("+");
         oper2.label("+");
         oper3.label("+");

         app.v1 = rand()%12+1;
         app.v2 = rand()%12+1;
         app.v3 = rand()%12+1;
         app.v4 = rand()%12+1;

         sprintf(buf, "%d", app.v1);
         card1.label(buf);
         sprintf(buf, "%d", app.v2);
         card2.label(buf);
         sprintf(buf, "%d", app.v3);
         card3.label(buf);
         sprintf(buf, "%d", app.v4);
         card4.label(buf);

      }

      if (oper1.status())
      {
         oper1.status(false);
         oper1.label(nextOper(oper1.label()));
      }
      if (oper2.status())
      {
         oper2.status(false);
         oper2.label(nextOper(oper2.label()));
      }
      if (oper3.status())
      {
         oper3.status(false);
         oper3.label(nextOper(oper3.label()));
      }

      app.force_redraw();
   }

   virtual void on_resize(int, int)
   {
      app.force_redraw();
      double w = app.rbuf_window().width();
      double h = app.rbuf_window().height();
      
      size = int(std::min(w*0.95, h*0.9));
      hshift = h - size;
      hshift -= hshift/4;
      wshift = w - size;
      wshift /= 2;
   }

   virtual void on_idle()
   {
   }

   void switchCards(int f, int s)
   {
      if (f < 1 || f > 4 || s < 1 || s > 4)
         return;

      agg::button_ctrl<agg::rgba8>& fc =
         f == 1? card1:
         f == 2? card2:
         f == 3? card3:
         card4;
      agg::button_ctrl<agg::rgba8>& sc =
         s == 1? card1:
         s == 2? card2:
         s == 3? card3:
         card4;

      int& fv = 
         f == 1? app.v1:
         f == 2? app.v2:
         f == 3? app.v3:
         app.v4;
      int& sv = 
         s == 1? app.v1:
         s == 2? app.v2:
         s == 3? app.v3:
         app.v4;

      int t = sv;
      sv = fv;
      fv = t;

      char tmp[50];
      strcpy(tmp, fc.label());
      fc.label(sc.label());
      sc.label(tmp);
   }

   virtual void on_mouse_button_up(int x, int y, unsigned flags)
   {
      if (m_ctrls.on_mouse_button_up(x, y))
      {
         app.on_ctrl_change();
         app.force_redraw();
      }
      if (card1.in_rect(x, y))
         switchCards(selCard, 1);
      if (card2.in_rect(x, y))
         switchCards(selCard, 2);
      if (card3.in_rect(x, y))
         switchCards(selCard, 3);
      if (card4.in_rect(x, y))
         switchCards(selCard, 4);
      selCard = 0;
   }

   virtual void on_mouse_button_down(int x, int y, unsigned flags)
   {
      if (m_ctrls.on_mouse_button_down(x, y))
      {
         app.on_ctrl_change();
         app.force_redraw();
      }
      selCard = 0;
      x0 = x;
      y0 = y;
      lineNr = rand()%9;
      m1x    = x+rand()%400-200;
      m1y    = y+rand()%400-200;
      m2x    = x+rand()%400-200;
      m2y    = y+rand()%400-200;
      

      if (card1.in_rect(x, y))
         selCard = 1;
      if (card2.in_rect(x, y))
         selCard = 2;
      if (card3.in_rect(x, y))
         selCard = 3;
      if (card4.in_rect(x, y))
         selCard = 4;
      m_curve1.curve(x0, y0, m1x, m1y, m2x, m2y, x, y);
   }

   virtual void on_mouse_move(int x, int y, unsigned flags)
   {
      if (m_ctrls.on_mouse_move(x, y, (flags & agg::mouse_left) != 0))
      {
         app.on_ctrl_change();
         app.force_redraw();
         return;
      }
      if (selCard)
      {
         m_curve1.curve(x0, y0, m1x, m1y, m2x, m2y, x, y);
         app.force_redraw();
      }
   }

   virtual void on_multi_gesture(float x, float y,
         float dTheta, float dDist, int numFingers)
   {
   }

   virtual void on_draw()
   {
      agg::pixfmt_bgr24 pf(app.rbuf_window());;
      agg::renderer_base<agg::pixfmt_bgr24> rbase(pf);
      agg::rasterizer_scanline_aa<> ras;
      agg::scanline_u8 sl;
      ras.reset();
      rbase.clear(lgray);

      double scale = app.rbuf_window().width()/500.0;
      static agg::trans_affine shape_mtx; shape_mtx.reset();
      shape_mtx *= agg::trans_affine_scaling(scale);
      shape_mtx *= agg::trans_affine_translation(0, 0);

      next.transform(shape_mtx);
      menu.transform(shape_mtx);
      oper1.transform(shape_mtx);
      oper2.transform(shape_mtx);
      oper3.transform(shape_mtx);
      card1.transform(shape_mtx);
      card2.transform(shape_mtx);
      card3.transform(shape_mtx);
      card4.transform(shape_mtx);
      bracket.transform(shape_mtx);

      agg::render_ctrl(ras, sl, rbase, next);
      agg::render_ctrl(ras, sl, rbase, menu);
      agg::render_ctrl(ras, sl, rbase, oper1);
      agg::render_ctrl(ras, sl, rbase, oper2);
      agg::render_ctrl(ras, sl, rbase, oper3);
      agg::render_ctrl(ras, sl, rbase, card1);
      agg::render_ctrl(ras, sl, rbase, card2);
      agg::render_ctrl(ras, sl, rbase, card3);
      agg::render_ctrl(ras, sl, rbase, card4);
      agg::render_ctrl(ras, sl, rbase, bracket);

      char* text;
      char buf[20];
      bool possible;
      getResult(24, possible, &text);
      sprintf(buf, "%d", app.scores);

      draw_text(app.rbuf_window(), 40, 220*scale, 15*scale, text);
      draw_text(app.rbuf_window(), 410*scale, 20*scale, 30*scale, buf);


      if (selCard)
      {
         pattern_src_brightness_to_alpha_rgba8 p1(app.rbuf_img(lineNr));
         agg::pattern_filter_bilinear_rgba8 fltr;           // Filtering functor


         //-- Create uninitialized and set the source
         pattern_type patt(fltr);        
         renderer_type ren_img(rbase, patt);
         rasterizer_type ras_img(ren_img);

         draw_curve(patt, ras_img, ren_img, p1, m_curve1.curve());
      }

   }

private:
   bool getResult(int expected, bool& possible, char **output)
   {
      static char buf[128];
      const char* fmt;
      int result;

      if (bracketPos == 0)
      {
         result = (int)calc(calc(calc(app.v1, oper1.label(), app.v2), oper2.label(), app.v3), oper3.label(), app.v4); 
         fmt = "((%d %s %d) %s %d) %s %d = %d";
      }
      if (bracketPos == 1)
      {
         result = (int)calc(calc(app.v1, oper1.label(), app.v2), oper2.label(), calc(app.v3, oper3.label(), app.v4));
         fmt = "(%d %s %d) %s (%d %s %d) = %d";
      }
      if (bracketPos == 2)
      {
         result = (int)calc(calc(app.v1, oper1.label(), calc(app.v2, oper2.label(), app.v3)), oper3.label(), app.v4);
         fmt = "%d %s (%d %s %d) %s %d = %d";
      }
      char solution[128];
      possible = solver::isSolution(expected, app.v1, app.v2, app.v3, app.v4, solution);
      sprintf(buf, fmt, app.v1, oper1.label(), app.v2, oper2.label(), app.v3, oper3.label(), app.v4, result); 
      if (output)
         *output = buf;
      return result == expected || result == -expected;
   }

    App& app;
    int size, wshift, hshift;
    agg::button_ctrl<agg::rgba8> next;
    agg::button_ctrl<agg::rgba8> menu;

    agg::button_ctrl<agg::rgba8> oper1;
    agg::button_ctrl<agg::rgba8> oper2;
    agg::button_ctrl<agg::rgba8> oper3;

    agg::button_ctrl<agg::rgba8> card1;
    agg::button_ctrl<agg::rgba8> card2;
    agg::button_ctrl<agg::rgba8> card3;
    agg::button_ctrl<agg::rgba8> card4;
    agg::button_ctrl<agg::rgba8> bracket;

    agg::bezier_ctrl<agg::rgba8> m_curve1;

    int animation;
    int selCard;
    int x0, y0, lineNr, m1x, m1y, m2x, m2y;
    int bracketPos;
};

class ResultView : public View
{
public:
   ResultView(App& application): app(application),
   newGame (20,  20, 120, 40,   "New Game",  !flip_y)
   {
      add_ctrl(newGame);
   }

   virtual void on_draw()
   {
      pixfmt        pf(app.rbuf_window());
      pixfmt_pre    pixf_pre(app.rbuf_window());
      agg::renderer_base<pixfmt> rbase(pf);
      renderer_base_pre rb_pre(pixf_pre);
      agg::rasterizer_scanline_aa<> ras;
      agg::scanline_u8 sl;
      agg::span_allocator<color_type> sa;
      ras.reset();
      rbase.clear(lgray);

      double scale = app.rbuf_window().width()/380.0;
      static agg::trans_affine shape_mtx; shape_mtx.reset();
      shape_mtx *= agg::trans_affine_scaling(scale);
      shape_mtx *= agg::trans_affine_translation(0, 0);
      newGame.transform(shape_mtx);

      int imgIdx = 9 + rand()%2;
      pixfmt img_pixf(app.rbuf_img(imgIdx));
      rbase.copy_from(app.rbuf_img(imgIdx), 0, app.rbuf_window().width()-app.rbuf_img(imgIdx).width(), 0);

      agg::render_ctrl(ras, sl, rbase, newGame);

      char string[200];
      sprintf(string, "Congratulations!\n\nYou finished the 24 game in %ds\n\n"
            "Top Times:\n\n%ds\n\n%ds\n\n%ds", app.gameSec,
            app.bestSec[0], app.bestSec[1], app.bestSec[2]);

      draw_text(app.rbuf_window(), 40, 200*scale, 10*scale, string);
   }

   virtual void on_ctrl_change()
   {
      if (newGame.status())
      {
         newGame.status(false);
         app.changeView("game");
         app.start_timer();
         app.scores = 0;
      }
   }

   virtual void on_mouse_button_up(int x, int y, unsigned flags)
   {
      if (m_ctrls.on_mouse_button_up(x, y))
      {
         app.on_ctrl_change();
         app.force_redraw();
      }
   }

   virtual void on_mouse_button_down(int x, int y, unsigned flags)
   {
      if (m_ctrls.on_mouse_button_down(x, y))
      {
         app.on_ctrl_change();
         app.force_redraw();
         return;
      }
   }

private:
    App& app;
    agg::button_ctrl<agg::rgba8> newGame;
};

class MenuView : public View
{
public:
   MenuView(App& application): app(application),
   exitApp (100, 20,   300, 50,    "Quit App",  !flip_y),
   exitMenu(100, 60,   300, 90,    "Return  ",  !flip_y),
   newGame (100, 100,  300, 130,   "New Game",  !flip_y)
   {
      exitMenu.background_color(red);
      exitApp.background_color(red);
      newGame.background_color(red);
      add_ctrl(exitMenu);
      add_ctrl(exitApp);
      add_ctrl(newGame);
   }

   virtual void on_draw()
   {
      double w = app.rbuf_window().width();
      double h = app.rbuf_window().height();

      pixfmt        pf(app.rbuf_window());
      pixfmt_pre    pixf_pre(app.rbuf_window());
      agg::renderer_base<pixfmt> rbase(pf);
      renderer_base_pre rb_pre(pixf_pre);
      agg::rasterizer_scanline_aa<> ras;
      agg::scanline_u8 sl;
      agg::span_allocator<color_type> sa;
      ras.reset();
      rbase.clear(lgray);

      double scale = app.rbuf_window().width()/380.0;
      static agg::trans_affine shape_mtx; shape_mtx.reset();
      shape_mtx *= agg::trans_affine_scaling(scale);
      shape_mtx *= agg::trans_affine_translation(0, 0);
      exitMenu.transform(shape_mtx);
      exitApp.transform(shape_mtx);
      newGame.transform(shape_mtx);

      agg::render_ctrl(ras, sl, rbase, exitMenu);
      agg::render_ctrl(ras, sl, rbase, exitApp);
      agg::render_ctrl(ras, sl, rbase, newGame);

      // print tip
      char solution[128];
      char string[256];
      bool possible = solver::isSolution(24, app.v1, app.v2, app.v3, app.v4, solution);
      sprintf(string, "%s:\n\n\t%s",
            rand()%2? "pssst, try this": "solution for a smile",
            possible? solution: "no solution");
      draw_text(app.rbuf_window(), 40, 200*scale, 15*scale, string);
   }

   virtual void on_ctrl_change()
   {
      if (exitMenu.status())
      {
         exitMenu.status(false);
         app.changeView("game");
      }
      if (newGame.status())
      {
         newGame.status(false);
         app.changeView("game");
         app.start_timer();
         app.scores = 0;
      }
      if (exitApp.status())
      {
         throw 0;
      }
   }

   virtual void on_mouse_button_up(int x, int y, unsigned flags)
   {
      if (m_ctrls.on_mouse_button_up(x, y))
      {
         app.on_ctrl_change();
         app.force_redraw();
      }
   }

   virtual void on_mouse_button_down(int x, int y, unsigned flags)
   {
      if (m_ctrls.on_mouse_button_down(x, y))
      {
         app.on_ctrl_change();
         app.force_redraw();
         return;
      }
   }

   virtual void on_mouse_move(int x, int y, unsigned flags)
   {
      if (m_ctrls.on_mouse_move(x, y, (flags & agg::mouse_left) != 0))
      {
         app.on_ctrl_change();
         app.force_redraw();
         return;
      }
   }

private:
    App& app;
    agg::button_ctrl<agg::rgba8> exitMenu;
    agg::button_ctrl<agg::rgba8> exitApp;
    agg::button_ctrl<agg::rgba8> newGame;
};


class the_application: public App
{
public:
   the_application(agg::pix_format_e format, bool flip_y) :
      App(format, flip_y)
   {
      game = new GameView(*this);
      menu = new MenuView(*this);
      resultView = new ResultView(*this);
      view = game;
      srand(time(NULL)); 
   }
   virtual void changeView(const char* name) 
   {
      if (strcmp(name, "menu") == 0)
         view = menu;
      if (strcmp(name, "game") == 0)
         view = game;
      if (strcmp(name, "result") == 0)
         view = resultView;
   };
private:
   GameView* game;
   MenuView* menu;
   ResultView* resultView;
};



int agg_main(int argc, char* argv[])
{
    the_application app(agg::pix_format_bgr24, flip_y);
    app.caption("24-Game");

    if(!app.load_img(0, "1") ||
       !app.load_img(1, "2") ||
       !app.load_img(2, "3") ||
       !app.load_img(3, "4") ||
       !app.load_img(4, "5") ||
       !app.load_img(5, "6") ||
       !app.load_img(6, "7") ||
       !app.load_img(7, "8") ||
       !app.load_img(8, "9") ||
       !app.load_img(9, "you_are_a_star") ||
       !app.load_img(10, "you_are_the_best"))
    {
        char buf[256];
        sprintf(buf, "There must be files 1%s...9%s\n",
                     app.img_ext(), app.img_ext());
        app.message(buf);
        return 1;
    }

    if (app.init(START_W, START_H, WINDOW_FLAGS))
    {
       try {
          return app.run();
       } catch (...) {
          exit(0);
       } 
    }
    return 1;
}
