/*
 * Game of Life AGG, the game of life simulation.
 * Copyright 2013 Przemyslaw Rzepecki
 * Contact: przemekr@sdfeu.org
 * 
 * This file is part of GameOfLifeAgg.
 * 
 * GameOfLifeAgg is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 * 
 * GameOfLifeAgg is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * GameOfLifeAgg.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "agg_basics.h"
#include "agg_rendering_buffer.h"
#include "agg_rasterizer_scanline_aa.h"
#include "agg_scanline_u.h"
#include "agg_scanline_p.h"
#include "agg_renderer_scanline.h"
#include "agg_ellipse.h"
#include "agg_pixfmt_gray.h"
#include "agg_pixfmt_rgb.h"
#include "agg_pixfmt_amask_adaptor.h"
#include "ctrl/agg_slider_ctrl.h"
#include "ctrl/agg_cbox_ctrl.h"
#include "platform/agg_platform_support.h"
#include <unistd.h>
#include <stdint.h>
#include "life.h"
#include "stdio.h"

#ifdef MOBILE
#define START_H  0
#define START_W  0
#define CTRL_SCALE 2.0
#define INIT_CSIZE 20
#define SLIDER_SIZE 35
#define SLIDER_LEN 200
#define CTRL_TEXT_THICKNESS 3
#define WINDOW_FLAGS agg::window_fullscreen
#else
#define START_H  320
#define START_W  320
#define CTRL_SCALE 1.0
#define INIT_CSIZE 10
#define SLIDER_SIZE 17
#define SLIDER_LEN 100
#define CTRL_TEXT_THICKNESS 1
#define WINDOW_FLAGS agg::window_resize
#endif


enum flip_y_e { flip_y = true };
const agg::rgba transp(0, 0, 0, 0);
const agg::rgba lblue(0,0,128,128);
const agg::rgba lgray(40,40,40);
const agg::rgba green(0,120,0);
const agg::rgba yellow(63,63,0);
const agg::rgba red(120,0,0);
const agg::rgba black(0,0,0);
const agg::rgba palete[] = {green, yellow, red};
static agg::trans_affine shape_mtx;
extern const char* library[];
extern int library_size;

class the_application : public agg::platform_support
{
public:
   the_application(agg::pix_format_e format, bool flip_y) :
      speed_ctr(120, 10, 120+SLIDER_LEN, SLIDER_SIZE, !flip_y),
      size_ctr(140+SLIDER_LEN, 10, 140+2*SLIDER_LEN, SLIDER_SIZE, !flip_y),
      run_ctr(10, 10, "Go", !flip_y),
      next_ctr(10, 25, "Next", !flip_y),
      agg::platform_support(format, flip_y),
      uni(w, h)
   {
      step = 0;
      nextInitial = 0;
      mx = my = lx = ly = 0;
      mouse_moved = false;
      w = int(rbuf_window().width());
      h = int(rbuf_window().height());
      speed_ctr.range(0, 10);
      speed_ctr.num_steps(10);
      speed_ctr.value(2);
      speed_ctr.label("Speed=%1.0f");
      speed_ctr.no_transform();
      speed_ctr.background_color(transp);
      speed_ctr.text_thickness(CTRL_TEXT_THICKNESS);
      size_ctr.range(1, 30);
      size_ctr.num_steps(29);
      size_ctr.value(INIT_CSIZE);
      size_ctr.label("Size=%1.0f");
      size_ctr.no_transform();
      size_ctr.background_color(transp);
      size_ctr.text_thickness(CTRL_TEXT_THICKNESS);
      shape_mtx *= agg::trans_affine_scaling(CTRL_SCALE);
      shape_mtx *= agg::trans_affine_translation(0, 0);
      run_ctr.transform(shape_mtx);
      next_ctr.transform(shape_mtx);
      csize = int(size_ctr.value());
      uni.setState(library[nextInitial++%library_size],
            w/3/csize, h/3/csize);
   }

   virtual void on_ctrl_change()
   {
      force_redraw();
      wait_mode(!run_ctr.status());

      if (next_ctr.status())
      {
         next_ctr.status(0);
         uni.setState(library[nextInitial++%library_size],
               w/3/csize, h/3/csize);
      }

      
      if (csize != int(size_ctr.value()))
      {
         csize = int(size_ctr.value());
         uni.resize(w/csize, h/csize);
      }
   }

   virtual void on_resize(int, int)
   {
      force_redraw();
      w = int(rbuf_window().width());
      h = int(rbuf_window().height());
      uni.resize(w/csize, h/csize);
   }


   virtual void on_idle()
   {
      static int cnt;
      int c = 10 - int(speed_ctr.value());
      c = 5<<c;
      usleep(1000);
      if (cnt++ < c)
         return;

      cnt = 0;
      step++;
      uni.tick();
      force_redraw();
   }
   int pix2l(int pixel, int shift, int length)
   {
      length /= csize;
      shift  /= csize;
      pixel  /= csize;
      int r = length-1  + shift - pixel;
      return r % length;
   }
   int l2pix(int l, int shift, int length)
   {
      length /= csize;
      shift  /= csize;
      int r = length-1;
      r += shift;
      r -= l;
      r %= length; if (r < 0) r += length;
      return r*csize;
   }

   virtual void on_mouse_button_up(int x, int y, unsigned flags)
   {
      lx = ly = 0;
      speed_ctr.on_mouse_button_up(x, y);
      size_ctr.on_mouse_button_up(x, y);
      run_ctr.on_mouse_button_up(x, y);
      next_ctr.on_mouse_button_up(x, y);

      if (mouse_moved)
      {
         mouse_moved = false;
         return;
      }

      int i = pix2l(x, mx, w);
      int j = pix2l(y, my, h);
      if (uni.count(Loc(i, j)))
         uni.erase(Loc(i, j));
      else
         uni[Loc(i, j)] = uni.newLife(Loc(i, j));
      force_redraw();
   }

   virtual void on_mouse_button_down(int x, int y, unsigned flags)
   {
      if (speed_ctr.on_mouse_button_down(x, y)
            || size_ctr.on_mouse_button_down(x, y)
            || run_ctr.on_mouse_button_down(x, y)
            || next_ctr.on_mouse_button_down(x, y))
      {
         mouse_moved = true;
         on_ctrl_change();
      }
   }

   virtual void on_mouse_move(int x, int y, unsigned flags)
   {
      if (flags & agg::mouse_left)
      {
         mouse_moved = true;
         if (speed_ctr.on_mouse_move(x, y, true)
               || size_ctr.on_mouse_move(x, y, true))
         {
            on_ctrl_change();
            return;
         }

         if (lx)
            mx += (x-lx);
         if (ly)
            my += (y-ly);
         lx = x; ly = y;
         force_redraw();
      }
   }


   virtual void on_draw()
   {
      agg::pixfmt_bgr24 pf(rbuf_window());;
      agg::renderer_base<agg::pixfmt_bgr24> rbase(pf);
      agg::rasterizer_scanline_aa<> ras;
      agg::scanline_u8 sl;
      ras.reset();
      rbase.clear(lgray);

      /* draw grid */
      if (csize > 4)
      {
         int i;
         for (i = 0; i< h/csize; i++)
            rbase.blend_hline(0, csize*i, w, lblue, 64);
         rbase.blend_bar(0, csize*(i), w, h, black, 255);

         for (i = 0; i< w/csize; i++)
            rbase.blend_vline(csize*i, 0, h, lblue, 64);
         rbase.blend_bar(csize*(i), 0, w, h, black, 255);
      }


      /* draw cells */
      for (Universe::iterator i = uni.begin(); i != uni.end(); i++)
         drawCell(rbase, i->first.x, i->first.y, i->second);

      char buf[50];
      sprintf(buf, "%d", step);
      draw_text(10, h-50, buf);
      agg::render_ctrl(ras, sl, rbase, speed_ctr);
      agg::render_ctrl(ras, sl, rbase, size_ctr);
      agg::render_ctrl(ras, sl, rbase, run_ctr);
      agg::render_ctrl(ras, sl, rbase, next_ctr);
   }
private:
    Universe uni;
    int step;
    int nextInitial;
    int csize;
    int w, h;
    int mx, my, lx, ly; 
    bool mouse_moved;
    agg::slider_ctrl<agg::rgba8> speed_ctr;
    agg::slider_ctrl<agg::rgba8> size_ctr;
    agg::cbox_ctrl<agg::rgba8>   run_ctr;
    agg::cbox_ctrl<agg::rgba8>   next_ctr;

    void drawCell(agg::renderer_base<agg::pixfmt_bgr24>& rbase,
          int x, int y, Life c)
    {
       rbase.blend_bar(
             l2pix(x, mx, w),
             l2pix(y, my, h),
             l2pix(x, mx, w)+csize,
             l2pix(y, my, h)+csize,
             palete[c.gender%3], 170-(10*c.age%150));
    }

    void draw_text(double x, double y, const char* str)
    {
       typedef agg::pixfmt_bgr24 pixfmt_type;
       agg::rasterizer_scanline_aa<> m_ras;
       agg::scanline_p8              m_sl;
       pixfmt_type pf(rbuf_window());
       agg::renderer_base<pixfmt_type> rb(pf);
       agg::renderer_scanline_aa_solid<agg::renderer_base<pixfmt_type> > ren(rb);

       agg::gsv_text txt;
       agg::conv_stroke<agg::gsv_text> txt_stroke(txt);
       txt_stroke.width(5.0);
       txt_stroke.line_cap(agg::round_cap);
       txt.size(30);
       txt.start_point(x, y);
       txt.text(str);
       m_ras.add_path(txt_stroke);

       ren.color(agg::rgba(0, 0, 0, 196));
       agg::render_scanlines(m_ras, m_sl, ren);
    }
};



int agg_main(int argc, char* argv[])
{
    the_application app(agg::pix_format_bgr24, flip_y);
    app.caption("The Game of Life");
    if (app.init(START_W, START_H, WINDOW_FLAGS))
    {
        return app.run();
    }
    return 1;
}
