package com.ubuntuone.android.files;

import android.app.AlarmManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;

import com.ubuntuone.android.files.service.MediaCatcher;
import com.ubuntuone.android.files.service.MetaService;
import com.ubuntuone.android.files.service.UpDownService;
import com.ubuntuone.android.files.util.DateUtilities;
import com.ubuntuone.android.files.util.Log;
import com.ubuntuone.android.files.util.StorageInfo;

public final class Alarms {
	private static final String TAG = Alarms.class.getSimpleName();
	
	private static UbuntuOneFiles sAppInstance = UbuntuOneFiles.getInstance();
	
	private static long sLastRetryFailedInterval;
	
	static {
		sAppInstance = UbuntuOneFiles.getInstance();
		sLastRetryFailedInterval = 0;
	}

	/**
	 * Gets saved media upload alarm frequency and registers with 3 min delay.
	 * {@link Alarms#maybeRegisterMediaUploadAlarm(long, long)}.
	 * 
	 * @return true if alarm has been registered<br/>
	 *         false if it either settings don't allow it or it was already
	 *         registered
	 */
	public static boolean maybeRegisterMediaUploadAlarm() {
		if (Preferences.getBoolean(Preferences.UPLOAD_PHOTOS_KEY, false)) {
			final long delay = DateUtilities.MILLIS_IN_MINUTE * 3;
			final long wakeInterval = Preferences.getLongFromString(
					Preferences.UPLOAD_FREQUENCY_KEY, AlarmManager.INTERVAL_HOUR);  
			return maybeRegisterMediaUploadAlarm(delay, wakeInterval);
		} else {
			Log.d(TAG, "not registering media upload alarm because of settings");
			return false;
		}
	}

	/**
	 * If upload alarm not yet registered, registers media upload alarm with
	 * given wakeInterval frequency. Assumes UPLOAD_PHOTOS_KEY preference is
	 * true. Also, depending on the setting starts/stops {@link MediaCatcher}.
	 * 
	 * @param wakeInterval
	 *            the wake interval to use
	 * @return true if alarm has been registered<br/>
	 *         false if it was already registered
	 */
	private static boolean maybeRegisterMediaUploadAlarm(
			long delay, long wakeInterval) {
		final Intent intent = new Intent(MetaService.ACTION_UPLOAD_MEDIA);

		// Also start up the Media Catcher, if storage setting allows it.
		boolean runMediaCatcher = false;
		try {
			runMediaCatcher = StorageInfo.shouldImmediatelyUploadPhotos();
		} catch (StorageInfo.StorageNotAvailable e) {
			Log.e(TAG, "Getting immediate-upload pref.", e);
		}
		if (runMediaCatcher) {
			// Start service here, or when Preferences are changed to turn it on.
			if (MediaCatcher.startFrom(sAppInstance)) {
				Log.e(TAG, "Could not start service "
						+ MediaCatcher.class.getSimpleName());
			} else {
				Log.i(TAG, "Started service "
						+ MediaCatcher.class.getSimpleName());
			}
		}

		PendingIntent pendingIntent = PendingIntent.getService(
				sAppInstance, 0, intent, PendingIntent.FLAG_NO_CREATE);
		if (pendingIntent == null) {
			// Not registered, so let's register it now.
			Log.d(TAG, "media upload alarm not registered yet");
			pendingIntent = PendingIntent.getService(
					sAppInstance, 0, intent, PendingIntent.FLAG_CANCEL_CURRENT);
			registerUploadAlarm(pendingIntent, delay, wakeInterval);
			return true;
		} else {
			Log.d(TAG, "media upload alarm already registered");
			return false;
		}
	}
	
	public static void registerUploadAlarm(final PendingIntent operation,
			final long delay, final long wakeInterval) {
		// Because this method is public, we again check if the setting is ON.
		if (!Preferences.getBoolean(Preferences.UPLOAD_PHOTOS_KEY, false)) {
			Log.d(TAG, "not registering media upload alarm because of settings");
			return;
		}
		// Cancel current.
		final AlarmManager am =
				(AlarmManager) sAppInstance.getSystemService(
						UbuntuOneFiles.ALARM_SERVICE);
		am.cancel(operation);
		
		// Can sync every 15 minutes, but must at least once a wakeInterval.
		Log.i(TAG, "(re)registering auto-upload alarm, wake: " + wakeInterval);
		
		int sleep = AlarmManager.RTC;
		int wakeup = AlarmManager.RTC_WAKEUP;
		
		long withSleepInterval = AlarmManager.INTERVAL_FIFTEEN_MINUTES;
		long withWakeupInterval = wakeInterval;
		
		long triggerAtTime = System.currentTimeMillis() + delay;
		
		am.setInexactRepeating(sleep, triggerAtTime,
				withSleepInterval, operation);
		am.setInexactRepeating(wakeup, triggerAtTime,
				withWakeupInterval, operation);
	}
	
	public static void resetRetryFailedInterval() {
		sLastRetryFailedInterval = 0;
	}
	
	public static PendingIntent getMediaUploadPendingIntent(
			final Context context, final int flags) {
		final Intent intent = new Intent(MetaService.ACTION_UPLOAD_MEDIA);
		return PendingIntent.getService(sAppInstance, 0, intent, flags);
	}
	
	public static void unregisterPictureUploadAlarm() {
		unregisterUploadAlarm(getMediaUploadPendingIntent(sAppInstance,
				PendingIntent.FLAG_NO_CREATE));
	}
	
	public static void unregisterUploadAlarm(final PendingIntent operation) {
		if (operation != null) {
			final AlarmManager am =
				(AlarmManager) sAppInstance.getSystemService(
						UbuntuOneFiles.ALARM_SERVICE);
			am.cancel(operation);
			Log.i(TAG, "unregistered upload alarm");
		} else {
			Log.d(TAG, "no alarm to unregister");
		}
	}
	
	/**
	 * Only {@link UpDownService} should call this method.
	 * 
	 * @return
	 */
	public static boolean maybeRegisterRetryFailedAlarm() {
		final boolean retryFailed = Preferences.getBoolean(
				Preferences.AUTO_RETRY_FAILED, false);
		if (retryFailed) {
			final long delay = DateUtilities.MILLIS_IN_MINUTE * 2;
			final long wakeInterval = getNextRetryFailedInterval();
			sLastRetryFailedInterval = wakeInterval; // Save it for laggy exp backoff.
			registerRetryFailedAlarm(delay, wakeInterval);
			return true;
		} else {
			Log.d(TAG, "not registering retry failed alarm because of settings");
			return false;
		}
	}
	
	private static void registerRetryFailedAlarm(
			final long delay, final long wakeInterval) {
		final Intent intent = new Intent(MetaService.ACTION_RETRY_FAILED);
		final PendingIntent operation = PendingIntent.getService(
				sAppInstance, 0, intent, PendingIntent.FLAG_CANCEL_CURRENT);
		
		// Cancel current.
		final AlarmManager am = (AlarmManager) sAppInstance
				.getSystemService(UbuntuOneFiles.ALARM_SERVICE);
		am.cancel(operation);
		
		// Can retry every 15 minutes, but must at least once a wakeInterval.
		Log.i(TAG, "(re)registering retry-failed alarm, wake: " + wakeInterval);
		
		final int sleep = AlarmManager.RTC;
		final int wakeup = AlarmManager.RTC_WAKEUP;
		
		final long withSleepInterval = AlarmManager.INTERVAL_FIFTEEN_MINUTES;
		final long withWakeupInterval = wakeInterval;
		
		long triggerAtTime = System.currentTimeMillis() + delay;
		
		am.setInexactRepeating(sleep, triggerAtTime,
				withSleepInterval, operation);
		am.setInexactRepeating(wakeup, triggerAtTime,
				withWakeupInterval, operation);
	}
	
	private static long getNextRetryFailedInterval() {
		long interval = sLastRetryFailedInterval;
		switch ((int) sLastRetryFailedInterval) {
		case 0:
			interval = AlarmManager.INTERVAL_FIFTEEN_MINUTES;
			break;
		case (int) AlarmManager.INTERVAL_FIFTEEN_MINUTES:
			interval = AlarmManager.INTERVAL_HALF_HOUR;
			break;
		case (int) AlarmManager.INTERVAL_HALF_HOUR:
			interval = AlarmManager.INTERVAL_HOUR;
			break;
		default:
			interval = AlarmManager.INTERVAL_HALF_DAY;
			break;
		}
		return interval;
	}
	
	public static void unregisterRetryFailedAlarm() {
		resetRetryFailedInterval();
		
		final Intent intent = new Intent(MetaService.ACTION_UPLOAD_MEDIA);
		final PendingIntent operation =
				PendingIntent.getService(sAppInstance, 0, intent, 0);
		if (operation != null) {
			final AlarmManager am = (AlarmManager) sAppInstance
					.getSystemService(UbuntuOneFiles.ALARM_SERVICE);
			am.cancel(operation);
			Log.i(TAG, "unregistered retry failed alarm");
		} else {
			Log.d(TAG, "no retry failed alarm to unregister");
		}
	}
	
	private Alarms() {
	}

}
