/*
 * Ubuntu One Files - access Ubuntu One cloud storage on Android platform.
 * 
 * Copyright (C) 2011 Canonical Ltd.
 * Author: Michał Karnicki <michal.karnicki@canonical.com>
 *   
 * This file is part of Ubuntu One Files.
 *  
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *  
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses 
 */

package com.ubuntuone.android.files;

import greendroid.app.GDApplication;

import java.util.logging.Level;

import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.PackageManager.NameNotFoundException;
import android.preference.PreferenceManager;

import com.ubuntuone.android.files.activity.FilesActivity;
import com.ubuntuone.android.files.activity.LoginActivity;
import com.ubuntuone.android.files.activity.StorageActivity;
import com.ubuntuone.android.files.provider.MetaUtilities;
import com.ubuntuone.android.files.receiver.NetworkReceiver;
import com.ubuntuone.android.files.service.MediaCatcher;
import com.ubuntuone.android.files.util.ConfigUtilities;
import com.ubuntuone.android.files.util.Log;
import com.ubuntuone.android.files.util.StorageInfo;
import com.ubuntuone.android.files.util.StorageInfo.StorageNotAvailable;
import com.ubuntuone.rest.RestApi;

public final class UbuntuOneFiles extends GDApplication {
	public static final String APP_TAG = "com.ubuntuone.u1f";
	
	private static final String TAG = UbuntuOneFiles.class.getSimpleName();
	
	private static UbuntuOneFiles sAppInstance;
	
	public UbuntuOneFiles() {
		super();
	}
	
	@Override
	public void onCreate() {
		super.onCreate();
		sAppInstance = this;
		
		setupLogging();
		
		Log.i(TAG, "Starting application");
		
		NetworkReceiver.init();
		
		Preferences.reload();
		PreferenceManager.setDefaultValues(this, R.xml.preferences, false);
		
		Thread.setDefaultUncaughtExceptionHandler(
				new UncaughtExceptionHandler(
						Thread.getDefaultUncaughtExceptionHandler()));
		
		if (ConfigUtilities.isExternalStorageMounted()) {
			// Initially, set storage limit equal to available SD storage.
			long limit = Preferences.getLocalStorageLimit();
			if (limit == -1L) {
				long extAvail = StorageActivity.getAvailableExternalStorageSize();
				Preferences.setLocalStorageLimit(extAvail);
			}
		}
		
		MetaUtilities.resetFailedTransfers();
		try {
			boolean autoUpload = Preferences.getBoolean(
					Preferences.UPLOAD_PHOTOS_KEY, false);
			if (autoUpload && StorageInfo.shouldImmediatelyUploadPhotos()) {
				MediaCatcher.startFrom(this);
			}
		} catch (StorageNotAvailable e) {
			Log.i(TAG, "Couldn't start Media Catcher");
		}
	}
	
	@Override
	public void onTerminate() {
		Log.i(TAG, "Terminating application");
		super.onTerminate();
	}
	
	private void setupLogging() {
		Log.setApplicationTag(APP_TAG);
		// Set application Logger logging level.
		Log.setLogLevel(Level.OFF);
		// Set logcat output verbosity.
		Log.setLogcatLogLevel(android.util.Log.INFO);
		// Set REST Logger logging level.
		RestApi.setLoggingLevel(Level.FINE);
		// Set User-Agent of the REST client.
		RestApi.USER_AGENT = UbuntuOneFiles.class.getSimpleName() + getVersion();
	}

	/**
	 * Gets Ubuntu One Files application instance.
	 * 
	 * @return the app instance
	 */
	public static UbuntuOneFiles getInstance() {
		return sAppInstance;
	}
	
	/**
	 * Gets current application version.
	 * 
	 * @return the app version
	 */
	public static String getVersion() {
		final Context ctx = sAppInstance;
		String version = null;
		PackageManager pm = ctx.getPackageManager();
		try {
			PackageInfo info = pm.getPackageInfo(ctx.getPackageName(), 0);
			version = info.versionName;
		} catch (NameNotFoundException e) {
			Log.e(TAG, "can't get app version", e);
			version = "?";
		}
		return version;
	}
	
	@Override
	public Class<?> getHomeActivityClass() {
		// We want the ActionBar to look the same on both of these, thus:
		if (Preferences.hasTokens(this))
			return FilesActivity.class;
		else
			return LoginActivity.class;
	}
	
	public static void showHome(Context context) {
		final Intent intent = new Intent(context,
				sAppInstance.getHomeActivityClass());
		context.startActivity(intent);
	}
	
	private class UncaughtExceptionHandler implements
			Thread.UncaughtExceptionHandler {
		
		private Thread.UncaughtExceptionHandler mDefaultHandler;
		
		public UncaughtExceptionHandler(
				Thread.UncaughtExceptionHandler defaultHandler) {
			mDefaultHandler = defaultHandler;
		}
		
		public void uncaughtException(Thread thread, Throwable ex) {
			// Set the crash flag.
			Preferences.set(Preferences.OOPS_FLAG_KEY);
			
			Log.d(TAG, "=== UNHANDLED === exception", ex);
			Log.stopLogging();
			
			// Pass it to default uncaught exception handler.
			mDefaultHandler.uncaughtException(thread, ex);
		}
	}
	
}
