/*
 * Ubuntu One Files - access Ubuntu One cloud storage on Android platform.
 * 
 * Copyright (C) 2011 Canonical Ltd.
 * Author: Michał Karnicki <michal.karnicki@canonical.com>
 *   
 * This file is part of Ubuntu One Files.
 *  
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *  
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses 
 */

package com.ubuntuone.android.files.activity;

import greendroid.app.GDListActivity;
import greendroid.widget.ActionBarItem;
import greendroid.widget.ActionBarItem.Type;
import greendroid.widget.LoaderActionBarItem;
import greendroid.widget.NormalActionBarItem;
import greendroid.widget.QuickActionGrid;
import greendroid.widget.QuickActionWidget;
import greendroid.widget.QuickActionWidget.OnQuickActionClickListener;

import java.io.File;
import java.net.URI;
import java.net.URISyntaxException;

import org.apache.http.client.methods.HttpPost;

import android.app.Activity;
import android.app.AlertDialog;
import android.app.Dialog;
import android.app.ProgressDialog;
import android.content.ActivityNotFoundException;
import android.content.ContentResolver;
import android.content.Context;
import android.content.DialogInterface;
import android.content.DialogInterface.OnClickListener;
import android.content.Intent;
import android.content.res.Configuration;
import android.database.Cursor;
import android.net.ConnectivityManager;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.view.ContextMenu;
import android.view.ContextMenu.ContextMenuInfo;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.widget.AdapterView.AdapterContextMenuInfo;
import android.widget.CursorAdapter;
import android.widget.EditText;
import android.widget.ImageButton;
import android.widget.ImageView;
import android.widget.ListView;
import android.widget.TextView;

import com.google.android.apps.analytics.GoogleAnalyticsTracker;
import com.ubuntuone.android.files.Analytics;
import com.ubuntuone.android.files.Preferences;
import com.ubuntuone.android.files.R;
import com.ubuntuone.android.files.provider.MetaContract.Nodes;
import com.ubuntuone.android.files.provider.MetaContract.ResourceState;
import com.ubuntuone.android.files.provider.MetaUtilities;
import com.ubuntuone.android.files.receiver.NetworkReceiver;
import com.ubuntuone.android.files.service.MediaCatcher;
import com.ubuntuone.android.files.service.MetaService;
import com.ubuntuone.android.files.service.MetaService.Status;
import com.ubuntuone.android.files.service.MetaServiceHelper;
import com.ubuntuone.android.files.service.UpDownService;
import com.ubuntuone.android.files.service.UpDownServiceHelper;
import com.ubuntuone.android.files.util.ChangeLogUtils;
import com.ubuntuone.android.files.util.ConfigUtilities;
import com.ubuntuone.android.files.util.DateUtilities;
import com.ubuntuone.android.files.util.DetachableResultReceiver;
import com.ubuntuone.android.files.util.DetachableResultReceiver.Receiver;
import com.ubuntuone.android.files.util.FileUtilities;
import com.ubuntuone.android.files.util.Log;
import com.ubuntuone.android.files.util.PathTracker;
import com.ubuntuone.android.files.util.StorageInfo;
import com.ubuntuone.android.files.util.UIUtil;
import com.ubuntuone.android.files.util.UIUtil.BlackQuickAction;
import com.ubuntuone.android.files.util.VersionUtilities;
import com.ubuntuone.rest.resources.NodeInfo;

public class FilesActivity extends GDListActivity implements Receiver {
	
	private static final String TAG = FilesActivity.class.getSimpleName(); 
	
	private static interface ActionBar {
		public static final int REFRESH = 0;
		public static final int UPLOAD = 1;
	}
	
	private static interface QuickAction {
		public static final int UPLOAD_IMAGE = 0;
		public static final int UPLOAD_VIDEO = 1;
		public static final int UPLOAD_AUDIO = 2;
		public static final int UPLOAD_FILE = 3;
		public static final int NEW_FOLDER = 4;
		
		public static final int DIRECTORY_DOWNLOAD = 0;
		public static final int DIRECTORY_RENAME = 1;
		public static final int DIRECTORY_DELETE = 2;
		
		public static final int FILE_RENAME = 0;
		public static final int FILE_DELETE = 1;
		public static final int FILE_SHARE = 2;
		public static final int FILE_PUBLISH = 3;
		public static final int FILE_SHARE_URL = 4;
		public static final int FILE_COPY_URL = 5;
	}
	
	private static final int REQUEST_SIGN_IN = 1;
	private static final int REQUEST_PREFERENCES = 2;
	private static final int REQUEST_ADD_FILE = 3;
	
	public static final String ACTION_PICK_AUTO_UPLOAD_DIRECTORY =
			"com.ubuntuone.android.files.ACTION_PICK_AUTO_UPLOAD_DIRECTORY";
	
	private final String EXTRA_DIALOG_EXTRAS = "extra_dialog_extras";
	private final String EXTRA_TOP_POSITION = "extra_top_position";
	private final String EXTRA_TOP = "extra_top";
	
	private final int DIALOG_DOWNLOAD_ID = 1;
	private final int DIALOG_RENAME_ID = 2;
	private final int DIALOG_DELETE_ID = 3;
	private final int DIALOG_EMPTY_DIRECTORY_ID = 4;
	private final int DIALOG_NEW_DIRECTORY_ID = 5;
	private final int DIALOG_FOUND_NEW_STORAGE = 6;
	private final int DIALOG_CHECK_DIRECTORY_SIZE_ID = 7;
	private final int DIALOG_PICK_AUTO_UPLOAD_DIRECTORY_ID = 8;
	private final int DIALOG_NO_NETWORK = 9;
	
	private Dialog mDownloadDialog;
	private Dialog mRenameDialog;
	private Dialog mDeleteDialog;
	private Dialog mEmptyDirDialog;
	private Dialog mNewDirectoryDialog;
	private Dialog mFoundNewStorageDialog;
	private Dialog mCheckDirectorySizeDialog;
	private Dialog mPickAutoUploadDirectory;
	private Bundle mDialogExtras;
	
	private GoogleAnalyticsTracker mTracker;

	private Handler mHandler;
	private DetachableResultReceiver mReceiver;
	
	private ContentResolver mResolver;
	
	private LoaderActionBarItem mLoaderItem;
	private QuickActionWidget mUploadGrid;
	
	private QuickActionWidget mDirQuickActions;
	private QuickActionWidget mFileQuickActions;
	private ViewHolder mContextViewHolder;
	
	private TextView mEmptyTextView;
	private int mTopPosition, mTop;
	
	private FilesAdapter mAdapter;
	private PathTracker mPathTracker;
	
	private boolean mIsPickDirectoryMode = false;
	
	@Override
	public int createLayout() {
		return R.layout.activity_list;
	}

	@Override
	protected void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		
		mTracker = GoogleAnalyticsTracker.getInstance();
		mTracker.start(Analytics.U1F_ACCOUNT, this);
		mTracker.trackPageView(TAG);

		mHandler = new Handler();
		mReceiver = new DetachableResultReceiver(mHandler);
		
		mResolver = getContentResolver();
		
		// Override splash screen background.
		getWindow().setBackgroundDrawableResource(
				android.R.drawable.screen_background_light);
		
		final ImageButton homeButton =
				(ImageButton) getActionBar().findViewById(
						R.id.gd_action_bar_home_item);
		homeButton.setImageResource(R.drawable.u1_logo);
		
		// Refresh
		addActionBarItem(Type.Refresh);
		mLoaderItem = (LoaderActionBarItem) getActionBar()
				.getItem(ActionBar.REFRESH);
		mLoaderItem.setDrawable(R.drawable.ic_act_action_bar_refresh);
		// Upload
		addActionBarItem(Type.Add);
		NormalActionBarItem addItem = (NormalActionBarItem) getActionBar()
				.getItem(ActionBar.UPLOAD);
		addItem.setDrawable(R.drawable.ic_act_action_bar_add);
		
		onCreateUploadQuickActionGrid();
		
		getActionBar().setBackgroundResource(R.drawable.action_bar_background);
		
		mEmptyTextView = (TextView) findViewById(android.R.id.empty);
		
		ListView listView = getListView();
		listView.setEmptyView(mEmptyTextView);
		listView.setSelector(R.drawable.list_selector);
		listView.setFastScrollEnabled(true);
		
		registerForContextMenu(getListView());
		
		mPathTracker = new PathTracker();
		
		if (ConfigUtilities.getAvailableNetworkType(this) == -1) {
			UIUtil.showToast(this, R.string.toast_working_offline, true);
		}
		
		if (Preferences.hasTokens(this)) {
			onRefresh();
			suggestAutoUploadConfiguration();
			ChangeLogUtils.suggestShowingChangelog(this);
		} else {
			final Intent intent = getIntent();
			final String data = (intent != null) ? intent.getDataString() : null;
			if (data != null && data.startsWith("x-ubuntuone-sso")) {
				validate();
			} else {
				signIn();
			}
		}
	}
	
	/*
	 * Sadly, FLAG_ACTIVITY_CLEAR_WHEN_TASK_RESET works only for the launcher.
	 * If home button long-pressed and used to get back to activity, a validation
	 * request leftover may be visible. I have no fix for this edge case ATM. 
	 */
	
	private void signIn() {
		final Intent intent = new Intent(LoginActivity.ACTION_SIGN_IN);
		intent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_WHEN_TASK_RESET);
		startActivityForResult(intent, REQUEST_SIGN_IN);
	}
	
	private void validate() {
		final Intent intent = new Intent(LoginActivity.ACTION_VALIDATE);
		intent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_WHEN_TASK_RESET);
		startActivityForResult(intent, REQUEST_SIGN_IN);
	}
	
	private void suggestAutoUploadConfiguration() {
		try {
			boolean seenStorage = StorageInfo.isAlreadyConfigured();
			if (! seenStorage) {
				showDialog(DIALOG_FOUND_NEW_STORAGE);
			}
		} catch (StorageInfo.StorageNotAvailable e) {
			UIUtil.showToast(this, R.string.toast_storage_is_not_accessible);
			Log.e(TAG, "Seen storage before?", e);
		}
	}
	
	private void onCreateUploadQuickActionGrid() {
		mUploadGrid = new QuickActionGrid(this);
		mUploadGrid.addQuickAction(new BlackQuickAction(this,
				R.drawable.ic_act_upload_photo,
				R.string.action_bar_add_image));
		mUploadGrid.addQuickAction(new BlackQuickAction(this,
				R.drawable.ic_act_upload_video,
				R.string.action_bar_add_video));
		mUploadGrid.addQuickAction(new BlackQuickAction(this,
				R.drawable.ic_act_upload_audio,
				R.string.action_bar_add_audio));
		mUploadGrid.addQuickAction(new BlackQuickAction(this,
				R.drawable.ic_act_upload_file,
				R.string.action_bar_add_file));
		mUploadGrid.addQuickAction(new BlackQuickAction(this,
				R.drawable.ic_act_new_folder,
				R.string.action_bar_new_directory));
		mUploadGrid.setOnQuickActionClickListener(mUploadQuickActionListener);
	}
	
	@Override
	public void onConfigurationChanged(Configuration newConfig) {
		super.onConfigurationChanged(newConfig);
		if (mUploadGrid != null && mUploadGrid.isShowing()) {
			mUploadGrid.dismiss();
		}
		if (mDirQuickActions != null && mDirQuickActions.isShowing()) {
			mDirQuickActions.dismiss();
		}
		if (mFileQuickActions != null && mFileQuickActions.isShowing()) {
			mFileQuickActions.dismiss();
		}
		onCreateUploadQuickActionGrid();
	}
	
	@Override
	protected void onSaveInstanceState(Bundle outState) {
		super.onSaveInstanceState(outState);
		// Save dialog extras.
		outState.putBundle(EXTRA_DIALOG_EXTRAS, mDialogExtras);
		// Save list position.
		final ListView l = getListView();
		int topPosition = l.getFirstVisiblePosition();
		final View topView = l.getChildAt(0);
		final int top = (topView == null) ? 0 : topView.getTop();
		outState.putInt(EXTRA_TOP_POSITION, topPosition);
		outState.putInt(EXTRA_TOP, top);
	}
	
	@Override
	protected void onRestoreInstanceState(Bundle savedInstanceState) {
		// Restore dialog extras.
		mDialogExtras = savedInstanceState.getBundle(EXTRA_DIALOG_EXTRAS);
		// Restore list view position.
		int topPosition = savedInstanceState.getInt(EXTRA_TOP_POSITION);
		int top = savedInstanceState.getInt(EXTRA_TOP);
		getListView().setSelectionFromTop(topPosition, top);
		
		super.onRestoreInstanceState(savedInstanceState);
	}
	
	@Override
	protected void onNewIntent(Intent intent) {
		super.onNewIntent(intent);
		final String data = intent.getDataString();
		if (data != null && data.startsWith("x-ubuntuone-sso")) {
			validate();
		} else if (ACTION_PICK_AUTO_UPLOAD_DIRECTORY.equals(intent.getAction())) {
			mIsPickDirectoryMode = true;
			showDialog(DIALOG_PICK_AUTO_UPLOAD_DIRECTORY_ID);
		}
	}
	
	@Override
	protected void onResume() {
		super.onResume();
		
		if (mReceiver != null) {
			mReceiver.setReceiver(this);
			UpDownServiceHelper.registerCallback(mReceiver);
			if (MetaService.sIsWorking) {
				showSpinner();
			} else {
				hideSpinner();
			}
			setCursorAdapterInBackground();
		}
	}
	
	/**
	 * Requests appropriate cursor and sets the {@link FilesAdapter} on
	 * a background thread, to avoid ANR due to database operation. Also,
	 * we're not using deprecated .requery() anymore.
	 */
	private void setCursorAdapterInBackground() {
		new Thread(new Runnable() {
			public void run() {
				final Cursor cursor = mPathTracker.isAtRoot() ?
						MetaUtilities.getVisibleTopNodesCursor() :
							getFilesCurosor(mPathTracker.getCurrentNode());
				startManagingCursor(cursor);
				// We have a new cursor, set a new list adapter.
				runOnUiThread(new Runnable() {
					public void run() {
						mAdapter = new FilesAdapter(FilesActivity.this,
								cursor, true);
						setListAdapter(mAdapter);
					}
				});
			}
		}).start();
	}
	
	@Override
	public void startManagingCursor(Cursor c) {
		if (Build.VERSION.SDK_INT < VersionUtilities.HONEYCOMB) {
			super.startManagingCursor(c);
		}
	}

	@Override
	protected void onPause() {
		if (mReceiver != null) {
			UpDownServiceHelper.unregisterCallback(mReceiver);
			mReceiver.detach();
		}
		super.onPause();
	}
	
	@Override
	public void onDestroy() {
		if (mTracker != null) {
			mTracker.dispatch();
			mTracker.stop();
		}
		super.onDestroy();
	}
	
	@Override
	protected Dialog onCreateDialog(int id) {
		Dialog dialog;
		switch (id) {
		case DIALOG_DOWNLOAD_ID:
			dialog = mDownloadDialog = buildDownloadDialog(mDialogExtras);
			break;
		case DIALOG_RENAME_ID:
			dialog = mRenameDialog = buildRenameDialog(mDialogExtras);
			break;
		case DIALOG_DELETE_ID:
			dialog = mDeleteDialog = buildDeleteDialog(mDialogExtras);
			break;
		case DIALOG_EMPTY_DIRECTORY_ID:
			dialog = mEmptyDirDialog = buildEmptyDirDialog(mDialogExtras);
			break;
		case DIALOG_NEW_DIRECTORY_ID:
			dialog = mNewDirectoryDialog = buildNewDirDialog(mDialogExtras);
			break;
		case DIALOG_FOUND_NEW_STORAGE:
			dialog = mFoundNewStorageDialog =
					buildSeenNewStorageDialog(mDialogExtras);
			break;
		case DIALOG_CHECK_DIRECTORY_SIZE_ID:
			dialog = mCheckDirectorySizeDialog =
					buildCheckDirectorySizeDialog(mDialogExtras);
			break;
		case DIALOG_PICK_AUTO_UPLOAD_DIRECTORY_ID:
			dialog = mPickAutoUploadDirectory =
					buildPickAutoUploadDirectory(mDialogExtras);
			break;
		case DIALOG_NO_NETWORK:
			dialog = buildNoNetworkDialog();
			break;
		default:
			dialog = null;
			break;
		}
		return dialog;
	}
	
	private OnClickListener onClickDismiss = new OnClickListener() {
		
		public void onClick(DialogInterface dialog, int which) {
			dialog.dismiss();
		}
		
	};
	
	private Dialog buildNoNetworkDialog() {
		final AlertDialog dialog = new AlertDialog.Builder(this)
				.setIcon(android.R.drawable.ic_dialog_alert)
				.setTitle(R.string.dialog_no_network_title)
				.setMessage(R.string.dialog_no_network_message)
				.setPositiveButton(R.string.ok, onClickDismiss)
				.create();
		return dialog;
	}
	
	private Dialog buildInsufficientSpaceDialog(long needed, long free) {
		final String message = String.format(
				getString(R.string.dialog_no_space_message_fmt),
				FileUtilities.getHumanReadableSize(needed),
				FileUtilities.getHumanReadableSize(free));
		final AlertDialog dialog = new AlertDialog.Builder(this)
				.setIcon(android.R.drawable.ic_dialog_alert)
				.setTitle(R.string.dialog_no_space_title)
				.setMessage(message)
				.setPositiveButton(R.string.ok, new OnClickListener() {
					public void onClick(DialogInterface dialog, int which) {
						dialog.dismiss();
						removeDialog(DIALOG_DOWNLOAD_ID);
					}
				})
				.create();
		return dialog;
	}
	
	private Dialog buildEmptyDirDialog(final Bundle extras) {
		final String title = extras.getString(MetaService.EXTRA_NAME);
		final AlertDialog dialog = new AlertDialog.Builder(this)
				.setIcon(android.R.drawable.ic_dialog_info)
				.setTitle(title)
				.setMessage(R.string.dialog_nothing_to_download)
				.setPositiveButton(R.string.ok, new OnClickListener() {
					public void onClick(DialogInterface dialog, int which) {
						dialog.dismiss();
						removeDialog(DIALOG_DOWNLOAD_ID);
					}
				})
				.create();
		return dialog;
	}

	private Dialog buildSeenNewStorageDialog(final Bundle extras) {
		final AlertDialog dialog = new AlertDialog.Builder(this)
				.setIcon(R.drawable.ic_dialog_new_sdcard)
				.setTitle("New Storage Detected")
				.setMessage(R.string.explain_configure_new_storage)
				.setPositiveButton(R.string.btn_act_upload_all, new OnClickListener() {
					public void onClick(DialogInterface dialog, int which) {
						Log.i(TAG, "will upload old and new photos");
						dialog.dismiss();
						removeDialog(DIALOG_FOUND_NEW_STORAGE);
						try {
							StorageInfo.setShouldImmediatelyUploadPhotos(true);
							StorageInfo.setHasConfiguredStorage();
							Preferences.setShouldImmediatelyUploadPhotos(true);
							Intent intent = new Intent(MetaService.ACTION_UPLOAD_MEDIA);
							intent.putExtra(MetaService.EXTRA_TIMESTAMP, 0L);
							FilesActivity.this.startService(intent);
							MediaCatcher.startFrom(FilesActivity.this);
							UIUtil.showToast(FilesActivity.this, R.string.toast_uploading_all_photos);
						} catch (StorageInfo.StorageNotAvailable e) {
							UIUtil.showToast(FilesActivity.this, R.string.storage_is_not_accessible_not_setting_preference);
							Log.e(TAG, "First seen storage; setting config.", e);
						}
					}
				}).setNeutralButton(R.string.btn_act_upload_new_only, new OnClickListener() {
					public void onClick(DialogInterface dialog, int which) {
						Log.i(TAG, "will upload new photos only");
						dialog.dismiss();
						try {
							long now = System.currentTimeMillis() / 1000;
							StorageInfo.setLastUploadedPhotoTimestamp(now); // seconds
							if (now != StorageInfo.getLastUploadedPhotoTimestamp()) {
								Log.e(TAG, "couldn't save current timestamp!");
							}
							
							Preferences.setShouldImmediatelyUploadPhotos(true);
							StorageInfo.setShouldImmediatelyUploadPhotos(true);
							StorageInfo.setHasConfiguredStorage();
							MediaCatcher.startFrom(FilesActivity.this);
						} catch (StorageInfo.StorageNotAvailable e) {
							UIUtil.showToast(FilesActivity.this, R.string.storage_is_not_accessible_not_setting_preference);
							Log.e(TAG, "First seen storage; setting config.", e);
						}
						removeDialog(DIALOG_FOUND_NEW_STORAGE);
					}
				}).setNegativeButton(R.string.btn_act_upload_none, new OnClickListener() {
					public void onClick(DialogInterface dialog, int which) {
						Log.i(TAG, "will not upload photos");
						dialog.dismiss();
						try {
							StorageInfo.setShouldImmediatelyUploadPhotos(false);
							StorageInfo.setHasConfiguredStorage();
							Preferences.setShouldImmediatelyUploadPhotos(false);
						} catch (StorageInfo.StorageNotAvailable e) {
							UIUtil.showToast(FilesActivity.this, R.string.storage_is_not_accessible_not_setting_preference);
							Log.e(TAG, "First seen storage; setting config.", e);
						}
						removeDialog(DIALOG_FOUND_NEW_STORAGE);
					}
				})
				.create();
		dialog.setOwnerActivity(this);
		return dialog;
	}
	
	private Dialog buildCheckDirectorySizeDialog(final Bundle extras) {
		final ProgressDialog dialog = new ProgressDialog(this);
		dialog.setMessage(getString(R.string.dialog_checking_size_message));
		dialog.setIndeterminate(true);
		dialog.setCancelable(true);
		return dialog;
	}
	
	/**
	 * This dialog applies to downloading directories only.
	 * 
	 * @param extras
	 * @return
	 */
	private Dialog buildDownloadDialog(final Bundle extras) {
		final String resourcePath =
				extras.getString(MetaService.EXTRA_RESOURCE_PATH);
		final long size = extras.getLong(MetaService.EXTRA_SIZE);
		final String sizeText = FileUtilities.getHumanReadableSize(size);
		
		if (size == 0L) {
			// We don't care if there are few empty files of 0 bytes size.
			return buildEmptyDirDialog(extras);
		}
		
		final long free = StorageActivity.getAvailableExternalStorageSize();
		if (free < size) {
			return buildInsufficientSpaceDialog(size, free);
		}
		
		final StringBuilder msgBuilder = new StringBuilder(32);
		msgBuilder.append(String.format(
				getString(R.string.dialog_downloading_dir_is_x_large), sizeText));
		int network = ConfigUtilities.getAvailableNetworkType(this);
		if (network == -1) {
			return buildNoNetworkDialog();
		} else if (network == ConnectivityManager.TYPE_MOBILE) {
			msgBuilder.append(" ");
			msgBuilder.append(getString(R.string.dialog_downloading_dir_on_mobile));
		}
		msgBuilder.append(" ");
		msgBuilder.append(getString(R.string.dialog_downloading_dir_are_you_sure));
		
		final OnClickListener onClick = new OnClickListener() {
			
			private void download() {
				UpDownServiceHelper.download(FilesActivity.this, resourcePath,
						false);
			}
			
			public void onClick(DialogInterface dialog, int which) {
				switch (which) {
				case Dialog.BUTTON_POSITIVE:
					download();
					break;
				case Dialog.BUTTON_NEGATIVE:
					// Will dismiss below.
					break;
				default:
					Log.e(TAG, "no such button");
					break;
				}
				removeDialog(DIALOG_DOWNLOAD_ID);
			}
			
		};
		
		final AlertDialog dialog = new AlertDialog.Builder(this)
				.setIcon(android.R.drawable.ic_dialog_info)
				.setTitle(R.string.dialog_downloading_dir_title)
				.setMessage(msgBuilder.toString())
				.setPositiveButton(R.string.yes, onClick)
				.setNegativeButton(R.string.no, onClick)
				.create();
		return dialog;
	}
	
	private Dialog buildRenameDialog(final Bundle extras) {
		final String resourcePath =
				extras.getString(MetaService.EXTRA_RESOURCE_PATH);
		final String name = extras.getString(MetaService.EXTRA_NAME);
		final String path = extras.getString(MetaService.EXTRA_PATH);
		Log.d(TAG, "dialog built with PATH: " + path);
		Log.d(TAG, "buildRenameDialog: name is " + name);
		
		final OnClickListener onClick = new OnClickListener() {
			
			private final String reserved = "/";
			
			private void rename(DialogInterface dialog) {
				EditText newNameEdit = (EditText) mRenameDialog
						.findViewById(R.id.dialog_edit_text);
				final String newName = newNameEdit.getText().toString();
				if (newName.length() == 0) {
					UIUtil.showToast(FilesActivity.this,
							R.string.toast_cant_rename_to_nothing);
					dialog.dismiss();
					return;
				} else if (newName.equals(name)) {
					UIUtil.showToast(FilesActivity.this,
							R.string.toast_no_change);
					return;
				}
				
				for (int i = 0; i < reserved.length(); i++) {
					if (newName.indexOf(reserved.charAt(i)) > -1) {
						final String msg = String.format(
								getString(R.string.toast_file_name_cant_contain),
								reserved.charAt(i));
								
						UIUtil.showToast(FilesActivity.this, msg, true);
						return;
					}
				}
				
				final int cutAt = path.lastIndexOf("/") + 1;
				final String parent = path.substring(0, cutAt); 
				final String newPath = parent.concat(newName);
				MetaServiceHelper.rename(FilesActivity.this, mReceiver,
						resourcePath, newPath, newName);
			}
			
			public void onClick(DialogInterface dialog, int which) {
				
				switch (which) {
				case Dialog.BUTTON_POSITIVE:
					rename(dialog);
					break;
				case Dialog.BUTTON_NEGATIVE:
					// Will dismiss below.
					break;
				default:
					Log.e(TAG, "no such button");
					break;
				}
				removeDialog(DIALOG_RENAME_ID);
			}
			
		};
		
		// TODO karni: Make this a regular text dialog.
		final View dialogView = getLayoutInflater()
				.inflate(R.layout.dialog_edittext,
						(ViewGroup) findViewById(R.id.dialog_root));
		final EditText editText =
			(EditText) dialogView.findViewById(R.id.dialog_edit_text);
		editText.setText(name);
		
		final AlertDialog dialog = new AlertDialog.Builder(this)
				.setView(dialogView)
				.setIcon(android.R.drawable.ic_dialog_info)
				.setTitle(R.string.dialog_rename_directory_title)
				.setPositiveButton(R.string.context_rename, onClick)
				.setNegativeButton(R.string.cancel, onClick)
				.create();
		return dialog;
	}
	
	private Dialog buildDeleteDialog(final Bundle extras) {
		final String resourcePath =
				extras.getString(MetaService.EXTRA_RESOURCE_PATH);
		final String name =
				extras.getString(MetaService.EXTRA_NAME);
		
		final OnClickListener onClick = new OnClickListener() {
			
			public void onClick(DialogInterface dialog, int which) {
				switch (which) {
				case Dialog.BUTTON_POSITIVE:
					MetaServiceHelper.deleteNode(
							FilesActivity.this, mReceiver, resourcePath);
					break;
				case Dialog.BUTTON_NEGATIVE:
					// Will dismiss below.
					break;
				default:
					Log.e(TAG, "no such button");
					break;
				}
				removeDialog(DIALOG_DELETE_ID);
			}
			
		};
		
		final AlertDialog dialog = new AlertDialog.Builder(this)
				.setIcon(android.R.drawable.ic_dialog_alert)
				.setTitle(name)
				.setMessage(R.string.dialog_delete_item_message)
				.setPositiveButton(R.string.yes, onClick)
				.setNegativeButton(R.string.no, onClick)
				.create();
		return dialog;
	}
	
	private Dialog buildNewDirDialog(final Bundle extras) {
		final String path = mPathTracker.isAtRoot()
				? Preferences.U1_RESOURCE : mPathTracker.getCurrentNode();
		Log.d(TAG, "dialog built with path: " + path);
		
		final OnClickListener onClick = new OnClickListener() {
			
			private final String reserved = "|\\?*<\":>+[]/'";
			
			private void create(DialogInterface dialog) {
				EditText editText = (EditText) mNewDirectoryDialog
						.findViewById(R.id.dialog_edit_text);
				final String newName = editText.getText().toString();
				if (newName.length() == 0) {
					// TODO karni: Disallow empty name.
					UIUtil.showToast(FilesActivity.this,
							R.string.toast_name_cant_be_nothing);
					dialog.dismiss();
					return;
				}
				
				for (int i = 0; i < reserved.length(); i++) {
					if (newName.indexOf(reserved.charAt(i)) > -1) {
						final String msg = String.format(
								getString(R.string.toast_file_name_cant_contain),
								reserved.charAt(i));
								
						UIUtil.showToast(FilesActivity.this, msg, true);
						return;
					}
				}
				
				final String resourcePath = path + "/" + newName;
				MetaServiceHelper.createDirectory(
						FilesActivity.this, mReceiver, resourcePath, newName);
			}
			
			public void onClick(DialogInterface dialog, int which) {
				
				switch (which) {
				case Dialog.BUTTON_POSITIVE:
					create(dialog);
					break;
				case Dialog.BUTTON_NEGATIVE:
					// Will dismiss below.
					break;
				default:
					Log.e(TAG, "no such button");
					break;
				}
				removeDialog(DIALOG_NEW_DIRECTORY_ID);
			}
			
		};
		
		final View dialogView = getLayoutInflater()
				.inflate(R.layout.dialog_edittext,
						(ViewGroup) findViewById(R.id.dialog_root));
		final TextView textView =
			(TextView) dialogView.findViewById(R.id.dialog_text);
		textView.setText(R.string.dialog_create_directory_title);
		
		final AlertDialog dialog = new AlertDialog.Builder(this)
				.setView(dialogView)
				.setIcon(android.R.drawable.ic_dialog_info)
				.setTitle(R.string.dialog_create_directory_title)
				.setPositiveButton(R.string.create, onClick)
				.setNegativeButton(R.string.cancel, onClick)
				.create();
		return dialog;
	}
	
	private Dialog buildPickAutoUploadDirectory(final Bundle extras) {
		final OnClickListener onClick = new OnClickListener() {
			public void onClick(DialogInterface dialog, int which) {
				dialog.dismiss();
			}
		};
		
		final AlertDialog dialog = new AlertDialog.Builder(this)
				.setIcon(android.R.drawable.ic_dialog_info)
				.setTitle(R.string.dialog_pick_photos_auto_upload_dialog_title)
				.setMessage(getText(R.string.dialog_pick_photos_auto_upload_directory_message))
				.setPositiveButton(R.string.ok, onClick)
				.create();
		return dialog;
	}

	@Override
	public boolean onHandleActionBarItemClick(ActionBarItem item, int position) {
		switch (position) {		
		case ActionBar.REFRESH:
			onActionBarRereshClicked();
			break;
		case ActionBar.UPLOAD:
			onActionBarUploadClicked(item.getItemView());
			break;
		default:
			Log.e(TAG, "unknown action bar action");
			return false;
		}
		return true;
	}
	
	private void onRefresh(String... strings) {
		if (strings.length == 0) {
			MetaServiceHelper.updateUserInfo(this, mReceiver);
			MetaServiceHelper.refreshNode(
					this, mReceiver, Preferences.U1_RESOURCE);
			prefetchPhotos();
		} else {
			MetaServiceHelper.refreshNode(
					this, mReceiver, strings[0]);
		}
		prefetchCurrent();
	}
	
	private static boolean sAlreadyPrefetchedPhotos = false;
	
	private void prefetchPhotos() {
		Log.i(TAG, "prefetchPhotos()");
		if (!sAlreadyPrefetchedPhotos) {
			sAlreadyPrefetchedPhotos = true;
			MetaServiceHelper.refreshNode(this, mReceiver,
					Preferences.getPhotosUploadResource());
		}
	}
	
	private void prefetchCurrent() {
		final boolean prefetch = Preferences.getBoolean(
				Preferences.ALLOW_PREFETCHING, false);
		if (prefetch) {
			String resourcePath = mPathTracker.isAtRoot()
			? Preferences.U1_RESOURCE : mPathTracker.getCurrentNode();
			Log.i(TAG, "prefetchCurrent() " + resourcePath);
			MetaServiceHelper.prefetchNodeRecurse(this, mReceiver, resourcePath);
		}
	}
	
	private void onActionBarRereshClicked() {
		if (!NetworkReceiver.isConnected()) {
			hideSpinner();
			showDialog(DIALOG_NO_NETWORK);
		} else if (mPathTracker.isAtRoot()) {
			onRefresh();
		} else {
			onRefresh(mPathTracker.getCurrentNode());
		}
	}
	
	private void onActionBarUploadClicked(View v) {
		mUploadGrid.show(v);
	}
	
	@Override
	public boolean onPrepareOptionsMenu(Menu menu) {
		MenuInflater inflater = getMenuInflater();
		menu.clear();
		if (mIsPickDirectoryMode) {
			inflater.inflate(R.menu.options_menu_pick_auto_upload, menu);
		} else {
			inflater.inflate(R.menu.options_menu_dashboard, menu);
		}
		return super.onPrepareOptionsMenu(menu);
	}
	
	@Override
	public boolean onOptionsItemSelected(MenuItem item) {
		switch (item.getItemId()) {
		case R.id.option_refresh:
			onOptionRefreshSelected();
			break;
		case R.id.option_preferences:
			onOptionPreferencesSelected();
			break;
		case R.id.option_auto_upload_pick:
			onOptionAutoUploadPickSelected();
			break;
		case R.id.option_auto_upload_cancel:
			onOptionAutoUploadCancelSelected();
			break;
		default:
			return false;
		}
		return true;
	}
	
	@SuppressWarnings("unused")
	private void onOptionUploadSelected() {
		// TODO karni: implement upload in options menu
	}
	
	private void onOptionRefreshSelected() {
		onRefresh();
	}
	
	private void onOptionPreferencesSelected() {
		final Intent intent = new Intent(this, PreferencesActivity.class);
		startActivityForResult(intent, REQUEST_PREFERENCES);
	}
	
	private void onOptionAutoUploadPickSelected() {
		final String node = mPathTracker.getCurrentNode();
		String uploadDirectory = null;
		if (node == null) {
			uploadDirectory = Preferences.U1_RESOURCE.substring(3);
		} else {
			uploadDirectory = node.substring(3);
		}
		Preferences.setPhotosUploadDirectory(uploadDirectory);
		UIUtil.showToast(this,
				getString(R.string.toast_photos_will_auto_upload_to_fmt, uploadDirectory),
				true);
		mIsPickDirectoryMode = false;
	}
	
	private void onOptionAutoUploadCancelSelected() {
		UIUtil.showToast(this,
				getString(R.string.toast_cancelled_selecting_auto_upload_directory),
				true);
		mIsPickDirectoryMode = false;
	}
	
	public void onReceiveResult(int resultCode, final Bundle resultData) {
		final String lastResourcePath =
				resultData.getString(MetaService.EXTRA_RESOURCE_PATH);
		
		switch (resultCode) {
		case Status.RUNNING:
			showSpinner();
			awaitWithListEmptyTextView();
			break;
		case Status.PROGRESS:
			// Unused.
			break;
		case Status.FINISHED:
			hideSpinner();
			
			final String currentResourcePath = mPathTracker.getCurrentNode();
			if (currentResourcePath == null) {
				mHandler.postDelayed(new Runnable() {
					public void run() {
						if (mPathTracker.getCurrentNode() == null) {
							// Most probably the user has no files yet. If he has,
							// he should already see them and this will have no impact.
							resetListEmptyTextView(null);
						}
					}
				}, 5000);
			} else if (currentResourcePath.equals(lastResourcePath)) {
				resetListEmptyTextView(lastResourcePath);
			}
			final String method =
					resultData.getString(UpDownService.EXTRA_METHOD);
			if (method != null) {
				final String fmt = method.equals(HttpPost.METHOD_NAME)
						? getString(R.string.toast_uploaded_x)
						: getString(R.string.toast_downloaded_x);
				final String msg = String.format(fmt,
						resultData.getString(UpDownService.EXTRA_NAME));
				runOnUiThread(new Runnable() {
					public void run() {
						UIUtil.showToast(FilesActivity.this, msg);
					}
				});
			}
			
			if (mCheckDirectorySizeDialog != null
					&& mCheckDirectorySizeDialog.isShowing()) {
				final String contextResourcePath =
						mDialogExtras.getString(MetaService.EXTRA_RESOURCE_PATH);
				if (lastResourcePath != null &&
						lastResourcePath.equals(contextResourcePath)) {
					mCheckDirectorySizeDialog.dismiss();
					removeDialog(DIALOG_CHECK_DIRECTORY_SIZE_ID);
					// Calculate the size of FILE entries from the directory.
					final long size =
						MetaUtilities.getDirectorySize(contextResourcePath, false);
					mDialogExtras.putLong(MetaService.EXTRA_SIZE, size);
					// Show directory dowload dialog.
					showDialog(DIALOG_DOWNLOAD_ID);
				}
			}
			break;
		case Status.ERROR:
			if (mCheckDirectorySizeDialog != null
					&& mCheckDirectorySizeDialog.isShowing()) {
				mCheckDirectorySizeDialog.dismiss();
				removeDialog(DIALOG_CHECK_DIRECTORY_SIZE_ID);
			}
			
			final String errorMessage = resultData
					.getString(MetaService.EXTRA_ERROR);
			// XXX This does the job, but in a terrible way. Needs more love.
			if (errorMessage == null || errorMessage.equals("auth failed")
					|| errorMessage.toLowerCase().contains("unauthorized")) {
				signIn();
			} else {
				runOnUiThread(new Runnable() {
					public void run() {
						hideSpinner();
						resetListEmptyTextView(lastResourcePath);
						final String msg = String.format(
								getString(R.string.error_fmt), errorMessage);
						UIUtil.showToast(FilesActivity.this, msg, true);
					}
				});
			}
			
			break;
		default:
			break;
		}
	}
	
	@Override
	public boolean onKeyDown(int keyCode, KeyEvent event) {
		// Was the Back button pressed?
		if (keyCode == KeyEvent.KEYCODE_BACK && event.getRepeatCount() == 0) {
			if (!mPathTracker.isAtRoot()) {
				mAdapter.cd(this);
				getListView().setSelectionFromTop(mTopPosition, mTop);
				mTopPosition = mTop = 0;
				MetaService.disallowPrefetching();
				return true;
			}
		}
		return super.onKeyDown(keyCode, event);
	}
	
	private OnQuickActionClickListener mUploadQuickActionListener =
		new OnQuickActionClickListener() {
	
		public void onQuickActionClicked(QuickActionWidget widget, int position) {
			switch (position) {
			case QuickAction.UPLOAD_IMAGE:
				onUploadImageClicked();
				break;
			case QuickAction.UPLOAD_VIDEO:
				onUploadVideoClicked();
				break;
			case QuickAction.UPLOAD_AUDIO:
				onUploadAudioClicked();
				break;
			case QuickAction.UPLOAD_FILE:
				onUploadFileClicked();
				break;
			case QuickAction.NEW_FOLDER:
				onNewFolderClicked();
				break;
				
			default:
				Log.e(TAG, "unknown upload quick action");
				break;
			}
		}
		
	};
	
	private void onUploadImageClicked() {
		onUpload("image/*", R.string.dialog_pick_image_title, false);
	}
	
	private void onUploadVideoClicked() {
		onUpload("video/*", R.string.dialog_pick_video_title, false);
	}
	
	private void onUploadAudioClicked() {
		onUpload("audio/*", R.string.dialog_pick_audio_title, false);
	}
	
	private void onUploadFileClicked() {
		onUpload("*/*", R.string.dialog_pick_file_title, true);
	}
	
	private void onNewFolderClicked() {
		showDialog(DIALOG_NEW_DIRECTORY_ID);
	}
	
	private void onUpload(final String type, final int titleResId,
			boolean openable) {
		final Intent intent = new Intent(Intent.ACTION_GET_CONTENT);
		intent.setType(type);
		if (openable) {
			intent.addCategory(Intent.CATEGORY_OPENABLE);
		}
		final Intent chooser = Intent.createChooser(intent,
				getString(titleResId));
		startActivityForResult(chooser, REQUEST_ADD_FILE);
	}
	
	@Override
	protected void onActivityResult(int requestCode, int resultCode, Intent data) {
		super.onActivityResult(requestCode, resultCode, data);

		switch (requestCode) {
		case REQUEST_SIGN_IN:
			if (resultCode == RESULT_OK) {
				onRefresh();
				suggestAutoUploadConfiguration();
				ChangeLogUtils.suggestShowingChangelog(this);
			} else if (resultCode != RESULT_OK){
				finish();
			}
			break;
			
		case REQUEST_PREFERENCES:
			if (resultCode == PreferencesActivity.RESULT_UNLINKED) {
				finish();
			}
			break;
		
		case REQUEST_ADD_FILE:
			if (resultCode == RESULT_OK && data != null) {
				final Uri uri = data.getData();
				String parentResourcePath = mPathTracker.getCurrentNode();
				if (parentResourcePath == null) {
					parentResourcePath = Preferences.U1_RESOURCE;
				}
				Log.d(TAG, "will upload " + uri + " to " + parentResourcePath);
				UpDownServiceHelper.upload(this, uri, parentResourcePath, false);
			} else {
				Log.i(TAG, "no file selected");
				UIUtil.showToast(this, R.string.files_activity_no_files);
			}			
			break;
			
		default:
			break;
		}
	}
	
	private OnQuickActionClickListener mOnDirQuickActionsClickListener =
		new OnQuickActionClickListener() {
	
		public void onQuickActionClicked(QuickActionWidget widget, int position) {
			switch (position) {
			case QuickAction.DIRECTORY_DOWNLOAD:
				onDirectoryDownloadClicked(mContextViewHolder);
				break;
			case QuickAction.DIRECTORY_RENAME:
				onDirectoryRenameClicked(mContextViewHolder);
				break;
			case QuickAction.DIRECTORY_DELETE:
				onDirectoryDeleteClicked(mContextViewHolder);
				break;
			
			default:
				Log.w(TAG, "unknown directory/volume quick action");
				break;
			}
		}
		
	};
	
	private void onDirectoryDownloadClicked(final ViewHolder holder) {
		final Bundle extras = mDialogExtras = new Bundle();
		extras.putString(MetaService.EXTRA_RESOURCE_PATH, holder.resourcePath);
		extras.putString(MetaService.EXTRA_NAME, holder.filename);
		// TODO karni: make recursive download an option
		showDialog(DIALOG_CHECK_DIRECTORY_SIZE_ID);
		MetaServiceHelper.refreshNode(this, mReceiver, holder.resourcePath);
	}
	
	private void onDirectoryRenameClicked(final ViewHolder holder) {
		final String path = MetaUtilities.getStringField(
				holder.resourcePath, Nodes.NODE_PATH);
		final Bundle extras = mDialogExtras = new Bundle();
		extras.putString(Nodes.NODE_RESOURCE_PATH, holder.resourcePath);
		extras.putString(Nodes.NODE_PATH, path);
		extras.putString(Nodes.NODE_NAME, holder.filename);
		showDialog(DIALOG_RENAME_ID);
	}
	
	private void onDirectoryDeleteClicked(final ViewHolder holder) {
		final Bundle extras = mDialogExtras = new Bundle();
		extras.putString(MetaService.EXTRA_RESOURCE_PATH, holder.resourcePath);
		extras.putString(MetaService.EXTRA_NAME, holder.filename);
		showDialog(DIALOG_DELETE_ID);
	}
	
	private OnQuickActionClickListener mOnFileQuickActionsListener =
			new OnQuickActionClickListener() {
		
		public void onQuickActionClicked(QuickActionWidget widget, int position) {
			switch (position) {
			case QuickAction.FILE_RENAME:
				onFileRenameClicked(mContextViewHolder);
				break;
			case QuickAction.FILE_DELETE:
				onFileDeleteClicked(mContextViewHolder);
				break;
			case QuickAction.FILE_SHARE:
				onFileShareClicked(mContextViewHolder);
				break;
			case QuickAction.FILE_PUBLISH:
				onFilePublishClicked(mContextViewHolder);
				break;
			case QuickAction.FILE_SHARE_URL:
				onFileShareUrlClicked(mContextViewHolder);
				break;
			case QuickAction.FILE_COPY_URL:
				onFileCopyUrlClicked(mContextViewHolder);
				break;
			default:
				Log.w(TAG, "unknown file quick action");
				break;
			}
		}
		
	};
	
	private void onFileRenameClicked(final ViewHolder holder) {
		final String path = MetaUtilities.getStringField(
				holder.resourcePath, Nodes.NODE_PATH);
		final Bundle extras = mDialogExtras = new Bundle();
		extras.putString(Nodes.NODE_RESOURCE_PATH, holder.resourcePath);
		extras.putString(Nodes.NODE_PATH, path);
		extras.putString(Nodes.NODE_NAME, holder.filename);
		showDialog(DIALOG_RENAME_ID);
	}
	
	private void onFileDeleteClicked(final ViewHolder holder) {
		final Bundle extras = mDialogExtras = new Bundle();
		extras.putString(MetaService.EXTRA_RESOURCE_PATH, holder.resourcePath);
		extras.putString(MetaService.EXTRA_NAME, holder.filename);
		showDialog(DIALOG_DELETE_ID);
	}
	
	private void onFileShareClicked(final ViewHolder holder) {
		final String data = MetaUtilities.getStringField(
				holder.resourcePath, Nodes.NODE_DATA);
		
		if (data != null) {
			try {
				final Intent intent = new Intent(Intent.ACTION_SEND);
				final String type = FileUtilities.getMime(holder.filename);
				final Uri uri = Uri.parse(Uri.encode(data, ":/"));
				Log.d(TAG, "sharing file, data: " + data + ", type: " + type);
				intent.setType(type);
				intent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION); 
				intent.putExtra(Intent.EXTRA_STREAM, uri);
				startActivity(Intent.createChooser(intent,
						getString(R.string.dialog_share_file_title)));
			} catch (ActivityNotFoundException e) {
				UIUtil.showToast(this, R.string.toast_cant_handle_file_type);
			}
		} else {
			// TODO karni: implement this as a dialog
			// 1) download first 2) share URL instead
			UIUtil.showToast(this, R.string.toast_download_to_share, true);
		}
	}
	
	private void onFilePublishClicked(final ViewHolder holder) {
		MetaServiceHelper.changePublicAccess(
				this, mReceiver, holder.resourcePath, !holder.isPublic);
	}
	
	private void onFileShareUrlClicked(final ViewHolder holder) {
		String url = MetaUtilities.getPublicUrl(holder.resourcePath);
		Intent intent = new Intent(Intent.ACTION_SEND);
		intent.setType("text/plain");
		final String sharedFrom = getString(R.string.shared_from_u1f);
		intent.putExtra(Intent.EXTRA_SUBJECT, sharedFrom);
		intent.putExtra(Intent.EXTRA_TEXT, url);
		try {
			startActivity(Intent.createChooser(intent,
					getString(R.string.dialog_share_url_title)));
		} catch (ActivityNotFoundException e) {
			UIUtil.showToast(this, R.string.toast_no_suitable_activity);
		}
	}
	
	private void onFileCopyUrlClicked(final ViewHolder holder) {
		final String publicUrl =
				MetaUtilities.getPublicUrl(holder.resourcePath);
		UIUtil.save2Clipboard(this, publicUrl);
		UIUtil.showToast(this, R.string.toast_file_url_copied);
	}
	
	private QuickActionWidget buildDirectoryContextMenu(final ViewHolder holder) {
		if (mDirQuickActions != null) {
			return mDirQuickActions;
		}
		mDirQuickActions = new QuickActionGrid(this);
		mDirQuickActions.addQuickAction(new BlackQuickAction(this,
				R.drawable.ic_act_download_folder,
				R.string.context_download));
		mDirQuickActions.addQuickAction(new BlackQuickAction(this,
				R.drawable.gd_action_bar_edit,
				R.string.context_rename));
		mDirQuickActions.addQuickAction(new BlackQuickAction(this,
				R.drawable.gd_action_bar_trashcan,
				R.string.context_delete));
		mDirQuickActions.setOnQuickActionClickListener(
				mOnDirQuickActionsClickListener);
		return mDirQuickActions;
	}
	
	private QuickActionWidget buildFileContextMenu(final ViewHolder holder) {
		// TODO karni: consider caching pub/unpub quick action grid 
		mFileQuickActions = new QuickActionGrid(this);
		mFileQuickActions.addQuickAction(new BlackQuickAction(this,
				R.drawable.gd_action_bar_edit,
				R.string.context_rename));
		mFileQuickActions.addQuickAction(new BlackQuickAction(this,
				R.drawable.gd_action_bar_trashcan,
				R.string.context_delete));
		mFileQuickActions.addQuickAction(new BlackQuickAction(this,
				R.drawable.ic_act_share_file,
				R.string.context_share_file));
		
		if (holder.isPublic) {
			mFileQuickActions.addQuickAction(new BlackQuickAction(this,
					R.drawable.ic_act_unpublish_file,
					R.string.context_unpublish));
			mFileQuickActions.addQuickAction(new BlackQuickAction(this,
					R.drawable.ic_act_share_url,
					R.string.context_share_url));
			mFileQuickActions.addQuickAction(new BlackQuickAction(this,
					R.drawable.ic_act_copy_url,
					R.string.context_copy_url));
		} else {
			mFileQuickActions.addQuickAction(new BlackQuickAction(this,
					R.drawable.ic_act_publish_file,
					R.string.context_publish));
		}
		mFileQuickActions.setOnQuickActionClickListener(
				mOnFileQuickActionsListener);
		return mFileQuickActions;
	}
	
	@Override
	public void onCreateContextMenu(ContextMenu menu, View v,
			ContextMenuInfo menuInfo) {
		// Use this method as a hook, but don't show the default context menu.
		closeContextMenu();
		
		final AdapterContextMenuInfo info = (AdapterContextMenuInfo) menuInfo;
		final ViewHolder holder = mContextViewHolder =
				(ViewHolder) info.targetView.getTag();
		mDialogExtras = null;
		
		final boolean isVolume = holder.parentResourcePath == null;
		final String type = holder.kind;
		
		QuickActionWidget quickActionWidget = null;
		if (isVolume) {
			// This is a volume, we don't show the context menu yet.
			return;
		} else if (NodeInfo.DIRECTORY.equals(type)) {
			quickActionWidget = buildDirectoryContextMenu(holder);
		} else if (NodeInfo.FILE.equals(type)) {
			quickActionWidget = buildFileContextMenu(holder);
		}
		quickActionWidget.show(info.targetView);
	}
	
	@Override
	protected void onListItemClick(ListView l, View v, int position, long id) {
		Log.d(TAG, "onListItemClick() " + id);
		MetaService.disallowPrefetching();
		
		final ViewHolder holder = (ViewHolder) v.getTag();		
		if (MetaUtilities.isDirectory(id)) {
			// This should avoid delayed update of the list-empty label.
			if (NetworkReceiver.isConnected()) {
				awaitWithListEmptyTextView();
			}
			
			// Save the position of the list, just for the previous screen.
			mTopPosition = l.getFirstVisiblePosition();
			View topView = l.getChildAt(0);
			mTop = (topView == null) ? 0 : topView.getTop();
			// Change to selected directory.
			mAdapter.cd(this, holder.resourcePath);
			// Scroll the list to top when entering a directory.
			getListView().setSelectionFromTop(0, 0);
			// Start prefetching.
			prefetchCurrent();
		} else {
			// Open selected file.
			if (holder.data != null) {
				onFileClicked(holder.resourcePath, holder.resourceState,
						holder.filename, holder.data);
			} else {
				downloadFile(holder.resourcePath);
			}
		}
	}

	private void showSpinner() {
		mHandler.post(new Runnable() {
			public void run() {
				if (mLoaderItem != null)
					mLoaderItem.setLoading(true);
			}
		});
	}
	
	private void hideSpinner() {
		mHandler.postDelayed(new Runnable() {
			public void run() {
				if (mLoaderItem != null && !MetaService.sIsWorking) {
					mLoaderItem.setLoading(false);
				}
			}
		}, 500);
	}
	
	private void awaitWithListEmptyTextView() {
		mHandler.post(new Runnable() {
			public void run() {
				Log.d(TAG, "Setting empty list view loading label.");
				if (mEmptyTextView != null) {
					mEmptyTextView.setText(R.string.files_activity_loading_files);
				}
			}
		});
	}
	
	private void resetListEmptyTextView(final String resourcePath) {
		mHandler.postDelayed(new Runnable() {
			public void run() {
				// Don't set the dir_is_empty label if MetaService still processing.
				if (!MetaService.sIsWorking) {
					if (resourcePath == null ||
							resourcePath.equals(mPathTracker.getCurrentNode())) {
						mEmptyTextView.setText(R.string.files_activity_directory_is_empty);
					}
				}
			}
		}, 1500);
	}
	
	public void downloadFile(final String resourcePath) {
		if (!UpDownServiceHelper.isDownloadPending(resourcePath)) {
			Log.d(TAG, "download not pending, scheduling");
			UpDownServiceHelper.download(this, resourcePath, false);
		} else {
			Log.d(TAG, "download pending, not scheduling");
		}
	}
	
	private void onFileClicked(final String resourcePath,
			final String resourceState, final String filename,
			final String data) {
		Log.d(TAG, "onFileClicked() data=" + data);
		
		final boolean isStateIdle = (resourceState == null);
		boolean isUploading, isFailedUpload;
		boolean isDownloading, isFailedDownload;
		isUploading = isFailedUpload = isDownloading = isFailedDownload = false;
		if (!isStateIdle) {
			isUploading =
					ResourceState.STATE_POSTING.equals(resourceState);
			isFailedUpload =
					ResourceState.STATE_POSTING_FAILED.equals(resourceState);
			isDownloading =
					ResourceState.STATE_GETTING.equals(resourceState);
			isFailedDownload =
					ResourceState.STATE_GETTING_FAILED.equals(resourceState);
		}
		
		final boolean validTarget = MetaUtilities.isValidUriTarget(data)
				&& !isUploading;
		
		Log.d(TAG, String.format("isStateIdle=%s, validTarget=%s",
				String.valueOf(isStateIdle), String.valueOf(validTarget)));
		if (validTarget && isStateIdle) {
			// File exists and is not being downloaded.
			if (data.startsWith(ContentResolver.SCHEME_CONTENT)) {
				// Open file from external content provider.
				Log.d(TAG, "opening file from content uri");
				
				// Validate the uri, the file can be already gone.
				if (!MetaUtilities.isValidUriTarget(data)) {
					final String text = String.format(
							getString(R.string.toast_downloading_x), filename);
					UIUtil.showToast(this, text);
					downloadFile(resourcePath);
				} else {
					final Intent intent = new Intent(Intent.ACTION_VIEW);
					final Uri uri = Uri.parse(data);
					intent.setDataAndType(uri, FileUtilities.getMime(filename));
					
					final Intent chooser = new Intent(Intent.ACTION_VIEW);
					chooser.putExtra(Intent.EXTRA_INTENT, intent);
					chooser.putExtra(Intent.EXTRA_TITLE,
							getText(R.string.dialog_open_with_title));
					try {
						startActivity(intent);
					} catch (ActivityNotFoundException e) {
						UIUtil.showToast(this, R.string.toast_no_suitable_activity);
					}
				}
			} else {
				Log.d(TAG, "opening file directly");
				// Open the file directly.
				File file = null;
				try {
					Log.d(TAG, "opening " + data);
					String newData = data;
					if (data == null) {
						throw new Exception("uri is empty!");
					}
					file = new File(URI.create(Uri.encode(newData, ":/")));
				} catch (URISyntaxException e) {
					Log.e(TAG, "error while opening file", e);
				} catch (Exception e) {
					Log.e(TAG, "file uri is empty", e);
				}
				if (file != null && file.exists()) {
					Intent intent = new Intent(Intent.ACTION_VIEW);
					intent.setDataAndType(Uri.fromFile(file),
							FileUtilities.getMime(data));
					
					Intent chooser = new Intent(Intent.ACTION_VIEW);
					chooser.putExtra(Intent.EXTRA_INTENT, intent);
					chooser.putExtra(Intent.EXTRA_TITLE,
							getText(R.string.dialog_open_with_title));
					try {
						startActivity(intent);
					} catch (ActivityNotFoundException e) {
						UIUtil.showToast(this, R.string.toast_no_suitable_activity);
					}
				} else {
					MetaUtilities.setIsCached(resourcePath, false);
					MetaUtilities.setStateAndData(resourcePath, "", null);
					UIUtil.showToast(this, R.string.toast_file_not_cached_anymore, false);
					mResolver.notifyChange(Nodes.CONTENT_URI, null);
				}
			}
		} else if (isFailedDownload) {
			// File does not exist or download failed.
			Log.d(TAG, "was: failed download or invalid");
			if (NetworkReceiver.isConnected()) {
				downloadFile(resourcePath);
			} else {
				UIUtil.showToast(this, R.string.toast_no_network);
			}
		} else if (isFailedUpload && validTarget) {
			// File exists and upload failed.
			Log.d(TAG, "was: failed upload");
			if (NetworkReceiver.isConnected()) {
				// For faster UI feedback.
				MetaUtilities.setState(resourcePath, ResourceState.STATE_POSTING);
				MetaUtilities.notifyChange(Nodes.CONTENT_URI);
				
				if (!UpDownServiceHelper.isUploadPending(resourcePath)) {
					final String currentNode = mPathTracker.getCurrentNode();
					final String parent = (currentNode == null)
							? Preferences.U1_RESOURCE : currentNode;
					UpDownServiceHelper.upload(this,
							Uri.parse(Uri.encode(data, ":/")), parent, false);
				}
			} else {
				UIUtil.showToast(this, R.string.toast_no_network);
			}
		} else if (isFailedUpload && !validTarget) {
			// No uri? Clean-up.
			MetaUtilities.deleteByResourcePath(resourcePath);
			MetaUtilities.notifyChange(Nodes.CONTENT_URI);
		} else if (isStateIdle && !validTarget) {
			// File removed from device, need to download.
			downloadFile(resourcePath);
		} else if (isUploading) {
			UIUtil.showToast(this, "Please wait while uploading...");
		} else if (isDownloading) {
			UIUtil.showToast(this, "Please wait while downloading...");
		} else {
			Log.e(TAG, "unhandled state: " + resourceState);
		}
	}
	
	public static class ViewHolder {
		public ImageView icon;
		public TextView itemName;
		public TextView itemTimestamp; // volume created, file changed
		public TextView itemInteger; // fileSize or "..." for non-empty dirs
		// For caching query results:
		public String parentResourcePath;
		public String resourcePath;
		public String resourceState;
		public String kind;
		public String filename;
		public boolean isPublic;
		public String data;
	}
	
	private Cursor getFilesCurosor(final String resourcePath) {
		Cursor filesCursor = null;
		if (resourcePath != null) {
			filesCursor = MetaUtilities
					.getVisibleNodesCursorByParent(resourcePath);
			startManagingCursor(filesCursor);
		}
		return filesCursor;
	}
	
	private class FilesAdapter extends CursorAdapter {
		
		private LayoutInflater mInflater;
		
		private final String CLOUD_FOLDER = getString(R.string.node_is_cloud_folder);
		
		private final static String PURCHASED_FROM_U1 =
				"Purchased from Ubuntu One";
		private final String PURCHASED_MUSIC =
				getString(R.string.node_is_purchased_music);
		
		public FilesAdapter(Context context, Cursor c, boolean autoRequery) {
			super(context, c, autoRequery);
			mInflater = LayoutInflater.from(context);
		}
		
		@Override
		protected void onContentChanged() {
			super.onContentChanged();
			if (getListAdapter().getCount() == 0) {
				resetListEmptyTextView(mPathTracker.getCurrentNode());
			} else {
				awaitWithListEmptyTextView();
			}
		}

		@Override
		public View getView(int position, View convertView, ViewGroup parent) {
			/* Fixes crash report:
			 * 
			 * java.lang.IllegalStateException: couldn't move cursor to position 9
			 * at android.widget.CursorAdapter.getView(CursorAdapter.java:178)
			 */
			try {
				return super.getView(position, convertView, parent);
			} catch (IllegalStateException e) {
				return super.getView(0, convertView, parent);
			}
		}
		
		@Override
		public View newView(Context context, Cursor cursor, ViewGroup parent) {
			final View view = mInflater.inflate(R.layout.list_file_row, null);
			
			final ViewHolder holder = new ViewHolder();
			holder.icon =
					(ImageView) view.findViewById(R.id.icon);
			holder.itemName =
					(TextView) view.findViewById(R.id.item_name);
			holder.itemTimestamp =
					(TextView) view.findViewById(R.id.last_modified);
			holder.itemInteger =
					(TextView) view.findViewById(R.id.size);
			
			view.setTag(holder);
			bindView(view, context, cursor);
			return view;
		}

		@Override
		public void bindView(View view, final Context context, Cursor cursor) {
			view.setBackgroundResource(R.color.list_item_bg_local);
			
			final ViewHolder holder = (ViewHolder) view.getTag();
			holder.parentResourcePath = cursor.getString(
					cursor.getColumnIndex(Nodes.NODE_PARENT_PATH));
			holder.resourcePath = cursor.getString(
					cursor.getColumnIndex(Nodes.NODE_RESOURCE_PATH));
			holder.resourceState = cursor.getString(
					cursor.getColumnIndex(Nodes.NODE_RESOURCE_STATE));
			holder.data = cursor.getString(
					cursor.getColumnIndex(Nodes.NODE_DATA));
			
			final String url = cursor.getString(cursor
					.getColumnIndex(Nodes.NODE_PUBLIC_URL));
			final boolean isPublic = !(url == null || "".equals(url));
			holder.isPublic = isPublic;
			int fileNameColor = isPublic
					? context.getResources().getColor(R.color.text_blue)
					: context.getResources().getColor(R.color.text_red);
			holder.itemName.setTextColor(fileNameColor);
			
			final boolean isVolume = holder.parentResourcePath == null;
			final boolean isDirectory = MetaUtilities.isDirectory(cursor);
			if (isDirectory) {
				holder.kind = Nodes.KIND_DIRECTORY;
				holder.icon.setImageResource(R.drawable.ic_folder);
			} else {
				holder.kind = Nodes.KIND_FILE;
				String mime =
						cursor.getString(cursor.getColumnIndex(Nodes.NODE_MIME));
				if (mime == null)
					holder.icon.setImageResource(R.drawable.ic_file);
				else {
					if (!isPublic) {
						if (mime.startsWith(FileUtilities.MIME_IMAGE))
							holder.icon.setImageResource(R.drawable.ic_photo);
						else if (mime.startsWith(FileUtilities.MIME_VIDEO))
							holder.icon.setImageResource(R.drawable.ic_video);
						else if (mime.startsWith(FileUtilities.MIME_AUDIO))
							holder.icon.setImageResource(R.drawable.ic_audio);
						else
							holder.icon.setImageResource(R.drawable.ic_file);
					} else {
						if (mime.startsWith(FileUtilities.MIME_IMAGE))
							holder.icon.setImageResource(R.drawable.ic_photo_published);
						else if (mime.startsWith(FileUtilities.MIME_VIDEO))
							holder.icon.setImageResource(R.drawable.ic_video_published);
						else if (mime.startsWith(FileUtilities.MIME_AUDIO))
							holder.icon.setImageResource(R.drawable.ic_audio_published);
						else
							holder.icon.setImageResource(R.drawable.ic_file_published);
					}
				}
			}
			
			final String filename = cursor.getString(cursor
					.getColumnIndex(Nodes.NODE_NAME));
			holder.filename = filename;
			if (isVolume) {
				if (filename.equals(PURCHASED_FROM_U1))
					holder.itemName.setText(PURCHASED_MUSIC);
				else
					holder.itemName.setText(holder.resourcePath.substring(3));
			} else {
				holder.itemName.setText(filename);
			}
			
			holder.itemInteger.setText("");
			if (!isDirectory) {
				final long size = cursor.getLong(cursor
						.getColumnIndex(Nodes.NODE_SIZE));
				final String sizeText = FileUtilities
						.getHumanReadableSize(size);
				holder.itemInteger.setText(sizeText);
			} else {
				final boolean hasChildren = cursor.getInt(cursor
						.getColumnIndex(Nodes.NODE_HAS_CHILDREN)) == 1;
				if (hasChildren) {
					holder.itemInteger.setText("...");
				}
			}
			
			// TODO karni: add state feedback in UI
			final String state = cursor.getString(
					cursor.getColumnIndex(Nodes.NODE_RESOURCE_STATE));
			if (!isDirectory && (holder.data == null)) {
				view.setBackgroundResource(R.color.list_item_bg_remote);
			}

			/*
			 * I have reformatted the code to support viewing last modified
			 * under directories as well, but untill we have deltas, it's not
			 * so trivial (if anything has changed 1 level <i>down</i>, we
			 * should update parent directory WHEN_CHANGE timestamp).
			 * Prefetching helps, but without prefetching - it may confuse users.
			 */
			
			if (isVolume) {
				holder.itemTimestamp.setVisibility(View.VISIBLE);
				holder.itemTimestamp.setText(CLOUD_FOLDER);
			} else {
				if (isDirectory) {
					holder.itemTimestamp.setVisibility(View.INVISIBLE);
				} else if (state == null) {
					holder.itemTimestamp.setVisibility(View.VISIBLE);
					// Display last modification time.
					final long modified = cursor.getLong(
							cursor.getColumnIndex(Nodes.NODE_WHEN_CHANGED));
					if (modified < 1000) {
						// Attempt to update the UI to quickly. This will be fixed
						// by updating cache directly from upload response NodeInfo.
						holder.itemTimestamp.setText("");
					} else {
						final String modifiedText = String.format(
								getString(R.string.node_last_modified),
								DateUtilities.getFriendlyDate(context, modified));
						holder.itemTimestamp.setText(modifiedText);
					}
				} else {
					holder.itemTimestamp.setVisibility(View.VISIBLE);
					// Provide state feedback.
					if (ResourceState.STATE_GETTING.equals(state)) {
						holder.itemTimestamp.setText(R.string.node_is_downloading);
					} else if (ResourceState.STATE_POSTING.equals(state)) {
						holder.itemTimestamp.setText(R.string.node_is_uploading);
					} else if (ResourceState.STATE_DELETING.equals(state)) {
						holder.itemTimestamp.setText(R.string.node_is_deleting);
					} else if (ResourceState.STATE_GETTING_FAILED.equals(state)) {
						holder.itemTimestamp.setText(R.string.node_download_has_failed);
					} else if (ResourceState.STATE_POSTING_FAILED.equals(state)) {
						holder.itemTimestamp.setText(R.string.node_upload_has_failed);
					}
				}
			}
		}
		
		/**
		 * cd ..
		 */
		public boolean cd(final Activity activity) {
			if (mPathTracker.isAtRoot()) {
				return false;
			}
			mPathTracker.cd();
			if (mPathTracker.isAtRoot()) {
				final Cursor filesCursor = MetaUtilities
						.getVisibleTopNodesCursor();
				Log.v(TAG, "changing cursor");
				changeCursor(filesCursor);
			} else {
				final String node = mPathTracker.getCurrentNode();
				final Cursor filesCursor = getFilesCurosor(node);
				Log.v(TAG, "changing cursor");
				changeCursor(filesCursor);
			}
			return true;
		}
		
		/**
		 * cd into dir
		 */
		public void cd(final Activity activity, final String resourcePath) {
			mPathTracker.cd(resourcePath);
			final Cursor filesCursor = getFilesCurosor(resourcePath);
			changeCursor(filesCursor);
			notifyDataSetInvalidated();
			
			onRefresh(resourcePath);
		}
		
	}
	
	public static void showFrom(Activity activity) {
		final Intent intent = new Intent(activity, FilesActivity.class);
		activity.startActivity(intent);
		activity.overridePendingTransition(
				android.R.anim.fade_in, android.R.anim.fade_out);
	}
}
