/*
 * Ubuntu One Files - access Ubuntu One cloud storage on Android platform.
 * 
 * Copyright (C) 2011 Canonical Ltd.
 * Author: Michał Karnicki <michal.karnicki@canonical.com>
 *   
 * This file is part of Ubuntu One Files.
 *  
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *  
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses 
 */

package com.ubuntuone.android.files.activity;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.InputStreamReader;

import android.accounts.AccountManager;
import android.app.AlarmManager;
import android.app.AlertDialog;
import android.app.Dialog;
import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.Environment;
import android.preference.CheckBoxPreference;
import android.preference.ListPreference;
import android.preference.Preference;
import android.preference.PreferenceActivity;
import android.preference.PreferenceScreen;
import android.preference.Preference.OnPreferenceChangeListener;
import android.preference.Preference.OnPreferenceClickListener;
import android.view.View;
import android.view.Window;

import com.google.android.apps.analytics.GoogleAnalyticsTracker;
import com.ubuntuone.android.files.Alarms;
import com.ubuntuone.android.files.Analytics;
import com.ubuntuone.android.files.Constants;
import com.ubuntuone.android.files.Preferences;
import com.ubuntuone.android.files.R;
import com.ubuntuone.android.files.UbuntuOneFiles;
import com.ubuntuone.android.files.provider.MetaUtilities;
import com.ubuntuone.android.files.receiver.NetworkReceiver;
import com.ubuntuone.android.files.service.MediaCatcher;
import com.ubuntuone.android.files.service.MetaService;
import com.ubuntuone.android.files.service.MetaServiceHelper;
import com.ubuntuone.android.files.service.UpDownService;
import com.ubuntuone.android.files.util.BrowserUtilities;
import com.ubuntuone.android.files.util.ChangeLogUtils;
import com.ubuntuone.android.files.util.ConfigUtilities;
import com.ubuntuone.android.files.util.DateUtilities;
import com.ubuntuone.android.files.util.FileUtilities;
import com.ubuntuone.android.files.util.Log;
import com.ubuntuone.android.files.util.MediaUtilities;
import com.ubuntuone.android.files.util.StorageInfo;
import com.ubuntuone.android.files.util.UIUtil;
import com.ubuntuone.android.files.util.StorageInfo.StorageNotAvailable;

/**
 * Activity to manipulate application preferences. {@link Preference} keys can
 * be found as public fields of {@link Preferences} class. The {@link Pref}
 * interface contains keys of preferences we need a handle for, but don't use
 * them to store actual values, just reference the according {@link View}s.
 */
public final class PreferencesActivity extends PreferenceActivity {
	private static final String TAG = PreferencesActivity.class.getSimpleName();
	
	public static final int RESULT_UNLINKED = 1;
	
	/**
	 * {@link Preference} keys to retrieve views, we're not using them to
	 * manipulate {@link Preference} values.
	 */
	private static interface Pref {
		public static final String PURCHASE_STORAGE = "purchase_storage";
		public static final String UPGRADE_PLAN = "upgrade_plan";
		
		public static final String MEDIA_UPLOAD = "media_upload";
		public static final String FORCE_MEDIA_UPLOAD = "force_media_upload";
		public static final String ADVANCED_OPTIONS = "advanced_options";
		public static final String REMOVE_DEVICE = "remove_device";
		
		public static final String RETRY_FAILED = "retry_failed";
		public static final String CANCEL_FAILED = "cancel_failed";
		
		public static final String MANAGE_ACCOUNT = "manage_online";
		public static final String SUPPORT_OPTIONS = "support_options";
		
		public static final String FEEDBACK = "feedback";
		public static final String REPORT_PROBLEM = "report_problem";
		public static final String COLLECT_LOGS = "collect_logs";
		public static final String REVIEW_LOGS = "review_logs";
		public static final String SEND_LOGS = "send_logs";
		
		public static final String ABOUT = "about";
		public static final String LICENSE = "license";
		public static final String CHANGELOG ="changelog";
		public static final String AUTHORS = "authors";
		public static final String WEBPAGE = "webpage";
		public static final String GREENDROID = "greendroid";
	}
	
	/** URLs to couple of useful pages. */
	private static interface Url {
		public static final String UPGRADE =
				"https://one.ubuntu.com/plans/";
		public static final String MANAGE =
				"https://one.ubuntu.com/account/";
		public static final String SUPPORT =
				"https://one.ubuntu.com/support/";
		public static final String FEEDBACK =
				"https://answers.edge.launchpad.net/ubuntuone-android-files/+faqs";
		public static final String LICENSE =
				"http://www.gnu.org/licenses/agpl.html";
		public static final String AUTHORS =
				"https://launchpad.net/~ubuntuone-android-hackers";
		public static final String WEBPAGE =
				"https://launchpad.net/ubuntuone-android-files";
		public static final String GREENDROID =
				"https://github.com/cyrilmottier/GreenDroid";
	}
	
	private GoogleAnalyticsTracker mTracker;
	
	private Context mContext;
	
	private Preference mUploadPhotosDir;
	private ListPreference mUploadFrequency;
	private Preference mRetryFailed;

	private Preference mCancelFailed;
	private CheckBoxPreference mCollectLogs;
	private Preference mReviewLogs;
	private Preference mSendLogs;
	
	@Override
	protected void onCreate(Bundle savedInstanceState) {
		mContext = this;
		
		mTracker = GoogleAnalyticsTracker.getInstance();
		mTracker.start(Analytics.U1F_ACCOUNT, this);
		mTracker.trackPageView(TAG);

		getWindow().requestFeature(Window.FEATURE_NO_TITLE);
		super.onCreate(savedInstanceState);
		addPreferencesFromResource(R.xml.preferences);
		
		// Account category. 
		
		final Preference username = findPreference(Preferences.USERNAME_KEY);
		
		final String usernameSummary =
				Preferences.getString(Preferences.USERNAME_KEY, "(?)");
		username.setSummary(usernameSummary);
		
		final Preference purchaseStorage = findPreference(Pref.PURCHASE_STORAGE);
		purchaseStorage.setOnPreferenceClickListener(mPurchaseStorageListener);
		
		final Preference upgradePlan = findPreference(Pref.UPGRADE_PLAN);
		upgradePlan.setOnPreferenceClickListener(mLinkListener);
		
		// Options category.
		
		final PreferenceScreen mediaUploadScreen = 
				(PreferenceScreen) findPreference(Pref.MEDIA_UPLOAD);
		mediaUploadScreen.setOnPreferenceClickListener(whiteHackClick);
		
		final CheckBoxPreference uploadPhotos =
				(CheckBoxPreference) findPreference(Preferences.UPLOAD_PHOTOS_KEY);
		uploadPhotos.setOnPreferenceClickListener(mUploadPhotosListener);
		
		mUploadPhotosDir =
				(Preference) findPreference(Preferences.UPLOAD_PHOTOS_DIR_KEY);
		mUploadPhotosDir.setOnPreferenceClickListener(mUploadPhotosDirListener);
		final String photosUploadDirectory = Preferences.getPhotosUploadDirectory();
		mUploadPhotosDir.setDefaultValue(photosUploadDirectory);
		mUploadPhotosDir.setSummary(photosUploadDirectory);
		
		mUploadFrequency =
				(ListPreference) findPreference(Preferences.UPLOAD_FREQUENCY_KEY);
		mUploadFrequency.setEntryValues(new String[] {
				String.valueOf(0),
				String.valueOf(AlarmManager.INTERVAL_FIFTEEN_MINUTES),
				String.valueOf(AlarmManager.INTERVAL_HALF_HOUR),
				String.valueOf(AlarmManager.INTERVAL_HOUR),
				String.valueOf(AlarmManager.INTERVAL_HALF_DAY),
				String.valueOf(AlarmManager.INTERVAL_DAY)
		});
		mUploadFrequency.setOnPreferenceChangeListener(mUploadFrequencyListener);
		final long frequency = Preferences.getLongFromString(
				Preferences.UPLOAD_FREQUENCY_KEY,
				Preferences.DEFAULT_UPLOAD_FREQUENCY);
		mUploadFrequency.setSummary(getFrequencySummary(frequency));
		
		final Preference forceMediaUpload = findPreference(Pref.FORCE_MEDIA_UPLOAD);
		forceMediaUpload.setOnPreferenceClickListener(mForceMediaUploadListener);
		
		try {
			boolean shouldUploadImmediately =
					StorageInfo.shouldImmediatelyUploadPhotos();
			if (shouldUploadImmediately) {
				Preferences.putString(Preferences.UPLOAD_FREQUENCY_KEY,
						String.valueOf(0)); // 0 frequency is immediate
			}
		} catch (StorageInfo.StorageNotAvailable e) {
			UIUtil.showToast(this, R.string.storage_is_not_accessible_not_setting_preference);
			Preferences.putString(Preferences.UPLOAD_FREQUENCY_KEY,
					String.valueOf(Preferences.DEFAULT_UPLOAD_FREQUENCY));
		}
		
		final PreferenceScreen advancedOptionsScreen = 
			(PreferenceScreen) findPreference(Pref.ADVANCED_OPTIONS);
		advancedOptionsScreen.setOnPreferenceClickListener(whiteHackClick);
		
		final Preference autoRetry = findPreference(Preferences.AUTO_RETRY_FAILED);
		autoRetry.setOnPreferenceChangeListener(mAutoRetryListener);
		
		final CheckBoxPreference allowPrefetching =
				(CheckBoxPreference) findPreference(Preferences.ALLOW_PREFETCHING);
		allowPrefetching.setOnPreferenceChangeListener(mAllowPrefetchingListener);
		
		final Preference removeDevice = findPreference(Pref.REMOVE_DEVICE);
		removeDevice.setOnPreferenceClickListener(mRemoveDeviceListener);
		
		mRetryFailed = findPreference(Pref.RETRY_FAILED);
		mRetryFailed.setOnPreferenceClickListener(mRetryFailedListener);
		
		mCancelFailed = findPreference(Pref.CANCEL_FAILED);
		mCancelFailed.setOnPreferenceClickListener(mCancelFailedListener);
		
		// Manage category.
		
		final Preference manageAccount = findPreference(Pref.MANAGE_ACCOUNT);
		manageAccount.setOnPreferenceClickListener(mLinkListener);
		
		final Preference supportOptions = findPreference(Pref.SUPPORT_OPTIONS);
		supportOptions.setOnPreferenceClickListener(mLinkListener);
		
		final Preference feedback = findPreference(Pref.FEEDBACK);
		feedback.setOnPreferenceClickListener(mFeedbackListener);
		
		final PreferenceScreen reportScreen =
				(PreferenceScreen) findPreference(Pref.REPORT_PROBLEM);
		reportScreen.setOnPreferenceClickListener(whiteHackClick);
		
		mCollectLogs = (CheckBoxPreference) findPreference(Pref.COLLECT_LOGS);
		mCollectLogs.setOnPreferenceClickListener(mCollectLogsListener);
		
		mReviewLogs = findPreference(Pref.REVIEW_LOGS);
		mReviewLogs.setOnPreferenceClickListener(mReviewLogsListener);
		
		mSendLogs = findPreference(Pref.SEND_LOGS);
		mSendLogs.setOnPreferenceClickListener(mSendLogsListener);
		
		final PreferenceScreen about =
				(PreferenceScreen) findPreference(Pref.ABOUT);
		about.setSummary(UbuntuOneFiles.getVersion());
		about.setOnPreferenceClickListener(whiteHackClick);
		
		final Preference license = findPreference(Pref.LICENSE);
		license.setOnPreferenceClickListener(mLinkListener);
		
		final Preference changelog = findPreference(Pref.CHANGELOG);
		changelog.setOnPreferenceClickListener(mChangelogListener);
		
		final Preference author = findPreference(Pref.AUTHORS);
		author.setOnPreferenceClickListener(mLinkListener);
		
		final Preference webpage = findPreference(Pref.WEBPAGE);
		webpage.setOnPreferenceClickListener(mLinkListener);
		
		final Preference greendroid = findPreference(Pref.GREENDROID);
		greendroid.setOnPreferenceClickListener(mLinkListener);
	}
	
	@Override
	protected void onPostResume() {
		super.onPostResume();
		
		final Preference plan = findPreference(Preferences.PLAN_KEY);
		setUpCurrentPlanSummary(plan);
	}
	
	private void setUpCurrentPlanSummary(Preference plan) {
		final String currentPlan =
			Preferences.getString(Preferences.PLAN_KEY, "(?)");
		final long usedBytes =
			Preferences.getLong(Preferences.USED_BYTES_KEY, 0L);
		final long maxBytes =
				Preferences.getLong(Preferences.MAX_BYTES_KEY, 1L);
		final double usedPercentage = 100 * usedBytes / (double) maxBytes;
		
		final String spaceDetails = getString(
				R.string.preferences_activity_bytes_used_summary_fmt,
				FileUtilities.getHumanReadableSize(usedBytes),
				FileUtilities.getHumanReadableSize(maxBytes),
				usedPercentage);
		final String planSummary = currentPlan + spaceDetails;
		// TODO karni: add a nice progress bar showing used bytes
		plan.setSummary(planSummary);
	}

	protected void onResume() {
		super.onResume();
		adjustTransferPreferencesState();
	}

	@Override
	public void onDestroy() {
		mTracker.dispatch();
		mTracker.stop();
		super.onDestroy();
	}

	private OnPreferenceClickListener mUploadPhotosListener =
			new OnPreferenceClickListener() {

		public boolean onPreferenceClick(Preference preference) {
			final boolean upload = Preferences.getBoolean(
					Preferences.UPLOAD_PHOTOS_KEY,
					false);
			mTracker.trackEvent("Settings", "AutoUpload",
					"photos", upload ? 1 : 0);
			
			final long uploadFrequency = Preferences.getLongFromString(
					Preferences.UPLOAD_FREQUENCY_KEY,
					Preferences.DEFAULT_UPLOAD_FREQUENCY);
			final boolean uploadInstantly = uploadFrequency == 0;
			
			if (upload) {
				// User has enabled the auto-upload. We should be uploading
				// only the pictures taken since *this* moment.
				try {
					StorageInfo.setLastUploadedPhotoTimestamp(
							System.currentTimeMillis() / 1000);
				} catch (StorageNotAvailable eek) {
					Log.e(TAG, "auto-upload enabled, " +
							"but can't update the timestamp", eek);
				}
			}
			
			boolean uploadServiceWanted = false;
			try {
				uploadServiceWanted = uploadInstantly &&
						StorageInfo.shouldImmediatelyUploadPhotos();
			} catch (StorageInfo.StorageNotAvailable e) {
				UIUtil.showToast(mContext, R.string.toast_storage_is_not_accessible);
				Log.e(TAG, "Should upload photos?", e);
			}
			Log.d(TAG, "Toggle photo upload, and instant is " +
					uploadServiceWanted);

			if (upload) {
				if (uploadServiceWanted) {
					// Unregister inexact alarm.
					Alarms.unregisterPictureUploadAlarm();
					// Start background service.
					startUploadService();
				} else {
					// Stop background service.
					stopUploadService();
					// Register inexact alarm.
					long wakeInterval = Preferences.getLongFromString(
							Preferences.UPLOAD_FREQUENCY_KEY,
							AlarmManager.INTERVAL_HALF_HOUR);
					Alarms.registerUploadAlarm(
							Alarms.getMediaUploadPendingIntent(mContext, 0),
							DateUtilities.MILLIS_IN_MINUTE,
							Long.valueOf(wakeInterval));
				}
			} else {
				Alarms.unregisterPictureUploadAlarm();
				stopUploadService();
			}
			return true;
		}
	};
	
	private OnPreferenceClickListener mUploadPhotosDirListener =
			new OnPreferenceClickListener() {
		public boolean onPreferenceClick(Preference preference) {
			final Intent intent = new Intent(
					FilesActivity.ACTION_PICK_AUTO_UPLOAD_DIRECTORY);
			intent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP);
			startActivity(intent);
			return false;
		}
	};
	
	private void startUploadService() {
		if (MediaCatcher.startFrom(this)) {
			UIUtil.showToast(mContext, R.string.toast_started_bg_service);
		} else {
			UIUtil.showToast(mContext, R.string.toast_failed_to_start_bg_service);
			mUploadFrequency.setValue(
					String.valueOf(Preferences.DEFAULT_UPLOAD_FREQUENCY));
		}
	}
	
	private void stopUploadService() {
		if (MediaCatcher.stopFrom(this)) {
			UIUtil.showToast(mContext, R.string.toast_stopped_bg_service);
		}
	}
	
	private OnPreferenceChangeListener mUploadFrequencyListener =
			new OnPreferenceChangeListener() {

		public boolean onPreferenceChange(Preference preference, Object newValue) {
			final long frequency = Long.parseLong((String) newValue);
			mTracker.trackEvent("Settings", "AutoUpload",
					"frequency", (int) frequency);
			
			final boolean immediate = frequency == 0; 
			Log.i(TAG, "upload freq set to: " + (immediate ? "immediate" : frequency));
			
			if (immediate) {
				// Upload immediately (+ retry on failures).
				try {
					StorageInfo.setShouldImmediatelyUploadPhotos(true);
				} catch (StorageInfo.StorageNotAvailable e) {
					UIUtil.showToast(mContext,
							R.string.storage_is_not_accessible_not_setting_preference);
					Log.e(TAG, "Setting should-immediately-upload", e);
					Preferences.setShouldImmediatelyUploadPhotos(false);
				}
				// Unregister alarm.
				Log.d(TAG, "unregister upload alarm");
				Alarms.unregisterPictureUploadAlarm();
				// Start service.
				startUploadService(); // TODO karni: check the result
			} else {
				// Do not upload immediately.
				try {
					StorageInfo.setShouldImmediatelyUploadPhotos(false);
				} catch (StorageNotAvailable e) {
					UIUtil.showToast(mContext,
							R.string.storage_is_not_accessible_not_setting_preference);
					Log.e(TAG, "Setting should-immediately-upload", e);
					Preferences.setShouldImmediatelyUploadPhotos(true);
				}
				// Stop service.
				stopUploadService();
				// Register alarm.
				Log.d(TAG, "register upload alarm immediately");
				Alarms.registerUploadAlarm(
						Alarms.getMediaUploadPendingIntent(mContext, 0),
						0L, Long.valueOf(frequency));
			}
			preference.setSummary(getFrequencySummary((int) frequency));
			return true;
		}
		
	};
	
	private String getFrequencySummary(long frequency) {
		int index = -1;
		if (frequency == 0) {
			index = 0;
		} else if (frequency == AlarmManager.INTERVAL_FIFTEEN_MINUTES) {
			index = 1;
		} else if (frequency == AlarmManager.INTERVAL_HALF_HOUR) {
			index = 2;
		} else if (frequency == AlarmManager.INTERVAL_HOUR) {
			index = 3;
		} else if (frequency == AlarmManager.INTERVAL_HALF_DAY) {
			index = 4;
		} else if (frequency == AlarmManager.INTERVAL_DAY) {
			index = 5;
		} else {
			Log.e(TAG, "no such frequency found: " + frequency);
		}
		
		if (index == -1)
			return "";
		else {
			final String[] upFreqEntries = getResources().getStringArray(
					R.array.upload_frequency_entries);
			return upFreqEntries[index];
		}
	}
	
	private OnPreferenceClickListener mForceMediaUploadListener =
			new OnPreferenceClickListener() {
		
		public boolean onPreferenceClick(Preference preference) {
			try {
				long lastTimestamp = StorageInfo.getLastUploadedPhotoTimestamp();
				int count = MediaUtilities.countImages2Sync(mContext, lastTimestamp);
				if (count > 0) {
					if (UpDownService.isUploadingPhotoMedia() == false) {
						UIUtil.showToast(mContext, R.string.toast_uploading_all_photos);
						final Intent intent = new Intent(MetaService.ACTION_UPLOAD_MEDIA);
						startService(intent);
					} else {
						UIUtil.showToast(mContext, R.string.toast_already_uploading);
					}
				} else {
					UIUtil.showToast(mContext, R.string.toast_no_photos_to_upload);
				}
			} catch (StorageInfo.StorageNotAvailable e) {
				UIUtil.showToast(mContext, R.string.toast_storage_is_not_accessible);
				return true;
			}
			return true;
		}
	};
	
	private OnPreferenceChangeListener mAutoRetryListener =
			new OnPreferenceChangeListener() {
		
		public boolean onPreferenceChange(Preference preference, Object newValue) {
			boolean retry = (Boolean) newValue;
			mTracker.trackEvent("Settings", "Advanced",
					"auto_retry", retry ? 1 : 0);
			
			if (retry) {
				if (MetaUtilities.getFailedTransfersCount() > 0) {
					MetaServiceHelper.triggerRetryFailed(mContext);
				}
			}
			return true;
		}
		
	};
	
	private OnPreferenceChangeListener mAllowPrefetchingListener =
			new OnPreferenceChangeListener() {
		
		public boolean onPreferenceChange(Preference preference, Object newValue) {
			final boolean prefetch = (Boolean) newValue;
			mTracker.trackEvent("Settings", "Advanced",
					"allow_prefetching", prefetch ? 1 : 0);
			Log.i(TAG, "prefetching has been turned " + (prefetch ? "ON" : "OFF"));
			if (!prefetch) {
				MetaService.disallowPrefetching();
			}
			return true;
		}
		
	};
	
	private OnPreferenceClickListener mRetryFailedListener =
			new OnPreferenceClickListener() {
		
		public boolean onPreferenceClick(Preference preference) {
			if (NetworkReceiver.isConnectedPreferred(mContext)) {
				if (adjustTransferPreferencesState()) {
					mTracker.trackEvent("Settings", "Options", "retry_failed", 1);
					UIUtil.showToast(mContext, R.string.toast_retrying_transfers_now);
					MetaServiceHelper.triggerRetryFailed(mContext);
				} else {
					UIUtil.showToast(mContext, R.string.toast_retrying_transfers_failed);
				}
			} else {
				UIUtil.showToast(mContext, R.string.toast_not_on_wifi);
			}
			return true;
		}
		
	};
	
	private OnPreferenceClickListener mCancelFailedListener =
			new OnPreferenceClickListener() {
	
		public boolean onPreferenceClick(Preference preference) {
			if (adjustTransferPreferencesState()) {
				mTracker.trackEvent("Settings", "Options", "cancel_failed", 1);
				MetaUtilities.cancelFailedTransfers();
				adjustTransferPreferencesState();
				UIUtil.showToast(mContext, R.string.toast_canceled_failed_transfers);
			} else {
				UIUtil.showToast(mContext, R.string.toast_cancel_failed_none);
			}
			return true;
		}
		
	};
	
	/**
	 * Adjust the android:enabled state of RETRY_FAILED and CANCEL_FAILED
	 * preferences based on number of pending failed transfers.
	 * 
	 * @return true if there are pending transfers, false otherwise
	 */
	private boolean adjustTransferPreferencesState() {
		int failedTransfers = MetaUtilities.getFailedTransfersCount();
		if (failedTransfers > 0) {
			mRetryFailed.setEnabled(true);
			mRetryFailed.setSummary(R.string.preferences_activity_retry_failed_summary_pending);
			mCancelFailed.setEnabled(true);
			mCancelFailed.setSummary(R.string.preferences_activity_cancel_failed_summary_pending);
			return true;
		} else {
			mRetryFailed.setEnabled(false);
			mRetryFailed.setSummary(R.string.preferences_activity_retry_failed_summary_none);
			mCancelFailed.setEnabled(false);
			mCancelFailed.setSummary(R.string.preferences_activity_cancel_failed_summary_none);
			return false;
		}
	}
	
	private OnPreferenceClickListener mRemoveDeviceListener =
			new OnPreferenceClickListener() {
		
		private DialogInterface.OnClickListener onClick =
				new DialogInterface.OnClickListener() {
			
			public void onClick(DialogInterface dialog, int which) {
				switch (which) {
				case Dialog.BUTTON_POSITIVE:
					// Remove access token.
					final String authToken = Preferences.getSerializedOAuthToken();
					if (authToken != null) {
						final AccountManager am =
								AccountManager.get(PreferencesActivity.this);
						Log.i(TAG, "Invalidating auth token.");
						am.invalidateAuthToken(
								Constants.ACCOUNT_TYPE, authToken);
					}
					Preferences.updateSerializedOAuthToken(null);
					
					mTracker.trackEvent("Settings", "Options", "unlink", 1);
					mTracker.dispatch();
					
					PreferencesActivity.this.setResult(RESULT_UNLINKED);
					finish();
					break;
				case Dialog.BUTTON_NEGATIVE:
					dialog.dismiss();
					break;
				default:
					Log.e(TAG, "no such button");
					break;
				}
			}
			
		};
		
		public boolean onPreferenceClick(Preference preference) {
			final AlertDialog dialog =
					new AlertDialog.Builder(mContext)
				.setTitle(R.string.dialog_remove_device_title)
				.setMessage(R.string.dialog_remove_device_message)
				.setPositiveButton(R.string.remove, onClick)
				.setNegativeButton(R.string.cancel, onClick)
				.create();
			dialog.setOwnerActivity(PreferencesActivity.this);
			dialog.show();
			return true;
		}
		
	};
	
	private OnPreferenceClickListener mPurchaseStorageListener =
			new OnPreferenceClickListener() {
		
		public boolean onPreferenceClick(Preference preference) {
			StoreActivity.startFrom(mContext);
			return true;
		}
	};
	
	private OnPreferenceClickListener mLinkListener =
			new OnPreferenceClickListener() {

		public boolean onPreferenceClick(Preference preference) {
			final String key = preference.getKey();
			if (key.equals(Pref.UPGRADE_PLAN)) {
				mTracker.trackEvent("Settings", "Links", "upgrade", 1);
				BrowserUtilities.open(mContext, Url.UPGRADE);
			} else if (key.equals(Pref.MANAGE_ACCOUNT)) {
				BrowserUtilities.open(mContext, Url.MANAGE);
			} else if (key.equals(Pref.SUPPORT_OPTIONS)) {
				mTracker.trackEvent("Settings", "Links", "support", 1);
				BrowserUtilities.open(mContext, Url.SUPPORT);
			} else if (key.equals(Pref.FEEDBACK)) {
				BrowserUtilities.open(mContext, Url.FEEDBACK);
			} else if (key.equals(Pref.LICENSE)) {
				BrowserUtilities.open(PreferencesActivity.this, Url.LICENSE);
			} else if (key.equals(Pref.AUTHORS)) {
				BrowserUtilities.open(mContext, Url.AUTHORS);
			} else if (key.equals(Pref.WEBPAGE)) {
				BrowserUtilities.open(mContext, Url.WEBPAGE);
			} else if (key.equals(Pref.GREENDROID)) {
				BrowserUtilities.open(mContext, Url.GREENDROID);
			}
			return true;
		}
		
	};
	
	private OnPreferenceClickListener mFeedbackListener =
			new OnPreferenceClickListener() {
		
		private final String EMAIL_TARGET = "ubuntuone-support@canonical.com";
		private final String EMAIL_SUBJECT = "Ubuntu One Files Feedback";
		
		public boolean onPreferenceClick(Preference preference) {
			final String details = getDetails();
			final String body = "Your feedback:\n";
			
			final Intent email = new Intent(Intent.ACTION_SEND);
			email.setType("message/rfc822");
			email.putExtra(Intent.EXTRA_EMAIL,
					new String[] { EMAIL_TARGET });
			email.putExtra(Intent.EXTRA_SUBJECT, EMAIL_SUBJECT);
			email.putExtra(Intent.EXTRA_TEXT, details + body);
			try {
				startActivity(email);
			} catch (ActivityNotFoundException e) {
				UIUtil.showToast(mContext, "No e-mail app?");
			}
			return false;
		}
		
	};
	
	private OnPreferenceClickListener mCollectLogsListener =
			new OnPreferenceClickListener() {
		
		public boolean onPreferenceClick(Preference preference) {
			if (ConfigUtilities.isExternalStorageMounted()) {
				final boolean isLogging = Log.startLogging();
				if (isLogging) {
					return false; // Let CheckBox be checked.
				} else {
					UIUtil.showToast(mContext, "Could not start logging.", true);
					return true;
				}
			} else {
				mCollectLogs.setChecked(false);
				UIUtil.showToast(mContext, R.string.need_to_mount_storage, true);
			}
			return true;
		}
		
	};
	
	private OnPreferenceClickListener mReviewLogsListener =
			new OnPreferenceClickListener() {
		
		public boolean onPreferenceClick(Preference preference) {
			if (ConfigUtilities.isExternalStorageMounted()) {
				reviewLogs();
			} else {
				UIUtil.showToast(mContext, R.string.need_to_mount_storage, true);
			}
			return false;
		}
		
	};
	
	private OnPreferenceClickListener mSendLogsListener =
			new OnPreferenceClickListener() {
		
		public boolean onPreferenceClick(Preference preference) {
			sendLogs();
			return false;
		}
		
	};
	
	private OnPreferenceClickListener mChangelogListener =
			new OnPreferenceClickListener() {
		
		public boolean onPreferenceClick(Preference preference) {
			ChangeLogUtils.showChangelog(PreferencesActivity.this);
			return true;
		}
		
	};
	
	@SuppressWarnings("unused")
	private void collectLogs() throws Exception {
		final File logFile = getLogFile();
		if (logFile.exists()) {
			logFile.delete();
		}
		boolean created = logFile.createNewFile();
		if (!created) {
			throw new Exception(getString(R.string.cant_create_log));
		}
		if (!logFile.canWrite()) {
			throw new Exception(getString(R.string.cant_write_log));
		}
		
		java.lang.Process logcat = null;
		BufferedReader reader = null;
		BufferedWriter writer = null;
		
		logcat = Runtime.getRuntime().exec(new String[] {
				"logcat", "-d", "AndroidRuntime:E System.err:V UbuntuOneFiles:D *:S" });
		reader = new BufferedReader(new InputStreamReader(
				logcat.getInputStream()));
		writer = new BufferedWriter(new FileWriter(logFile));
		
		Log.d(TAG, "collecting logs now");
		String line;
		final String newLine = System.getProperty("line.separator");
		while ((line = reader.readLine()) != null) {
			writer.write(line);
			writer.write(newLine);
		}
		writer.flush();
		writer.close();
	}
	
	private void reviewLogs() {
		final File logFile = getLogFile();
		final Uri uri = Uri.fromFile(logFile);
		Log.i(TAG, "reviewing " + uri.toString());
		final Intent intent = new Intent(Intent.ACTION_VIEW);
		intent.setFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);
		intent.setDataAndType(uri, "text/plain");
		try {
			startActivity(intent);
		} catch (ActivityNotFoundException e) {
			UIUtil.showToast(mContext, R.string.no_text_viewer, true);
		}
	}
	
	private final String getDetails() {
		final String version = UbuntuOneFiles.getVersion();
		final String details = Build.MODEL
				+ " running " + Build.VERSION.RELEASE
				+ ", Ubuntu One Files " + version + "\n\n";
		return details;
	}
	
	private static final String LOG_EMAIL_TARGET = "ubuntuone-support@canonical.com";
	private static final String LOG_EMAIL_SUBJECT = "Ubuntu One Files Logs";
	private static final String LOG_FILENAME = "logs.txt";
	
	private void sendLogs() {
		if (ConfigUtilities.isExternalStorageMounted()) {
			final String details = getDetails();
			final File logFile = getLogFile();
			final Uri uri = Uri.fromFile(logFile);
			
			if (logFile.exists()) {
				final Intent email = new Intent(Intent.ACTION_SEND);
				email.setType("message/rfc822");
				email.putExtra(Intent.EXTRA_EMAIL,
						new String[] { LOG_EMAIL_TARGET });
				email.putExtra(Intent.EXTRA_SUBJECT, LOG_EMAIL_SUBJECT);
				email.putExtra(Intent.EXTRA_TEXT, details);
				email.putExtra(Intent.EXTRA_STREAM, uri);
				startActivity(email);
				mCollectLogs.setChecked(false);
			}
		}
	}
	
	public static final File getLogFile() {
		final File extStorageDir =
				Environment.getExternalStorageDirectory();
		final String pkg =
				UbuntuOneFiles.class.getPackage().getName();
		final String dir =
				String.format("%s/Android/data/%s/files/log",
						extStorageDir, pkg);
		final File logDir = new File(dir);
		logDir.mkdirs();
		final File logFile = new File(logDir, LOG_FILENAME);
		return logFile;
	}

	public static void showFrom(Context context) {
		final Intent intent = new Intent(context, PreferencesActivity.class);
		context.startActivity(intent);
	}
	
	/** Hack for Theme.Light in sub {@link PreferenceScreen}. */
	private OnPreferenceClickListener whiteHackClick =
			new OnPreferenceClickListener() {
		
		public boolean onPreferenceClick(Preference preference) {
			final PreferenceScreen screen = (PreferenceScreen) preference;
			final Window window = screen.getDialog().getWindow();
			window.setBackgroundDrawableResource(android.R.color.white);
					
			// TODO karni: track down list view id and set the list selector
			return true;
		}
		
	};
}
