/*
 * Ubuntu One Files - access Ubuntu One cloud storage on Android platform.
 * 
 * Copyright (C) 2011 Canonical Ltd.
 * Author: Michał Karnicki <michal.karnicki@canonical.com>
 *   
 * This file is part of Ubuntu One Files.
 *  
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *  
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses 
 */

package com.ubuntuone.android.files.provider;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.Set;
import java.util.TreeSet;

import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;

import android.content.ContentResolver;
import android.content.ContentValues;
import android.database.Cursor;
import android.net.Uri;
import android.provider.MediaStore.MediaColumns;

import com.ubuntuone.android.files.Preferences;
import com.ubuntuone.android.files.UbuntuOneFiles;
import com.ubuntuone.android.files.provider.MetaContract.Nodes;
import com.ubuntuone.android.files.provider.MetaContract.ResourceState;
import com.ubuntuone.android.files.provider.MetaContract.Volumes;
import com.ubuntuone.android.files.util.Log;
import com.ubuntuone.rest.util.HashUtil;


public final class MetaUtilities {
	private static final String TAG = MetaUtilities.class.getSimpleName();
	
	private static ContentResolver sResolver;
	
	static {
		sResolver = UbuntuOneFiles.getInstance().getContentResolver();
	}
	
	public static void notifyChange(Uri uri) {
		sResolver.notifyChange(uri, null);
	}
	
	final static String sSelection = Nodes.NODE_RESOURCE_PATH + "=?";
	
	public static String getStringField(Uri uri, String columnName) {
		String result = null;
		final String[] projection = new String[] { columnName };
		final Cursor c = sResolver.query(uri, projection, null, null, null);
		try {
			if (c.moveToFirst()) {
				result = c.getString(c.getColumnIndex(columnName));
			}
		} finally {
			c.close();
		}
		return result;
	}
	
	public static String getStringField(String resourcePath, String columnName) {
		String result = null;
		final String[] projection = new String[] { columnName };
		final String[] selectionArgs = new String[] { resourcePath };
		final Cursor c = sResolver.query(Nodes.CONTENT_URI,
				projection, sSelection, selectionArgs, null);
		try {
			if (c.moveToFirst()) {
				result = c.getString(c.getColumnIndex(columnName));
			}
		} finally {
			c.close();
		}
		return result;
	}
	
	public static Cursor getNodeCursorByResourcePath(String resourcePath,
			String[] projection) {
		final String[] selectionArgs = new String[] { resourcePath };
		final Cursor cursor = sResolver.query(Nodes.CONTENT_URI, projection,
				sSelection, selectionArgs, null);
		return cursor;
	}
	
	public static Cursor getChildDirectoriesCursorByResourcePath(
			String resourcePath, String[] projection) {
		final String selection = Nodes.NODE_PARENT_PATH + "=?"
				+ " AND " + Nodes.NODE_KIND + "=?";
		final String[] selectionArgs =
				new String[] { resourcePath, Nodes.KIND_DIRECTORY };
		final Cursor cursor = sResolver.query(Nodes.CONTENT_URI, projection,
				selection, selectionArgs, null);
		return cursor;
	}
	
	public static String getPublicUrl(final String resourcePath) {
		final String[] projection = new String[] { Nodes.NODE_PUBLIC_URL };
		final String selection = Nodes.NODE_RESOURCE_PATH + "=?";
		final String[] selectionArgs = new String[] { resourcePath };
		Cursor c = sResolver.query(Nodes.CONTENT_URI,
				projection, selection, selectionArgs, null);
		String url = null;
		try {
			if (c.moveToFirst()) {
				url = c.getString(c.getColumnIndex(Nodes.NODE_PUBLIC_URL));
			}
		} finally {
			c.close();
		}
		if (url.length() == 0)
			url = null;
		return url;
	}
	
	public static int getCount(String resourcePath) {
		final String[] projection = new String[] { Nodes._ID };
		final String[] selectionArgs = new String[] { resourcePath };
		final Cursor c = sResolver.query(Nodes.CONTENT_URI,
				projection, sSelection, selectionArgs, null);
		int count = 0;
		try {
			count = c.getCount();
		} finally {
			c.close();
		}
		return count;
	}
	
	public static Set<Integer> getRootNodeIds() {
		Set<Integer> ids = new TreeSet<Integer>();
		final String[] projection = new String[] { Nodes._ID };
		final String selection = Nodes.NODE_PARENT_PATH + " IS NULL";
		final Cursor c = sResolver.query(Nodes.CONTENT_URI, projection,
				selection, null, null);
		try {
			if (c.moveToFirst()) {
				int id;
				do {
					id = c.getInt(c.getColumnIndex(Nodes._ID));
					ids.add(id);
				} while (c.moveToNext());
			}
		} finally {
			c.close();
		}
		return ids;
	}
	
	public static Set<String> getUserNodePaths() {
		Set<String> userNodePaths = new TreeSet<String>();
		final String[] projection =
				new String[] { Nodes._ID, Nodes.NODE_RESOURCE_PATH };
		final String selection = Nodes.NODE_PARENT_PATH + " IS NULL";
		final Cursor c = sResolver.query(Nodes.CONTENT_URI, projection,
				selection, null, null);
		try {
			if (c.moveToFirst()) {
				String resourcePath;
				do {
					resourcePath = c.getString(c.getColumnIndex(
							Nodes.NODE_RESOURCE_PATH));
					userNodePaths.add(resourcePath);
				} while (c.moveToNext());
			}
		} finally {
			c.close();
		}
		return userNodePaths;
	}
	
	public static Set<Integer> getChildrenIds(String resourcePath) {
		Set<Integer> ids = new TreeSet<Integer>();
		final String[] projection =
				new String[] { Nodes._ID, Nodes.NODE_RESOURCE_STATE };
		final String selection = Nodes.NODE_PARENT_PATH + "=?";
				//+ " AND " + Nodes.NODE_RESOURCE_STATE + " IS NULL"; // FIXME
		final String[] selectionArgs =
				new String[] { resourcePath };
		final Cursor c = sResolver.query(Nodes.CONTENT_URI, projection,
				selection, selectionArgs, null);
		try {
			if (c.moveToFirst()) {
				int id;
				do {
					id = c.getInt(
							c.getColumnIndex(Nodes._ID));
					// We check the state, above SQL is failing to filter out
					// nodes which are in non-null state. No idea why.
					String s = c.getString(
							c.getColumnIndex(Nodes.NODE_RESOURCE_STATE));
					if (s == null) {
						ids.add(id);
					} else {
						Log.d("MetaUtilities", "child state != null, ignoring");
					}
				} while (c.moveToNext());
			}
		} finally {
			c.close();
		}
		return ids;
	}
	
	public static String getHiddenSelection() {
		return Preferences.getShowHidden()
				? null : Nodes.NODE_NAME + " NOT LIKE '.%' AND "
						+ Nodes.NODE_NAME + " NOT LIKE '~/.%' ";
	}
	
	public static Cursor getVisibleVolumesCursor() {
		final Cursor c = sResolver.query(Volumes.CONTENT_URI,
				Volumes.getDefaultProjection(), null, null, null);
		c.setNotificationUri(sResolver, Volumes.CONTENT_URI);
		return c;
	}
	
	public static Cursor getVisibleTopNodesCursor() {
		String showHidden = Preferences.getShowHidden() ? "" :
			" AND " + getHiddenSelection();
	
		final String u1root = Preferences.U1_RESOURCE;
		
		String[] projection = Nodes.getDefaultProjection();
		String selection = "(" + Nodes.NODE_PARENT_PATH + " IS NULL AND " // UDFs
				+ Nodes.NODE_RESOURCE_PATH + " NOT LIKE ?) OR " // without ~/Ubuntu One
				+ Nodes.NODE_PARENT_PATH + "=?" // contents of ~/Ubuntu One
				+ showHidden;
		String[] selectionArgs = new String[] { "%" + u1root, u1root };
		final Cursor c = sResolver.query(Nodes.CONTENT_URI, projection,
				selection, selectionArgs, null);
		c.setNotificationUri(sResolver, Nodes.CONTENT_URI);
		return c;
	}
	
	public static Cursor getVisibleNodesCursorByParent(String parentPath) {
		String showHidden = Preferences.getShowHidden() ? "" :
				" AND " + getHiddenSelection();
		
		String[] projection = Nodes.getDefaultProjection();
		String selection = Nodes.NODE_PARENT_PATH + "=? "
				+ showHidden;
		String[] selectionArgs = new String[] { parentPath };
		final Cursor c = sResolver.query(Nodes.CONTENT_URI, projection,
				selection, selectionArgs, null);
		c.setNotificationUri(sResolver, Nodes.CONTENT_URI);
		return c;
	}

	/**
	 * Calculates directory content size, recursively if necessary.
	 * 
	 * @param resourcePath
	 *            the directory resource path to calculate size of
	 * @param recursive
	 *            the flag indicating recursive calculation
	 * @return the resorucePath defined directory size
	 */
	public static long getDirectorySize(final String resourcePath,
			final boolean recursive) {
		final String[] projection = new String[] { Nodes.NODE_RESOURCE_PATH,
				Nodes.NODE_KIND, Nodes.NODE_SIZE };
		final String selection = Nodes.NODE_PARENT_PATH + "=?";
		final String[] selectionArgs =
				new String[] { resourcePath };
		final Cursor c = sResolver.query(Nodes.CONTENT_URI, projection,
				selection, selectionArgs, null);
		String kind;
		long size = 0L;
		try {
			if (c.moveToFirst()) {
				do {
					kind = c.getString(c.getColumnIndex(Nodes.NODE_KIND));
					if (Nodes.KIND_FILE.equals(kind)) {
						size += c.getLong(c.getColumnIndex(Nodes.NODE_SIZE));
					} else if (Nodes.KIND_DIRECTORY.equals(kind) && recursive) {
						final String subDirResourcePath = c.getString(c
								.getColumnIndex(Nodes.NODE_RESOURCE_PATH));
						size += getDirectorySize(subDirResourcePath, true);
					}
				} while (c.moveToNext());
			}
		} finally {
			c.close();
		}
		return size;
	}
	
	public static Cursor getFailedTransfers() {
		final String[] projection =	new String[] {
				Nodes.NODE_RESOURCE_PATH, Nodes.NODE_RESOURCE_STATE,
				Nodes.NODE_PARENT_PATH, Nodes.NODE_DATA };
		final String selection = Nodes.NODE_RESOURCE_STATE
				+ " LIKE '%" + ResourceState.FAILED + "'";
		final String sortOrder = Nodes.NODE_RESOURCE_STATE + " ASC";
		return sResolver.query(Nodes.CONTENT_URI, projection, selection,
				null, sortOrder);
	}
	
	public static int getFailedTransfersCount() {
		final String[] projection =	new String[] { Nodes._ID };
		final String selection = Nodes.NODE_RESOURCE_STATE
				+ " LIKE '%" + ResourceState.FAILED + "'";
		final String sortOrder = Nodes.NODE_RESOURCE_STATE + " ASC";
		final Cursor c = sResolver.query(Nodes.CONTENT_URI, projection,
				selection, null, sortOrder);
		int result = c.getCount();
		c.close();
		return result;
	}
	
	public static int resetFailedTransfers() {
		int failed = 0;
		failed += resetFailedTransfers(HttpPost.METHOD_NAME);
		failed += resetFailedTransfers(HttpGet.METHOD_NAME);
		return failed;
	}
	
	public static int resetFailedTransfers(String method) {
		String ongoing;
		String failed;
		if (HttpPost.METHOD_NAME.equals(method)) {
			ongoing = ResourceState.STATE_POSTING;
			failed = ResourceState.STATE_POSTING_FAILED;
		} else if (HttpGet.METHOD_NAME.equals(method)) {
			ongoing = ResourceState.STATE_GETTING;
			failed = ResourceState.STATE_GETTING_FAILED;
		} else {
			Log.e(TAG, "Bad method name: " + method);
			return 0;
		}
		
		final ContentValues values = new ContentValues(1);
		values.put(Nodes.NODE_RESOURCE_STATE, failed);
		final String where = Nodes.NODE_RESOURCE_STATE + "=?";
		final String[] selectionArgs = new String[] { ongoing };
		return sResolver.update(Nodes.CONTENT_URI, values, where, selectionArgs);
	}
	
	public static void cancelFailedTransfers() {
		ContentValues values;
		final String where = Nodes.NODE_RESOURCE_STATE + "=?";
		String[] selectionArgs;
		String clearState = null;
		
		// Reset uploads.
		selectionArgs = new String[] { ResourceState.STATE_POSTING_FAILED };
		sResolver.delete(Nodes.CONTENT_URI, where, selectionArgs);
		
		// Reset downloads.
		selectionArgs = new String[] { ResourceState.STATE_GETTING_FAILED };
		values = new ContentValues(1);
		values.put(Nodes.NODE_RESOURCE_STATE, clearState);
		sResolver.update(Nodes.CONTENT_URI, values, where, selectionArgs);
	}
	
	public static void updateLongField(String resourcePath, String column, Long value) {
		final String[] selectionArgs = new String[] { resourcePath };
		final ContentValues values = new ContentValues();
		values.put(column, value);
		sResolver.update(Nodes.CONTENT_URI, values, sSelection, selectionArgs);
	}
	
	public static void updateStringField(String resourcePath, String column, String value) {
		final String[] selectionArgs = new String[] { resourcePath };
		final ContentValues values = new ContentValues();
		values.put(column, value);
		sResolver.update(Nodes.CONTENT_URI, values, sSelection, selectionArgs);
	}
	
	public static void setState(final String resourcePath, final String state) {
		final ContentValues values = new ContentValues();
		values.put(Nodes.NODE_RESOURCE_STATE, state);
		final String where = Nodes.NODE_RESOURCE_PATH + "=?";
		final String[] selectionArgs = new String[] { resourcePath };
		sResolver.update(Nodes.CONTENT_URI, values, where, selectionArgs);
	}
	
	public static void setStateAndData(final String resourcePath,
			final String state, final String data) {
		final String[] selectionArgs = new String[] { resourcePath };
		final ContentValues values = new ContentValues();
		values.put(Nodes.NODE_RESOURCE_STATE, state);
		values.put(Nodes.NODE_DATA, data);
		sResolver.update(Nodes.CONTENT_URI, values, sSelection, selectionArgs);
	}
	
	public static void setIsCached(String resourcePath, boolean isCached) {
		ContentValues values = new ContentValues(2);
		values.put(Nodes.NODE_RESOURCE_PATH, resourcePath);
		values.put(Nodes.NODE_IS_CACHED, isCached);
		String where = Nodes.NODE_RESOURCE_PATH + "=?";
		String[] selectionArgs = new String[] { resourcePath };
		sResolver.update(Nodes.CONTENT_URI, values, where, selectionArgs);
		sResolver.notifyChange(Nodes.CONTENT_URI, null);
	}
	
	public static void deleteByResourcePath(String resourcePath) {
		String where = Nodes.NODE_RESOURCE_PATH + "=?";
		String[] selectionArgs = new String[] { resourcePath };
		sResolver.delete(Nodes.CONTENT_URI, where, selectionArgs);
	}
	
	public static Uri buildNodeUri(int id) {
		return Nodes.CONTENT_URI.buildUpon()
				.appendPath(String.valueOf(id)).build();
	}
	
	public static boolean isValidUri(Uri uri) {
		try {
			final InputStream in = sResolver.openInputStream(uri);
			in.close();
		} catch (FileNotFoundException e) {
			return false;
		} catch (IOException e) {
			return false;
		}
		return true;
	}
	
	public static boolean isValidUriTarget(String data) {
		if (data == null) {
			Log.d(TAG, "uri is null");
			return false;
		} else if (data.startsWith(ContentResolver.SCHEME_CONTENT)) {
			Log.d(TAG, "checking content uri");
			return isValidUri(Uri.parse(data));
		} else if (data.startsWith(ContentResolver.SCHEME_FILE)) {
			try {
				Log.d(TAG, "checking file uri");
				final URI fileURI = URI.create(Uri.encode(data, ":/"));
				final boolean exists = new File(fileURI).exists();
				return exists;
			} catch (IllegalArgumentException e) {
				// We have received wrong uri. Failed upload shall be cleaned up.
				return false;
			}
		} else {
			Log.e(TAG, "unknown uri: " + data);
			return false;
		}
	}
	
	public static boolean isDirectory(long id) {
		String[] projection = new String[] { Nodes.NODE_KIND };
		String selection = Nodes._ID + "=?";
		String[] selectionArgs = new String[] { String.valueOf(id) };
		final Cursor c = sResolver.query(Nodes.CONTENT_URI, projection,
				selection, selectionArgs, null);
		String type = null;
		try {
			if (c.moveToFirst()) {
				type = c.getString(c.getColumnIndex(Nodes.NODE_KIND));
			}
		} finally {
			c.close();
		}
		return (type != null) ? Nodes.KIND_DIRECTORY.equals(type) : false;
	}
	
	public static boolean isDirectory(Cursor c) {
		final String type = c.getString(c.getColumnIndex(Nodes.NODE_KIND));
		return Nodes.KIND_DIRECTORY.equals(type);
	}

	/**
	 * For a given file {@link Uri} string, we check if its hash has a
	 * corresponding entry in the {@link MetaProvider}, telling thus whether the
	 * file from under given {@link Uri} string has been already uploaded.
	 * 
	 * @param uriString
	 *            the uri string which content we are checking
	 * @return resourcePath if content under uri has been already uploaded,
	 *         null otherwise
	 */
	public static String isUploaded(String uriString) {
		File file = null;
		String fileHash = null;
		
		if (uriString.startsWith(ContentResolver.SCHEME_CONTENT)) {
			final String[] projection = new String[] { MediaColumns.DATA };
			final Cursor c = sResolver.query(Uri.parse(uriString),
					projection, null, null, null);
			try {
				if (c.moveToFirst()) {
					String data = c.getString(c.getColumnIndex(MediaColumns.DATA));
					file = new File(data);
				} else {
					return null;
				}
			} finally {
				c.close();
			}
		} else if (uriString.startsWith(ContentResolver.SCHEME_FILE)) {
			final URI fileURI = URI.create(Uri.encode(uriString, ":/"));
			file = new File(fileURI);
		} else {
			Log.e(TAG, "Tried to check malformed uri string: " + uriString);
			return null;
		}
		
		try {
			if (file != null && file.exists()) {
				fileHash = HashUtil.sha1(file);
				Log.d(TAG, String.format("Computed hash: '%s'", fileHash));
			} else {
				throw new FileNotFoundException("isUploaded()");
			}
		} catch (Exception e) {
			Log.e(TAG, "Can't compute file hash!", e);
			return null;
		}
		
		final String[] projection =
				new String[] { Nodes.NODE_RESOURCE_PATH };
		final String selection = Nodes.NODE_HASH + "=?";
		final String[] selectionArgs = new String[] { fileHash };
		final Cursor c = sResolver.query(Nodes.CONTENT_URI, projection,
				selection, selectionArgs, null);
		String resourcePath = null;
		try {
			if (c.moveToFirst()) {
				resourcePath = c.getString(c.getColumnIndex(
						Nodes.NODE_RESOURCE_PATH));
				Log.d(TAG, "Corresponding file hash found: " + resourcePath);
			} else {
				Log.d(TAG, "Corresponding file hash not found.");
			}
		} finally {
			c.close();
		}
		return resourcePath;
	}

	private MetaUtilities() {
	}
}
