/*
 * Ubuntu One Files - access Ubuntu One cloud storage on Android platform.
 * 
 * Copyright (C) 2011 Canonical Ltd.
 * Author: Michał Karnicki <michal.karnicki@canonical.com>
 *   
 * This file is part of Ubuntu One Files.
 *  
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *  
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses 
 */

package com.ubuntuone.android.files.receiver;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.SystemClock;

import com.ubuntuone.android.files.Preferences;
import com.ubuntuone.android.files.UbuntuOneFiles;
import com.ubuntuone.android.files.service.MetaServiceHelper;
import com.ubuntuone.android.files.util.DateUtilities;
import com.ubuntuone.android.files.util.Log;

public class NetworkReceiver extends BroadcastReceiver {
	private static final String TAG = NetworkReceiver.class.getSimpleName();
	
	private static boolean sHasWifiNetwork;
	private static boolean sHasMobileNetwork;
	private static boolean sCanUseBackgroundData;
	
	private static Context sContext;
	private static ConnectivityManager sConnectivityManager;

	/**
	 * Initialize networks state flags and background data preference.
	 * 
	 * @param context
	 *            the context to use
	 */
	public static void init() {
		sContext = UbuntuOneFiles.getInstance().getApplicationContext();
		sConnectivityManager = (ConnectivityManager) sContext
				.getSystemService(Context.CONNECTIVITY_SERVICE);
		sCanUseBackgroundData = sConnectivityManager.getBackgroundDataSetting();
		final NetworkInfo ni = sConnectivityManager.getActiveNetworkInfo();
		saveNetworkStatus(ni);
	}

	/**
	 * Saves currently available network status in static class fields.
	 * 
	 * @param networkInfo
	 *            the {@link NetworkInfo} to save
	 */
	private static void saveNetworkStatus(NetworkInfo networkInfo) {
		if (networkInfo != null && networkInfo.isConnected()) {
			switch (networkInfo.getType()) {
			case ConnectivityManager.TYPE_WIFI:
				Log.d(TAG, "saveNetworkStatus() Wi-Fi connected");
				sHasWifiNetwork = true;
				sHasMobileNetwork = false;
				break;
			case ConnectivityManager.TYPE_WIMAX:
			case ConnectivityManager.TYPE_MOBILE:
				Log.d(TAG, "saveNetworkStatus() Mobile connected");
				// TODO karni: roaming check/preference
				sHasMobileNetwork = true;
				sHasWifiNetwork = false;
				break;
			}
		} else {
			Log.d(TAG, "saveNetworkStatus() disconnected");
			sHasMobileNetwork = false;
			sHasWifiNetwork = false;
		}
	}
	
	/**
	 * Tells if we are connected.
	 * 
	 * @return true, if connected; false otherwise
	 */
	public static boolean isConnected() {
		init();
		
		final boolean isConnected = sHasWifiNetwork || sHasMobileNetwork;
		Log.d(TAG, "isConnected() " + isConnected);
		return isConnected;
	}

	/**
	 * Tells if we are connected, but considers USE_WIFI_ONLY preference.
	 * 
	 * @param context
	 *            the context to use
	 * @return true, if connected; false otherwise
	 */
	public static boolean isConnectedPreferred(Context context) {
		init();
		
		final boolean useWifiOnly = Preferences.getBoolean(
				Preferences.USE_WIFI_ONLY_KEY, false);
		if (useWifiOnly) {
			Log.d(TAG, "isConnectedPreferred() Wi-Fi -> " + sHasWifiNetwork);
			return sHasWifiNetwork;
		} else {
			final boolean connected = sHasWifiNetwork || sHasMobileNetwork;
			Log.d(TAG, "isConnectedPreferred() any -> " + connected);
			return connected;
		}
	}
	

	@Override
	public void onReceive(Context context, Intent intent) {
		Log.d(TAG, "onReceive() " + intent.toString());
		final String action = intent.getAction();
		if (ConnectivityManager.CONNECTIVITY_ACTION.equals(action)) {
			onConnectivityAction(context, intent);
		} else if (ConnectivityManager
				.ACTION_BACKGROUND_DATA_SETTING_CHANGED.equals(action)) {
			onBackgroundDataSettingChanged(context);
			return;
		}
	}

	/**
	 * Processes the {@link ConnectivityManager#CONNECTIVITY_ACTION} intent.
	 * 
	 * @param context
	 *            the context to use
	 * @param intent
	 *            the {@link Intent} with {@link NetworkInfo}
	 */
	private void onConnectivityAction(Context context, Intent intent) {
		final NetworkInfo networkInfo = intent.getParcelableExtra(
				ConnectivityManager.EXTRA_NETWORK_INFO);
		saveNetworkStatus(networkInfo);
		
		switch (networkInfo.getState()) {
		case CONNECTED:
			Log.i(TAG, "onConnectivityAction() connected");
			// Don't trigger actions immediately after phone boot.
			if (SystemClock.uptimeMillis() < DateUtilities.MILLIS_IN_MINUTE * 3) {
				Log.d(TAG, "Ignorning connectivity event, too short after system boot");
				return;
			}
			
			if (isConnectedPreferred(context)) {
				if (Preferences.getBoolean(Preferences.UPLOAD_PHOTOS_KEY, false)) {
					MetaServiceHelper.triggerAutoUpload(context);
				}
				if (Preferences.getBoolean(Preferences.AUTO_RETRY_FAILED, false)) {
					MetaServiceHelper.triggerRetryFailed(context);
				}
			}
			break;
		case DISCONNECTING:
			//$FALL-THROUGH$
		case DISCONNECTED:
			Log.i(TAG, "onConnectivityAction() disconnecting/disconnected");
			break;
		}
	}

	/**
	 * Processes the
	 * {@link ConnectivityManager#ACTION_BACKGROUND_DATA_SETTING_CHANGED}
	 * event.
	 * 
	 * @param context
	 *            the context to use
	 */
	private void onBackgroundDataSettingChanged(Context context) {
		final ConnectivityManager cm = (ConnectivityManager) context
				.getSystemService(Context.CONNECTIVITY_SERVICE);
		sCanUseBackgroundData = cm.getBackgroundDataSetting();
		Log.i(TAG, "background data allowed: " + sCanUseBackgroundData);
	}
}
