/*
 * Ubuntu One Files - access Ubuntu One cloud storage on Android platform.
 * 
 * Copyright (C) 2011 Canonical Ltd.
 * Author: Michał Karnicki <michal.karnicki@canonical.com>
 *   
 * This file is part of Ubuntu One Files.
 *  
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *  
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses 
 */

package com.ubuntuone.android.files.receiver;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;

import com.ubuntuone.android.files.Alarms;
import com.ubuntuone.android.files.Preferences;
import com.ubuntuone.android.files.service.MediaCatcher;
import com.ubuntuone.android.files.service.MetaServiceHelper;
import com.ubuntuone.android.files.util.Log;
import com.ubuntuone.android.files.util.StorageInfo;

public class SystemReceiver extends BroadcastReceiver {
	private final static String TAG = SystemReceiver.class.getSimpleName();

	@Override
	public void onReceive(Context context, Intent intent) {		
		Log.d(TAG, "onReceive() " + intent.toString());
		final String action = intent.getAction();
		
		boolean upload = Preferences.getBoolean(
				Preferences.UPLOAD_PHOTOS_KEY, false);
		long freq = Preferences.getLongFromString(
				Preferences.UPLOAD_FREQUENCY_KEY,
				Preferences.DEFAULT_UPLOAD_FREQUENCY);
		boolean immediate = (freq == 0L);
		
		if (Intent.ACTION_BOOT_COMPLETED.equals(action)) {
			// Cool, if we're here the application has already started and
			// "maybeRegistered" the upload alarm. All that we needed.
			return;
		} else if (Intent.ACTION_MEDIA_MOUNTED.equals(action) && upload) {
			onMediaMountedWithUpload(context, immediate);
			return;
		} else if (Intent.ACTION_MEDIA_UNMOUNTED.equals(action) && upload) {
			onMediaRemovedWithUpload(context, immediate);
			return;
		}
		Log.w(TAG, "unhandled broadcast: " + action);
	}
	
	private void onMediaMountedWithUpload(Context context, boolean immediate) {
		if (immediate) {
			MediaCatcher.startFrom(context);
			try {
				boolean should = StorageInfo.shouldImmediatelyUploadPhotos();
				boolean connected =
						NetworkReceiver.isConnectedPreferred(context);
				if (connected && should) {
					MetaServiceHelper.triggerAutoUpload(context);
				}
			} catch (StorageInfo.StorageNotAvailable e) {
				Log.e(TAG, "", e);
			}
		} else {
			Alarms.maybeRegisterMediaUploadAlarm();
			Alarms.maybeRegisterRetryFailedAlarm();
		}
	}
	
	private void onMediaRemovedWithUpload(Context context, boolean immediate) {
		if (immediate) {
			MediaCatcher.stopFrom(context);
		} else {
			Alarms.unregisterPictureUploadAlarm();
		}
	}
}
