/*
 * Ubuntu One Files - access Ubuntu One cloud storage on Android platform.
 * 
 * Copyright (C) 2011 Canonical Ltd.
 * Author: Chad Miller <chad.miller@canonical.com>
 *   
 * This file is part of Ubuntu One Files.
 *  
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *  
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses 
 */

/** Upload files taken with camera. */

package com.ubuntuone.android.files.service;

import android.app.Service;
import android.content.ContentResolver;
import android.content.Context;
import android.content.Intent;
import android.database.ContentObserver;
import android.database.Cursor;
import android.net.Uri;
import android.os.Handler;
import android.os.IBinder;
import android.provider.MediaStore;

import com.ubuntuone.android.files.util.Log;

public class MediaCatcher extends Service
{
	private static final String TAG = MediaCatcher.class.getSimpleName();

	private Handler handler;
	private Cursor photoCursor;
	private PhotoObserver photoObserver;

	class CursorGetterRunnable implements Runnable {
		private long CURSOR_RETRY_INTERVAL_MS = 5000L;
		private final Uri STORAGE_URI = 
				MediaStore.Images.Media.EXTERNAL_CONTENT_URI;

		public void run() {
			ContentResolver cr = MediaCatcher.this.getContentResolver();
			Cursor cursor = MediaStore.Images.Media.query(cr, STORAGE_URI,
					new String [] { MediaStore.Images.Media._ID, });

			if (cursor == null) {
				Log.w(TAG, "Failed to create new cursor.  Trying again soon.");
				MediaCatcher.this.handler.postDelayed(this, CURSOR_RETRY_INTERVAL_MS);
			} else {
				MediaCatcher.this.photoObserver = new PhotoObserver(MediaCatcher.this.handler);
				MediaCatcher.this.photoCursor = cursor;
				MediaCatcher.this.photoCursor.registerContentObserver(MediaCatcher.this.photoObserver);
				Log.i(TAG, "Now listening on cursor for Photo changes.");
			}
		}
	};

	@Override
	public IBinder onBind(Intent startService)
	{
		Log.v(TAG, "onBind() " + startService.toString());
		return null;
	}

	@Override
	public void onCreate()
	{
		super.onCreate();
		Log.d(TAG, "onCreate();");
		this.handler = new Handler();
	}

	@Override
	public int onStartCommand(Intent startService, int flags, int startId)
	{
		super.onStartCommand(startService, flags, startId);
		Log.d(TAG, "onStart() " + startService + " " + startId);
		this.handler.post(new CursorGetterRunnable());
		return Service.START_STICKY;
	}

	@Override
	public void onDestroy() {
		Log.d(TAG, "onDestroy()");
		if ((this.photoCursor != null) && (this.photoObserver != null)) {
			this.photoCursor.unregisterContentObserver(this.photoObserver);
			this.photoCursor.close();
		}
		super.onDestroy();
	}

	class TriggerPhotosUploadRunnable implements Runnable {
		final Intent actionIntent = 
				new Intent(MetaService.ACTION_UPLOAD_MEDIA);
		public void run() {
			MediaCatcher.this.startService(actionIntent);
		}
	}

	private class PhotoObserver extends ContentObserver
	{
		private final String TAG = PhotoObserver.class.getSimpleName();
		private final long RATE_LIMIT_MS = 3000L;
		private final long DELAY_MS = 3000L;
		private long lastChangeNotified = 0;

		public PhotoObserver(Handler handler)
		{
			super(handler);
			Log.v(TAG, "PhotoObserver(" + handler.toString() + ");");
		}

		@Override
		public void onChange(boolean selfChange)
		{
			super.onChange(selfChange);
			long now = System.currentTimeMillis();
			if (now > (lastChangeNotified + RATE_LIMIT_MS)) {
				this.lastChangeNotified = now;
				MediaCatcher.this.handler.postDelayed(
						new TriggerPhotosUploadRunnable(), DELAY_MS);
				Log.i(TAG, "Firing change notification in " + DELAY_MS/1000 + "sec.");
			} else {
				Log.d(TAG, "Not firing change notification again so soon.");
			}
		}
	}
	
	public static boolean startFrom(Context context) {
		final Intent mediaCatcher = new Intent(context, MediaCatcher.class); 
		return context.startService(mediaCatcher) != null;
	}
	
	public static boolean stopFrom(Context context) {
		final Intent mediaCatcher = new Intent(context, MediaCatcher.class); 
		return context.stopService(mediaCatcher);
	}

}
