/*
 * Ubuntu One Files - access Ubuntu One cloud storage on Android platform.
 * 
 * Copyright (C) 2011 Canonical Ltd.
 * Author: Michał Karnicki <michal.karnicki@canonical.com>
 *   
 * This file is part of Ubuntu One Files.
 *  
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *  
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses 
 */

package com.ubuntuone.android.files.service;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;

import oauth.signpost.OAuthConsumer;

import org.apache.http.client.HttpClient;

import android.content.ContentProviderOperation;
import android.content.ContentResolver;
import android.content.ContentValues;
import android.content.Context;
import android.content.Intent;
import android.content.OperationApplicationException;
import android.database.Cursor;
import android.net.Uri;
import android.os.Bundle;
import android.os.Environment;
import android.os.IBinder;
import android.os.RemoteException;
import android.os.ResultReceiver;
import android.provider.MediaStore;
import android.provider.MediaStore.MediaColumns;

import com.ubuntuone.android.files.Preferences;
import com.ubuntuone.android.files.R;
import com.ubuntuone.android.files.UbuntuOneFiles;
import com.ubuntuone.android.files.provider.MetaContract;
import com.ubuntuone.android.files.provider.MetaUtilities;
import com.ubuntuone.android.files.provider.MetaContract.Nodes;
import com.ubuntuone.android.files.provider.MetaContract.ResourceState;
import com.ubuntuone.android.files.receiver.NetworkReceiver;
import com.ubuntuone.android.files.util.AwakeIntentService;
import com.ubuntuone.android.files.util.FileUtilities;
import com.ubuntuone.android.files.util.HttpManager;
import com.ubuntuone.android.files.util.Log;
import com.ubuntuone.android.files.util.MediaUtilities;
import com.ubuntuone.android.files.util.Authorizer;
import com.ubuntuone.android.files.util.StorageInfo;
import com.ubuntuone.android.files.util.UIUtil;
import com.ubuntuone.rest.RestApi;
import com.ubuntuone.rest.Exceptions.InternalServerErrorException;
import com.ubuntuone.rest.Exceptions.NotFoundException;
import com.ubuntuone.rest.Exceptions.UnauthorizedException;
import com.ubuntuone.rest.resources.NodeInfo;
import com.ubuntuone.rest.resources.UserInfo;
import com.ubuntuone.rest.resources.VolumeInfo;

/**
 * This service is used to fetch and update meta data. For transfers,
 * please use {@link UpDownService}.<br />
 * <br />
 * Intents which do not contain EXTRA_PREFETCH are assumed user action and
 * all other activity should be canceled to handle such request.
 */
public class MetaService extends AwakeIntentService {
	
	private static final String TAG = MetaService.class.getSimpleName();
	
	private static final String BASE = "com.ubuntuone.android.files";
	/** Refresh user info. */
	public static final String ACTION_REFRESH_USER = BASE + ".ACTION_REFRESH_USER";
	/** Create a new volume. */
	public static final String ACTION_CREATE_VOLUME = BASE + ".ACTION_CREATE_VOLUME";
	/** Refresh cache of a volume resource. */
	public static final String ACTION_REFRESH_VOLUME = BASE + ".ACTION_REFRESH_VOLUME";
	/** Create a volume. Currently no fields are editable. */
	public static final String ACTION_UPDATE_VOLUME = BASE + ".ACTION_UPDATE_VOLUME";
	/** Delete a volume. */
	public static final String ACTION_DELETE_VOLUME = BASE + ".ACTION_DELETE_VOLUME";
	/** Create a new node. Use for directories only. Use {@link UpDownService} for files. */
	public static final String ACTION_CREATE_NODE = BASE + ".ACTION_CREATE_NODE";
	/** Refresh cache of a node resource. */
	public static final String ACTION_REFRESH_NODE = BASE + ".ACTION_REFRESH_NODE";
	/** Update node resource. */
	public static final String ACTION_UPDATE_NODE = BASE + ".ACTION_UPDATE_NODE";
	/** Delete remote resource. */
	public static final String ACTION_DELETE_NODE = BASE + ".ACTION_DELETE_NODE";
	/** Share media from gallery. */
	public static final String ACTION_SHARE_MEDIA = BASE + ".ACTION_SHARE_MEDIA";
	/** Auto-upload media request. */
	public static final String ACTION_UPLOAD_MEDIA = BASE + ".ACTION_UPLOAD_MEDIA";
	/** Retry failed transfers request. */
	public static final String ACTION_RETRY_FAILED = BASE + ".ACTION_RETRY_FAILED";
	
	public static final String EXTRA_CALLBACK = "extra_callback";
	public static final String EXTRA_TIMESTAMP = "extra_timestamp";
	public static final String EXTRA_PREFETCH = "extra_prefetch";
	public static final String EXTRA_ERROR = "extra_error";
	public static final String EXTRA_RESOURCE_PATH = Nodes.NODE_RESOURCE_PATH;
	public static final String EXTRA_ID = Nodes._ID;
	public static final String EXTRA_PATH = Nodes.NODE_PATH;
	public static final String EXTRA_KIND = Nodes.NODE_KIND;
	public static final String EXTRA_NAME = Nodes.NODE_NAME;
	public static final String EXTRA_SIZE = Nodes.NODE_SIZE;
	public static final String EXTRA_HAS_CHILDREN = Nodes.NODE_HAS_CHILDREN;
	
	public static interface Status {
		public final int PENDING = 1;
		public final int RUNNING = 2;
		public final int PROGRESS = 3;
		public final int FINISHED = 4;
		public final int ERROR = 5;
	}
	
	public static boolean sIsWorking = false;
	private static boolean sAllowPrefetching = true;
	
	private ContentResolver mResolver;
	
	private HttpClient mHttpClient;
	private OAuthConsumer mConsumer;
	
	public MetaService() {
		super(MetaService.class.getSimpleName());
		mResolver = UbuntuOneFiles.getInstance().getContentResolver();
		mHttpClient = HttpManager.getClient();
	}
	
	public static void allowPrefetching() {
		sAllowPrefetching = true;
	}
	
	public static void disallowPrefetching() {
		sAllowPrefetching = false;
	}

	@Override
	public IBinder onBind(Intent intent) {
		Log.d(TAG, "onBind() " + intent.toString());
		return null;
	}
	
	@Override
	public void onCreate() {
		super.onCreate();
		Log.d(TAG, "onCreate()");
		sIsWorking = true;
		
		try {
			mConsumer = Authorizer.getInstance(false);
		} catch (Exception e) {
			Log.e(TAG, "Auth client failure", e);
			stopSelf();
			return;
		}
		allowPrefetching();
	}

	// Track how many user actions are pending before prefetching is allowed.
	private AtomicInteger mUserActions = new AtomicInteger();
	
	@Override
	public int onStartCommand(Intent intent, int flags, int startId) {
		Log.d(TAG, "onStart() " + intent + " " + startId);
		final boolean userAction = !intent.hasExtra(EXTRA_PREFETCH);
		if (userAction) {
			mUserActions.incrementAndGet();
		}
		return super.onStartCommand(intent, flags, startId);
	}
	
	@Override
	protected void onHandleIntent(Intent intent) {
		final String action = intent.getAction();
		// These are sorted more or less by frequency of use.
		if (ACTION_CREATE_NODE.equals(action)) {
			createNode(intent);
		} else if (ACTION_REFRESH_NODE.equals(action)) {
			refreshNode(intent);
		} else if (ACTION_UPDATE_NODE.equals(action)) {
			updateNode(intent);
		} else if (ACTION_DELETE_NODE.equals(action)) {
			deleteNode(intent);
		} else if (ACTION_REFRESH_USER.equals(action)) {
			refreshUser(intent);
		} else if (ACTION_CREATE_VOLUME.equals(action)) {
			createVolume(intent);
		} else if (ACTION_REFRESH_VOLUME.equals(action)) {
			refreshVolume(intent);
		} else if (ACTION_UPDATE_VOLUME.equals(action)) {
			updateVolume(intent);
		} else if (ACTION_DELETE_VOLUME.equals(action)) {
			deleteVolume(intent);
		} else if (ACTION_SHARE_MEDIA.equals(action)) {
			sharePicture(intent);
		} else  if (ACTION_UPLOAD_MEDIA.equals(action)) {
			uploadPictures(intent);
		} else if (ACTION_RETRY_FAILED.equals(action)) {
			retryFailed(intent);
		} else {
			Log.e(TAG, "unhandled action!");
		}
		
		final boolean userAction = !intent.hasExtra(EXTRA_PREFETCH);
		if (userAction) {
			int act = mUserActions.decrementAndGet();
			// Allow prefetch if there are no user actions pending.
			if (act == 0) {
				allowPrefetching();
			}
		}
	}
	
	@Override
	public void onDestroy() {
		Log.d(TAG, "onDestroy()");
		sIsWorking = false;
		allowPrefetching();
		super.onDestroy();
	}

	/**
	 * Tells if we should ignore some common exceptions, usually connectivity
	 * related.
	 * 
	 * @param e
	 *            the exception to check
	 * @return true, if the Exception should be ignored and no explicit feedback
	 *         provided back to the UI
	 */
	public static boolean catchException(Exception e) {
		if (e instanceof UnknownHostException) {
			return true;
		} else if (e instanceof IOException) {
			return true;
		} else if (e instanceof NotFoundException) {
			return true;
		} else if (e instanceof InternalServerErrorException) {
			return true;
		}
		return false;
	}
	
	/**
	 * Handles exception thrown during REST API call or any other
	 * {@link MetaService} unexpected exception.
	 * 
	 * @param e
	 *            the exception to handle
	 */
	public void handleException(final Exception e) {
		if (e instanceof UnauthorizedException) {
			Log.e(TAG, "Stopping service because of "
					+ UnauthorizedException.class.getSimpleName());
			stopSelf();
		}
	}

	/**
	 * Refreshes cached user info.
	 * 
	 * @param intent
	 *            the intent that started the request
	 */
	private void refreshUser(Intent intent) {
		final ResultReceiver receiver =
				intent.getParcelableExtra(EXTRA_CALLBACK);
		try {
			receiver.send(Status.RUNNING, Bundle.EMPTY);
			refreshUserInfo();
			receiver.send(Status.FINISHED, Bundle.EMPTY);
		} catch (Exception e) {
			if (!catchException(e)) {
				final Bundle resultData = new Bundle();
				resultData.putString(EXTRA_ERROR, e.getMessage());
				receiver.send(Status.ERROR, resultData);
				handleException(e);
			}
		}
	}
	
	private void createVolume(Intent intent) {
		// TODO karni: not used yet
	}
	
	private void refreshVolume(Intent intent) {
		// TODO karni: not used yet
	}
	
	private void updateVolume(Intent intent) {
		// TODO karni: not used yet
	}
	
	private void deleteVolume(Intent intent) {
		// TODO karni: not used yet
	}

	/**
	 * Creates a given node resource. Use for directories only. Files should be
	 * created by uploading content.
	 * 
	 * @param intent
	 *            the intent that started the request
	 */
	private void createNode(Intent intent) {
		final ResultReceiver receiver =
				intent.getParcelableExtra(EXTRA_CALLBACK);
		final String resourcePath = intent.getStringExtra(EXTRA_RESOURCE_PATH);
		final String kind = intent.getStringExtra(EXTRA_KIND);
		try {
			receiver.send(Status.RUNNING, Bundle.EMPTY);
			NodeInfo nodeInfo = createNode(resourcePath, kind);
			refreshNode(nodeInfo);
			receiver.send(Status.FINISHED, Bundle.EMPTY);
		} catch (Exception e) {
			if (!catchException(e)) {
				final Bundle resultData = intent.getExtras();
				resultData.putString(EXTRA_ERROR, e.getMessage());
				receiver.send(Status.ERROR, resultData);
			}
		}
	}

	/**
	 * Refreshes a given node resource. If it's a cached file node and there is
	 * a new version of it, its cache is invalidated and file is removed.
	 * 
	 * @param intent
	 *            the intent that started the request
	 */
	private void refreshNode(Intent intent) {
		if (intent.hasExtra(EXTRA_PREFETCH)) {
			Log.d(TAG, "has extra_prefetch");
			prefetchNode(intent);
			return;
		} else {
		}
		
		final ResultReceiver receiver =
				intent.getParcelableExtra(EXTRA_CALLBACK);
		final String resourcePath =
				intent.getStringExtra(Nodes.NODE_RESOURCE_PATH);
		try {
			receiver.send(Status.RUNNING, Bundle.EMPTY);
			refreshNode(resourcePath);
			receiver.send(Status.FINISHED, intent.getExtras());
		} catch (Exception e) {
			if (!catchException(e)) {
				final Bundle resultData = intent.getExtras();
				resultData.putString(EXTRA_ERROR, e.getMessage());
				receiver.send(Status.ERROR, resultData);
			}
		}
	}
	
	private void prefetchNode(Intent intent) {
		final ResultReceiver receiver =
				intent.getParcelableExtra(EXTRA_CALLBACK);
		final String resourcePath =
				intent.getStringExtra(Nodes.NODE_RESOURCE_PATH);
		final String[] projection =
				new String[] { Nodes.NODE_RESOURCE_PATH };
		final Cursor c = MetaUtilities.getChildDirectoriesCursorByResourcePath(
				resourcePath, projection);
		try {
			if (c.moveToFirst()) {
				do {
					if (!sAllowPrefetching) {
						break;
					}
					final String childResourcePath = c.getString(
							c.getColumnIndex(Nodes.NODE_RESOURCE_PATH));
					try {
						Log.d(TAG, "prefetching " + childResourcePath);
						// TODO karni: improve with AbortableRequest
						refreshNode(childResourcePath);
					} catch (Exception e) {
						if (!catchException(e)) {
							final Bundle resultData = intent.getExtras();
							resultData.putString(EXTRA_ERROR, e.getMessage());
						}
					}
				} while (c.moveToNext());
			}
		} finally {
			c.close();
			receiver.send(Status.FINISHED, Bundle.EMPTY);
		}
	}

	/**
	 * Updates a given node resource. Use to rename or (un)publish a file node.
	 * 
	 * @param intent
	 *            the intent that started the request
	 */
	private void updateNode(Intent intent) {
		final ResultReceiver receiver =
			intent.getParcelableExtra(EXTRA_CALLBACK);
		final String resourcePath =
				intent.getStringExtra(Nodes.NODE_RESOURCE_PATH);
		Boolean isPublic = null;
		if (intent.hasExtra(Nodes.NODE_IS_PUBLIC)) {
			isPublic = intent.getBooleanExtra(Nodes.NODE_IS_PUBLIC, false);
		}
		String newPath = null;
		if (intent.hasExtra(Nodes.NODE_PATH)) {
			newPath = intent.getStringExtra(EXTRA_PATH);
		}
		String newName = null;
		if (intent.hasExtra(Nodes.NODE_NAME)) {
			newName = intent.getStringExtra(EXTRA_NAME);
		}
		
		try {
			receiver.send(Status.RUNNING, Bundle.EMPTY);
			NodeInfo nodeInfo = null;
			if (isPublic != null) {
				nodeInfo = changePublicAccess(resourcePath, isPublic);
			} else if (newPath != null){
				nodeInfo = rename(resourcePath, newPath);
				if (nodeInfo != null) {
					// This is special case when we update resourcePath, which
					// is our unique key. Thus, need to update manually.
					final String newResourcePath = nodeInfo.getResourcePath();
					MetaUtilities.updateStringField(resourcePath,
							Nodes.NODE_RESOURCE_PATH, newResourcePath);
					
					// Rename the file, if cached.
					String path = FileUtilities
							.getFilePathFromResourcePath(resourcePath);
					path = path.substring(0, path.lastIndexOf('/'));
					
					final String oldName = resourcePath.substring(
							resourcePath.lastIndexOf('/') + 1);
					
					// This changes filename in place, mv not supported yet. 
					final File oldFile = new File(path, oldName);
					if (oldFile.exists()) {
						final File newFile = new File(path, newName);
						Log.d(TAG, "Renaming cached file "
								+ oldFile + " to " + newFile);
						oldFile.renameTo(newFile);
						// Update the cache.
						if (newFile.isFile()) {
							final String newFileData =
									FileUtilities.getFileSchemeFromResourcePath(
											newResourcePath);
							MetaUtilities.updateStringField(newResourcePath,
									Nodes.NODE_DATA, newFileData);
						}
						
					}
				}
			}
			if (nodeInfo != null) {
				refreshNode(nodeInfo);
				MetaUtilities.notifyChange(Nodes.CONTENT_URI);
			}
			receiver.send(Status.FINISHED, Bundle.EMPTY);
		} catch (Exception e) {
			if (!catchException(e)) {
				final Bundle resultData = intent.getExtras();
				resultData.putString(EXTRA_ERROR, e.getMessage());
				receiver.send(Status.ERROR, resultData);
			}
		}
	}

	/**
	 * Deletes a given node resource. On success, deletes cached entry.
	 * 
	 * @param intent
	 *            the intent that started the request
	 */
	private void deleteNode(Intent intent) {
		final ResultReceiver receiver =
				intent.getParcelableExtra(EXTRA_CALLBACK);
		final String resourcePath =
				intent.getStringExtra(EXTRA_RESOURCE_PATH);
		try {
			receiver.send(Status.RUNNING, Bundle.EMPTY);
			MetaUtilities.setState(resourcePath, ResourceState.STATE_DELETING);
			MetaUtilities.notifyChange(Nodes.CONTENT_URI);
			RestApi.deleteNode(mHttpClient, mConsumer, resourcePath);
			MetaUtilities.deleteByResourcePath(resourcePath);
			MetaUtilities.notifyChange(Nodes.CONTENT_URI);
			receiver.send(Status.FINISHED, Bundle.EMPTY);
		} catch (Exception e) {
			if (e instanceof NotFoundException) {
				MetaUtilities.deleteByResourcePath(resourcePath);
			}
			if (!catchException(e)) {
				MetaUtilities.setState(resourcePath, null);
				final Bundle resultData = intent.getExtras();
				resultData.putString(EXTRA_ERROR, e.getMessage());
				receiver.send(Status.ERROR, resultData);
				handleException(e);
			}
		}
	}
	
	private boolean isPictureUploadUdfAvailable() {
		if (!NetworkReceiver.isConnected()) {
			return false;
		}
		
		// Check if UDF exists.
		final String resourcePath = Preferences.getPhotosUploadResource();
		Log.d(TAG, "checking if already exists: " + resourcePath);
		try {
			final NodeInfo nodeInfo = RestApi.getNodeInfo(mHttpClient,
					mConsumer, resourcePath, false);
			if (nodeInfo != null) {
				// UDF exists.
				return true;
			}
		} catch (NotFoundException e) {
			try {
				if (resourcePath.startsWith(Preferences.U1_RESOURCE)) {
					final NodeInfo node = createNode(resourcePath, NodeInfo.DIRECTORY);
					if (node != null) {
						refreshNode(node);
						return true;
					}
				} else {
					final VolumeInfo vol = createVolume(resourcePath);
					if (vol != null) {
						// We need to insert UDF manually, as it's name is not
						// the last part of the path, but the full path.
						// UDF names are not updated when UDF-root node is refreshed.
						final ContentValues values = new ContentValues();
						values.put(Nodes.NODE_RESOURCE_PATH, resourcePath);
						values.put(Nodes.NODE_KIND, Nodes.KIND_DIRECTORY);
						final String path = resourcePath.substring(1);
						values.put(Nodes.NODE_PATH, path);
						values.put(Nodes.NODE_NAME, path);
						getContentResolver().insert(Nodes.CONTENT_URI, values);
						
						// We need the parents content path for uploads, so refresh
						// rest of the details.
						final NodeInfo nodeInfo = RestApi.getNodeInfo(mHttpClient,
								mConsumer, resourcePath, false);
						refreshNode(nodeInfo);
						Log.d(TAG, "UDF created");
						return true;
					}
				}
			} catch (Exception eek) {
				Log.e(TAG, "Can't create picture upload UDF, " +
						"will try again later.", eek);
			}
		} catch (Exception e) {
			Log.d(TAG, "failed to check UDF");
		}
		return false;
	}
	
	private void sharePicture(Intent intent) {
		if (isPictureUploadUdfAvailable()) {
			final Uri uri = intent.getParcelableExtra(Intent.EXTRA_STREAM);
			UpDownServiceHelper.upload(
					this, uri, Preferences.getPhotosUploadResource(), false);
		} else {
			// TODO karni: add "Failed to share. Tap to retry." notification.
		}
	}
	
	private void uploadPictures(Intent intent) {
		if (!NetworkReceiver.isConnectedPreferred(this)) {
			Log.d(TAG, "Not uploading photos, not preferred/no network");
			return;
		}
		
		// Check last photo upload timestamp (saved as 'epoch' seconds).
		long lastTimestamp; 
		long lastPhotoUpload;
		try {
			lastTimestamp = StorageInfo.getLastUploadedPhotoTimestamp();
			lastPhotoUpload = intent.getLongExtra(EXTRA_TIMESTAMP,
					lastTimestamp);
		} catch (StorageInfo.StorageNotAvailable e) {
			UIUtil.showToast(this, R.string.toast_storage_is_not_accessible);
			Log.e(TAG, "Getting last upload timestamp.", e);
			return;
		}
		long now = System.currentTimeMillis() / 1000;
		Log.d(TAG, "Diff from last photo upload in seconds: "
				+ (now - lastPhotoUpload));
		
		// Check how many photos should be uploaded.
		int count = MediaUtilities.countImages2Sync(this, lastPhotoUpload);
		Log.d(TAG, "photos to upload: " + count);
		if (count == 0) {
			return;
		}
		
		// First check if we should be uploading anything.
		if (Preferences.getAutoUploadPhotos()) {
			// Only then check if target directory exists. 
			if (isPictureUploadUdfAvailable()) {
				// We have photos to upload, so let's do it.
				UpDownService.setIsUploadingPhotoMedia(true);
				uploadImages(this, lastPhotoUpload);
			}
		}
	}
	
	private void retryFailed(Intent intent) {
		// We may filter this in the future to retry only auto-uploads or
		// only given types of transfers, files, etc.
		retryFailed();
	}
	
	/**
	 * Gets user info and updates cached details. Delegates rebuilding
	 * UDF root nodes to {@link MetaService#updateNodeFromScratch(String)}, so
	 * that we don't fetch all the volumes (we don't need to at this stage).
	 */
	public void refreshUserInfo() throws Exception {
		Log.d(TAG, "updating user info");
		UserInfo userInfo = RestApi.getUserInfo(mHttpClient, mConsumer);
		Preferences.updateAccountInfo(userInfo.getVisibleName(),
				FileUtilities.getHumanReadableSize(userInfo.getMaxBytes()),
				userInfo.getMaxBytes(),
				userInfo.getUsedBytes());
		final List<String> userNodePaths = userInfo.getUserNodePaths();
		userNodePaths.add(userInfo.getRootNodePath());
		refreshVolumesFromUserNodePaths(userNodePaths);
	}
	
	private VolumeInfo createVolume(final String resourcePath)
			throws Exception {
		Log.d(TAG, "creating volume: " + resourcePath);
		return RestApi.createVolume(mHttpClient, mConsumer, resourcePath);
	}
	
	
	private NodeInfo createNode(final String resourcePath, final String kind)
			throws Exception {
		if (Nodes.KIND_FILE.equals(kind)) {
			Log.e(TAG, "createNode called with kind = KIND_FILE. " +
					"Upload the file instead.");
		}
		Log.d(TAG, "creating node: " + resourcePath);
		return RestApi.createDirectory(mHttpClient, mConsumer, resourcePath);
	}
	
	public void refreshVolumesFromUserNodePaths(List<String> volumePaths)
			throws Exception {
		// Get cached user node paths.
		final Set<String> userNodePaths = MetaUtilities.getUserNodePaths();
		NodeInfo v;
		// Insert or update volume root nodes.
		for (String volume : volumePaths) {
			v = RestApi.getNodeInfo(mHttpClient, mConsumer, volume, false);
			refreshNode(v);
			userNodePaths.remove(v.getResourcePath());
		}
		// Invalidate cache of deleted volumes.
		for (String volume : userNodePaths) {
			MetaUtilities.deleteByResourcePath(volume);
			// Web UI says: "Are you sure [...]? This will NOT delete files on your computer."
			/*
			final String path = Preferences.getBaseDirectory() + "/" +
					FileUtilities.getFilePathFromResourcePath(volume);
			FileUtilities.remove(path, true);
			*/
		}
	}

	/**
	 * Given parents resource path and {@link ArrayList} of {@link NodeInfo}s of
	 * its children, syncs cached info of these children. Updating children in
	 * one method enables us to make use of database transaction.<br />
	 * <ul>
	 * <li>- inserts if child is new</li>
	 * <li>- updates if child has changed [thus marks is_cached = false]</li>
	 * <li>- deletes if child is missing [dead node]</li>
	 * </ul>
	 * 
	 * @param parentResourcePath
	 *            the resource path of childrens parent
	 * @param children
	 *            {@link NodeInfo}s of the parents children
	 * @throws OperationApplicationException 
	 * @throws RemoteException 
	 */
	public void updateNodeChildren(final String parentResourcePath,
			final ArrayList<NodeInfo> children) throws RemoteException,
			OperationApplicationException {
		// Query id's of all children to detect dead nodes.
		final Set<Integer> childrenIds =
				MetaUtilities.getChildrenIds(parentResourcePath);
		
		final ArrayList<ContentProviderOperation> operations =
				new ArrayList<ContentProviderOperation>();
		
		// Update or insert child nodes.
		final String[] projection = new String[] { Nodes._ID,
				Nodes.NODE_RESOURCE_PATH, Nodes.NODE_GENERATION };
		final String selection = Nodes.NODE_RESOURCE_PATH + "=?";
		
		for (NodeInfo child : children) {			
			final String[] selectionArgs =
					new String[] { child.getResourcePath() };
			final Cursor c = mResolver.query(Nodes.CONTENT_URI, projection,
					selection, selectionArgs, null);
			try {
				ContentValues values = Nodes.valuesFromRepr(child);
				if (c.moveToFirst()) {
					final int id = c.getInt(c.getColumnIndex(Nodes._ID));
					// Node is live.
					childrenIds.remove(id);
					
					// Update node.
					final long generation =
							c.getLong(c.getColumnIndex(Nodes.NODE_GENERATION));
					final long newGeneration = child.getGeneration();
					if (generation < newGeneration) {
						Log.v(TAG, "updating child node, new generation");
						values.put(Nodes.NODE_IS_CACHED, false);
						values.put(Nodes.NODE_DATA, "");
						
						Uri uri = MetaUtilities.buildNodeUri(id);
						ContentProviderOperation op = ContentProviderOperation
								.newUpdate(uri)
								.withValues(values)
								.build();
						operations.add(op);
					} else {
						Log.v(TAG, "child up to date");
					}
				} else {
					// Insert node.
					Log.v(TAG, "inserting child");
					ContentProviderOperation op = ContentProviderOperation
							.newInsert(Nodes.CONTENT_URI)
							.withValues(values)
							.build();
					operations.add(op);
				}
			} finally {
				c.close();
			}
		}
		
		// Remove nodes, which ids are left in childrenIds set.
		if (!childrenIds.isEmpty()) {
			Log.v(TAG, "childrenIDs not empty: " + childrenIds.size());
			final Iterator<Integer> it = childrenIds.iterator();
			while (it.hasNext()) {
				int id = it.next();
				Uri uri = MetaUtilities.buildNodeUri(id);
				ContentProviderOperation op = ContentProviderOperation
						.newDelete(uri).build();
				operations.add(op);
			}
		} else {
			Log.v(TAG, "childrenIDs empty");
		}
		
		long then = System.currentTimeMillis();
		mResolver.applyBatch(MetaContract.CONTENT_AUTHORITY, operations);
		long now = System.currentTimeMillis();
		Log.d(TAG, "time to update children: " + (now-then));
		mResolver.notifyChange(Nodes.CONTENT_URI, null);
	}
	
	/**
	 * Refresh the node cache from response.
	 * 
	 * @param nodeInfo
	 */
	public void refreshNode(NodeInfo nodeInfo) {
		final String resourcePath = nodeInfo.getResourcePath();
		boolean isDirectory = Nodes.KIND_DIRECTORY.equals(nodeInfo.getKind());
		boolean dirHasUpdates = false;
		
		final String[] projection =
				new String[] { Nodes.NODE_NAME, Nodes.NODE_IS_CACHED };
		final String selection = Nodes.NODE_RESOURCE_PATH + "=?";
		final String[] selectionArgs =
			new String[] { resourcePath };
		
		final Cursor c = mResolver.query(Nodes.CONTENT_URI, projection,
				selection, selectionArgs, null);
		try {
			ContentValues values = Nodes.valuesFromRepr(nodeInfo);
			if (c.moveToFirst()) {
				// Update node.
				if (isDirectory) {
					Log.v(TAG, "updating dir node");
					dirHasUpdates = true;
					mResolver.update(Nodes.CONTENT_URI,
							values, selection, selectionArgs);
				} else {
					Log.v(TAG, "updating file node");
					mResolver.update(Nodes.CONTENT_URI,
							values, selection, selectionArgs);
				}
			} else {
				// Insert node.
				if (isDirectory || (!isDirectory && nodeInfo.getSize() != null)) {
					Log.v(TAG, "inserting node");
					dirHasUpdates = true;
					mResolver.insert(Nodes.CONTENT_URI, values);
				}
			}
			mResolver.notifyChange(Nodes.CONTENT_URI, null);
		} finally {
			c.close();
		}
		
		if (isDirectory && dirHasUpdates) {
			final ArrayList<NodeInfo> children =
					(ArrayList<NodeInfo>) nodeInfo.getChildren();
			if (children != null) {
				try {
					updateNodeChildren(resourcePath, children);
					// We have successfully cached the content for this generation.
					MetaUtilities.setIsCached(resourcePath, true);
				} catch (RemoteException e) {
					handleException(e);
				} catch (OperationApplicationException e) {
					// Dir caching interrupted.
					MetaUtilities.setIsCached(resourcePath, false);
					handleException(e);
				}
			}
		}
	}
	
	private void refreshNode(final Cursor c)
			throws Exception {
		String resourcePath = null;
		String hash = null;
		boolean isDirectory = false;
		boolean includeChildren = false;
		try {
			if (c.moveToFirst()) {
				resourcePath = c.getString(
						c.getColumnIndex(Nodes.NODE_RESOURCE_PATH));
				String kind = c.getString(
						c.getColumnIndex(Nodes.NODE_KIND));
				hash = c.getString(
						c.getColumnIndex(Nodes.NODE_HASH));
				isDirectory = Nodes.KIND_DIRECTORY.equals(kind);
				if (isDirectory) {
					// The update request comes from a cursor. Thus, we assume
					// it comes from the list, so we want to update contents
					// of the directory as well.
					includeChildren = true;
				}
			} else {
				Log.w(TAG, "entry missing in provider!");
				return;
			}
		} finally {
			c.close();
		}
		
		Log.v(TAG, "resourcePath: " + resourcePath
				+ ", includeChildren: " + includeChildren);
		NodeInfo nodeInfo = RestApi.getNodeInfo(mHttpClient, mConsumer,
				resourcePath, includeChildren);
		final boolean hasChanges = (hash == null
				|| (hash != null && !hash.equals(nodeInfo.getHash())) );
		Log.d(TAG, "hasChanged=" + hasChanges
				+ ", includeChildren=" + includeChildren);
		
		if (nodeInfo != null && hasChanges) {
			refreshNode(nodeInfo);
		}
	}
	
	public void refreshNode(final String resourcePath) throws Exception {
		final String selection = Nodes.NODE_RESOURCE_PATH + "=?";
		final String[] selectionArgs = new String[] { resourcePath };
		final Cursor c = mResolver.query(Nodes.CONTENT_URI,
				Nodes.getDefaultProjection(), selection, selectionArgs, null);
		refreshNode(c);
	}
	
	public NodeInfo changePublicAccess(final String resourcePath,
			final boolean isPublic) throws Exception {
		Log.d(TAG, "changing public acess: " + resourcePath + " " + isPublic);
		return RestApi.changePublicAccess(
				mHttpClient, mConsumer, resourcePath, isPublic);
	}

	public NodeInfo rename(final String resourcePath, final String newPath)
			throws Exception {
		Log.d(TAG, "renaming: " + resourcePath + " to path " + newPath);
		return RestApi.rename(
				mHttpClient, mConsumer, resourcePath, newPath);
	}
	
	public void uploadImages(Context context, long since) {
		if (Preferences.getAutoUploadPhotos()) {
			uploadMedia(context, since,
					MediaStore.Images.Media.EXTERNAL_CONTENT_URI);
		}
	}
	
	private void uploadMedia(Context context, long since, Uri uri) {
		// TODO karni: should grab a wake lock here already before UpDown starts transfers?
		Thread uploadThread = new Thread(new UploadMediaRunnable(since, uri));
		uploadThread.setPriority(Thread.MIN_PRIORITY);
		uploadThread.start();
	}
	
	private class UploadMediaRunnable implements Runnable {
		
		private long since = 0L;
		private Uri uri;
		
		public UploadMediaRunnable(long since, Uri uri) {
			this.since = since;
			this.uri = uri;
		}

		public void run() {
			final Context context = UbuntuOneFiles.getInstance();
			
			final String[] projection =
				new String[] { MediaColumns._ID, MediaColumns.DATE_ADDED,
					MediaColumns.SIZE, MediaColumns.DATA };
			final String selection = MediaColumns.DATE_ADDED + ">?";
			final String[] selectionArgs = new String[] { String.valueOf(since) };
		
			final Cursor c = context.getContentResolver()
					.query(uri, projection, selection, selectionArgs, null);
			String id; // Let's fetch a string already.
			long dateAdded;
			
			try {
				final File dcimDir = new File(
						Environment.getExternalStorageDirectory(), "DCIM");
				final String dcimDirPath = dcimDir.getPath().toLowerCase();
				
				if (dcimDir.exists() && c != null && c.moveToFirst()) {
					do {
						// Ignore files of 0 bytes size.
						final long size =
								c.getLong(c.getColumnIndex(MediaColumns.SIZE));
						if (size == 0) {
							continue;
						}
						
						final String data =
								c.getString(c.getColumnIndex(MediaColumns.DATA));
						
						// Ignore files not under DCIM.
						if (!data.toLowerCase().startsWith(dcimDirPath)) {
							Log.i(TAG, "Ignoring: " + data + ", not under " + dcimDirPath);
							continue;
						}
						
						// Ignore non-existent files (verify MediaStore cache)
						final File file = new File(data);
						if (!file.exists()) {
							Log.i(TAG, "Invalid entry from MediaStore, ignoring.");
							continue;
						}
						
						id = c.getString(c.getColumnIndex(MediaColumns._ID));
						dateAdded = c.getLong(c.getColumnIndex(MediaColumns.DATE_ADDED));
						Uri newUri = uri.buildUpon().appendPath(id).build();
						Log.d(TAG, "will upload media from " + newUri.toString());
						// We assume here all grouped media uploads are auto-transfers.
						UpDownServiceHelper.upload(context, newUri,
								Preferences.getPhotosUploadResource(), dateAdded,
								true);
					} while (c.moveToNext());
				}
			} finally {
				if (c != null)
					c.close();
			}
		}
		
	}
	
	private void retryFailed() {
		if (!NetworkReceiver.isConnectedPreferred(this)) {
			Log.d(TAG, "Not retrying failed, not preferred/no network.");
			return;
		}
		
		final Cursor c = MetaUtilities.getFailedTransfers();
		String resourcePath;
		String resourceState;
		String parentResourcePath;
		String data;
		try {
			if (c.moveToFirst() && NetworkReceiver.isConnectedPreferred(this)) {
				do {
					resourcePath = c.getString(
							c.getColumnIndex(Nodes.NODE_RESOURCE_PATH));
					resourceState = c.getString(
							c.getColumnIndex(Nodes.NODE_RESOURCE_STATE));
					parentResourcePath = c.getString(
							c.getColumnIndex(Nodes.NODE_PARENT_PATH));
					data = c.getString(
							c.getColumnIndex(Nodes.NODE_DATA));
					// Validate data column content.
					if (data == null || (data != null && data.equals(""))) {
						Log.e(TAG, "Data Uri missing: " + data);
						MetaUtilities.deleteByResourcePath(resourcePath);
						continue;
					}
					// Check if the file is still there.
					final URI uri = URI.create(Uri.encode(data, ":/"));
					final File file = new File(uri);
					if (!file.exists()) {
						Log.i(TAG, "File is gone, ignoring and removing.");
						MetaUtilities.deleteByResourcePath(resourcePath);
						continue;
					}
					
					if (ResourceState.STATE_POSTING_FAILED.equals(
							resourceState)) {
						retryUpload(parentResourcePath, data);
					} else if (ResourceState.STATE_GETTING_FAILED.equals(
							resourceState)) {
						retryDownload(resourcePath);
					}
				} while (c.moveToNext());
			}
		} finally {
			c.close();
		}
	}
	
	private void retryUpload(String parent, String data) {
		final Uri uri = Uri.parse(Uri.encode(data, ":/"));
		Log.d(TAG, "will retry: " + data);
		UpDownServiceHelper.upload(this, uri, parent, true);
	}
	
	private void retryDownload(String resourcePath) {
		UpDownServiceHelper.download(this, resourcePath, true);
	}

}
