/*
 * Ubuntu One Files - access Ubuntu One cloud storage on Android platform.
 * 
 * Copyright (C) 2011 Canonical Ltd.
 * Author: Michał Karnicki <michal.karnicki@canonical.com>
 *   
 * This file is part of Ubuntu One Files.
 *  
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *  
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses 
 */

package com.ubuntuone.android.files.service;

import java.util.List;
import java.util.Queue;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;

import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;

import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.content.res.Resources;
import android.text.format.Time;
import android.widget.ProgressBar;
import android.widget.RemoteViews;

import com.ubuntuone.android.files.R;
import com.ubuntuone.android.files.activity.FilesActivity;
import com.ubuntuone.android.files.provider.MetaUtilities;
import com.ubuntuone.android.files.service.UpDownService.RunnableTransferTask;
import com.ubuntuone.rest.util.IOUtil.Callback;

/**
 * Single threaded {@link RunnableTransferTask} {@link Executor}.<br />
 * Handles ongoing and completed notifications on it's own for each type of
 * transfers:<br />
 * - uploads ({@link HttpPost#METHOD_NAME} method) and<br />
 * - downloads ({@link HttpGet#METHOD_NAME} method).
 */
public class TransferExecutor implements Executor {
	private final Queue<Runnable> mTasks =
			new LinkedBlockingQueue<Runnable>();
	private final ExecutorService mExecutor;
	private final SimpleCallback mDone;
	private int mTasksSubmitted = 0;
	private int mTasksProcessed = 0;
	private int mTasksFailed = 0;
	
	private final Context mContext;
	private int mOngoingNotificationId;
	private int mOngoingIconResId;
	private int mOngoingTextResId;
	private int mCompletedNotificationId;
	private int mCompletedTextResId;
	private int mFailedNotificationId;
	private int mFailedTextResId;
	
	/** Rate limit displaying notifications ticker of same transfer type. */
	private final long RATE_LIMIT_MS = 60 * 1000L;
	/** Rate limit progress notification {@link ProgressBar} updates. */
	private final long RATE_LIMIT_PROGRESS_MS = 1700;

	private final String mMethod;
	private final NotificationManager mNotificationManager;
	private final Callback mProgress;
	private Notification mNotification;
	private PendingIntent mPendingIntent;
	private RemoteViews mNotificationViews;
	private long mLastNotifiedTimestamp = 0L;
	private Runnable mActiveTransfer;
	private long mActiveSize;
	
	public TransferExecutor(Context context, String method, SimpleCallback done) {
		this.mContext = context;
		this.mExecutor = Executors.newSingleThreadExecutor();
		this.mDone = done;
		this.mActiveTransfer = null;
		this.mMethod = method;
		
		final Intent intent = new Intent(context, FilesActivity.class);
		this.mPendingIntent = PendingIntent.getActivity(mContext, 0, intent,
				Intent.FLAG_ACTIVITY_NEW_TASK);
		
		this.mNotificationManager = (NotificationManager) mContext
				.getSystemService(Context.NOTIFICATION_SERVICE);
		this.mNotificationViews = new RemoteViews(mContext.getPackageName(),
				R.layout.notification_progress);
		
		this.mProgress = new Callback() {
			long lastTimeNotified = 0;
			
			public void callback(long progress) {
				final long now = System.currentTimeMillis();
				if (lastTimeNotified + RATE_LIMIT_PROGRESS_MS < now
						&& mNotification != null) {
					lastTimeNotified = now;
//					final int percent =
//							(int) ((progress/((float) mActiveSize)) * 100);
					
					//				final String progressBarText = String.format(
					//						"%1$d%% (%2$s of %3$s)", percent,
					//						FileUtilities.getHumanReadableSize(progress),
					//						FileUtilities.getHumanReadableSize(mCurrentSize));
					//				mNotificationViews.setString(
					//						R.id.progress, "setText", progressBarText);
					mNotificationViews.setProgressBar(
							R.id.progress, (int) mActiveSize, (int) progress, false);
					mNotificationManager.notify(
							mOngoingNotificationId, mNotification);
				}
			}
		};
		
		if (HttpPost.METHOD_NAME.equals(method)) {
			mOngoingNotificationId = R.id.stat_ongoing_upload_id;
			mOngoingTextResId = R.string.uploading_file_n_of_n;
			mOngoingIconResId = R.drawable.stat_sys_upload;
			mCompletedNotificationId = R.id.stat_completed_upload_id;
			mCompletedTextResId = R.plurals.uploaded_n_files;
			mFailedNotificationId = R.id.stat_failed_upload_id;
			mFailedTextResId = R.plurals.failed_to_upload_n_files;
		} else if (HttpGet.METHOD_NAME.equals(method)) {
			mOngoingNotificationId = R.id.stat_ongoing_download_id;
			mOngoingTextResId = R.string.downloading_file_n_of_n;
			mOngoingIconResId = R.drawable.stat_sys_download;
			mCompletedNotificationId = R.id.stat_completed_download_id;
			mCompletedTextResId = R.plurals.downloaded_n_files;
			mFailedNotificationId = R.id.stat_failed_download_id;
			mFailedTextResId = R.plurals.failed_to_download_n_files;
		}
	}

	public synchronized void execute(final Runnable transfer) {
		final RunnableTransferTask task = (RunnableTransferTask) transfer;
		mTasksSubmitted++;
		if (mNotification != null) {
			// Update transfer count.
			showProgressNotification(false);
		}
		mTasks.offer(new Runnable() {
			public void run() {
				try {
					// Get task size for ProgressBar updates.
					mActiveSize = task.getSize();
					// Set the notification ProgressBar callback.
					task.setProgressCallback(mProgress);
					// Show the user which file transfer is running.
					mTasksProcessed++;
					showProgressNotification(true);
					
					transfer.run();
				} finally {
					// karni: It would be nicer to make run() throw.
					if (task.mException != null) {
						mTasksFailed++;
					}
					scheduleNext();
				}
			}
		});
		if (mActiveTransfer == null) {
			scheduleNext();
		}
	}
	
	protected synchronized void scheduleNext() {
		if ((mActiveTransfer = mTasks.poll()) != null) {
			mExecutor.execute(mActiveTransfer);
		} else {
			hideProgressNotification();
			showCompletedTransfersNotification();
			showFailedTransfersNotification();
			mDone.callback();
		}
	}
	
	public boolean isRunning() {
		return mActiveTransfer != null;
	}
	
	public int getQueueSize() {
		return mTasks.size();
	}
	
	public int getTasksSubmitted() {
		return mTasksSubmitted;
	}
	
	public int getTasksProcessed() {
		return mTasksProcessed;
	}
	
	public int getTasksCompleted() {
		return mTasksProcessed - mTasksFailed;
	}
	
	public int getTasksFailed() {
		return mTasksFailed;
	}
	
	public synchronized int cancelAll() {
		final List<Runnable> pending = mExecutor.shutdownNow();
		mTasksFailed += pending.size();
		mTasks.clear();
		mActiveTransfer = null;
		mNotificationManager.cancel(mOngoingNotificationId);
		return MetaUtilities.resetFailedTransfers(mMethod);
	}
	
	/**
	 * Shows current transfer {@link Notification} with {@link ProgressBar}.
	 * One must update {@link TransferExecutor#mActiveSize} with transfer size
	 * before this method call.
	 * 
	 * @param showTicker
	 *            true, if ticker should be shown. If the {@link Notification}
	 *            only needs to be updated, this should be false.
	 */
	private synchronized void showProgressNotification(boolean showTicker) {
		// Prepare notification text.
		final String subtitleText = mContext.getString(mOngoingTextResId,
				mTasksProcessed, mTasksSubmitted);
		
		// Rate limit. Control ticker display.
		final long now = System.currentTimeMillis();
		if (mLastNotifiedTimestamp + RATE_LIMIT_MS < now) {
			mLastNotifiedTimestamp = now;
		} else {
			showTicker = false;
		}
		
		final RemoteViews views = mNotificationViews;
		// Is it the first time or we want to show the ticker?
		if (mNotification == null || showTicker) {
			mNotification = new Notification(
					mOngoingIconResId, subtitleText, now);
			views.setImageViewResource(R.id.icon, mOngoingIconResId);
			views.setTextViewText(R.id.title, "Ubuntu One");
			final Time time = new Time();
			time.setToNow();
			final String formattedTime = time.format("%H:%M");
			views.setTextViewText(R.id.time, formattedTime);
			mNotification.contentView = views;
			
			mNotification.flags |=
					Notification.FLAG_ONGOING_EVENT |
					Notification.FLAG_NO_CLEAR |
					Notification.FLAG_ONLY_ALERT_ONCE;
		}
		views.setTextViewText(R.id.subtitle, subtitleText);
		views.setProgressBar(R.id.progress, 100, 0, false);
		
		// Display the notification.
		final Intent notificationIntent =
				new Intent(mContext, FilesActivity.class);
		final PendingIntent contentIntent =
				PendingIntent.getActivity(mContext, 0, notificationIntent, 0);
		mNotification.contentIntent = contentIntent;
		
		mNotificationManager.notify(mOngoingNotificationId, mNotification);
		/* karni: I would like to use startForeground() here, but:
		 * - calling startForeground() without setting FLAG_ONGOING_EVENT flag
		 *   causes the second notification to be intermittent, not ongoing
		 * - if we call stopForeground() for one type of transfers,
		 *   the service is probably not in fg any longer. Why? Cuz the second
		 *   call to stopForeground() does not dismiss the second notification,
		 *   since the service is no longer in foreground (d'oh).
		 */
	}
	
	private void hideProgressNotification() {
		mNotificationManager.cancel(mOngoingNotificationId);
	}
	
	private synchronized void showCompletedTransfersNotification() {
		final int tasksCompleted = mTasksProcessed - mTasksFailed;
		if (tasksCompleted == 0) {
			return;
		}
		
		// Prepare notification text.
		final Resources r = mContext.getResources();
		final String notificationText = r.getQuantityString(
				mCompletedTextResId, tasksCompleted, tasksCompleted);
		
		// Display the notification.
		final long now = System.currentTimeMillis();
		final Notification notification = new Notification(
				R.drawable.ic_stat_launcher, notificationText, now);
		notification.flags += Notification.FLAG_AUTO_CANCEL;
		notification.setLatestEventInfo(
				mContext, "Ubuntu One", notificationText, mPendingIntent);
		mNotificationManager.notify(mCompletedNotificationId, notification);
	}
	
	private synchronized void showFailedTransfersNotification() {
		if (mTasksFailed == 0) {
			return;
		}
		
		// Prepare notification text.
		final Resources r = mContext.getResources();
		final String notificationText = r.getQuantityString(
				mFailedTextResId, mTasksFailed, mTasksFailed);
		
		// Display the notification.
		final long now = System.currentTimeMillis();
		// TODO karni: This needs new icons from Design team.
		final Notification notification = new Notification(
				R.drawable.ic_stat_launcher, notificationText, now);
		notification.flags += Notification.FLAG_AUTO_CANCEL;
		notification.setLatestEventInfo(
				mContext, "Ubuntu One", notificationText, mPendingIntent);
		mNotificationManager.notify(mFailedNotificationId, notification);
	}
}
