/*
 * Ubuntu One Files - access Ubuntu One cloud storage on Android platform.
 * 
 * Copyright (C) 2011 Canonical Ltd.
 * Author: Michał Karnicki <michal.karnicki@canonical.com>
 *   
 * This file is part of Ubuntu One Files.
 *  
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *  
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses 
 */

package com.ubuntuone.android.files.service;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.net.URI;
import java.net.UnknownHostException;
import java.text.ParseException;

import oauth.signpost.OAuthConsumer;
import oauth.signpost.exception.OAuthCommunicationException;
import oauth.signpost.exception.OAuthExpectationFailedException;
import oauth.signpost.exception.OAuthMessageSignerException;

import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.json.JSONException;

import android.content.ContentResolver;
import android.content.Context;
import android.content.Intent;
import android.media.MediaScannerConnection;
import android.net.Uri;
import android.os.Bundle;
import android.os.IBinder;
import android.os.ResultReceiver;

import com.ubuntuone.android.files.Alarms;
import com.ubuntuone.android.files.Preferences;
import com.ubuntuone.android.files.UbuntuOneFiles;
import com.ubuntuone.android.files.provider.MetaUtilities;
import com.ubuntuone.android.files.provider.MetaContract.Nodes;
import com.ubuntuone.android.files.provider.MetaContract.ResourceState;
import com.ubuntuone.android.files.receiver.NetworkReceiver;
import com.ubuntuone.android.files.service.MetaService.Status;
import com.ubuntuone.android.files.util.HttpManager;
import com.ubuntuone.android.files.util.Authorizer;
import com.ubuntuone.android.files.util.Log;
import com.ubuntuone.rest.RestApi;
import com.ubuntuone.rest.Exceptions.BadRequestException;
import com.ubuntuone.rest.Exceptions.ExpectationFailedException;
import com.ubuntuone.rest.Exceptions.ForbiddenException;
import com.ubuntuone.rest.Exceptions.InternalServerErrorException;
import com.ubuntuone.rest.Exceptions.MethodNotAllowedException;
import com.ubuntuone.rest.Exceptions.NotFoundException;
import com.ubuntuone.rest.Exceptions.NotImplementedException;
import com.ubuntuone.rest.Exceptions.RequestTimeOutException;
import com.ubuntuone.rest.Exceptions.ServiceUnavailableException;
import com.ubuntuone.rest.Exceptions.UnauthorizedException;
import com.ubuntuone.rest.resources.NodeInfo;
import com.ubuntuone.rest.util.IOUtil.Callback;

public class UpDownService extends AwakeService {
	private static final String TAG = UpDownService.class.getSimpleName();

	private static final String BASE = "com.ubuntuone.android.files.updown";
	public static final String ACTION_UPLOAD = BASE + ".ACTION_UPLOAD";
	public static final String ACTION_DOWNLOAD = BASE + ".ACTION_DOWNLOAD";
	public static final String ACTION_CANCEL_ALL = BASE + ".ACTION_CANCEL_ALL";

	public static final String EXTRA_CALLBACK = "extra_callback";
	public static final String EXTRA_AUTO_TRANSFER = "extra_auto_transfer";
	public static final String EXTRA_ERROR = "extra_error";
	public static final String EXTRA_METHOD = "extra_method";
	public static final String EXTRA_RESOURCE_PATH = Nodes.NODE_RESOURCE_PATH;
	public static final String EXTRA_CONTENT_PATH = Nodes.NODE_CONTENT_PATH;
	public static final String EXTRA_PARENT = Nodes.NODE_PARENT_PATH;
	public static final String EXTRA_NAME = Nodes.NODE_NAME;
	// L8R karni: This could be used to cancel the download if the response
	// header contains same content SHA1 hash.
	public static final String EXTRA_HASH = Nodes.NODE_HASH; 
	public static final String EXTRA_SIZE = Nodes.NODE_SIZE;
	public static final String EXTRA_MIME = Nodes.NODE_MIME;
	public static final String EXTRA_DATA = Nodes.NODE_DATA;
	
	private static final String PARTIAL_SUFFIX = ".part";
	
	private static boolean sIsWorking = false;
	private static boolean sIsUploadingPhotoMedia = false;
	private static boolean sIsUploadingVideoMedia = false;
	private static boolean sIsUploadingAudioMedia = false;

	private HttpClient mHttpClient;
	private static OAuthConsumer mConsumer;

	private TransferExecutor mUploads;
	private TransferExecutor mDownloads;

	final SimpleCallback mCompletionCallback = new SimpleCallback() {
		public synchronized void callback() {
			if (!mUploads.isRunning() && !mDownloads.isRunning()) {
				Log.d(TAG, "UpDownService is done.");
				stopSelf();
			}
		}
	};

	@Override
	public void onCreate() {
		AwakeService.TAG = TAG;
		super.onCreate();
		Log.i(TAG, "Starting Ubuntu One UpDownService.");
		sIsWorking = true;
		
		try {
			mConsumer = Authorizer.getInstance(false);
		} catch (Exception e) {
			Log.e(TAG, "Auth client failure", e);
			stopSelf();
			return;
		}
		mHttpClient = HttpManager.getClient();
		
		mUploads = new TransferExecutor(this, HttpPost.METHOD_NAME,
				mCompletionCallback);
		mDownloads = new TransferExecutor(this, HttpGet.METHOD_NAME,
				mCompletionCallback);
	}

	@Override
	public int onStartCommand(Intent intent, int flags, int startId) {
		Log.d(TAG, "onStartCommand() " + intent + " " + startId);
		
		if (!NetworkReceiver.isConnected()) {
			return START_NOT_STICKY;
		} else {
			final String action = intent.getAction();
			if (ACTION_UPLOAD.equals(action)) {
				mUploads.execute(new UploadTask(intent));
			} else if (ACTION_DOWNLOAD.equals(action)) {
				mDownloads.execute(new DownloadTask(intent));
			} else if (ACTION_CANCEL_ALL.equals(action)) {
				Log.i(TAG, "Received request to cancel all transfers.");
				cancelUploads();
				cancelDownloads();
				stopSelf();
			}
		}
		
		return super.onStartCommand(intent, flags, startId);
	}

	@Override
	public void onDestroy() {
		Log.i(TAG, "Stopping Ubuntu One UpDownService.");
		
		UpDownServiceHelper.removePendingTransfers();
		
		if (mUploads.isRunning()) {
			cancelUploads();
		}
		if (mDownloads.isRunning()) {
			cancelDownloads();
		}
		
		final int uploadsFailed = mUploads.getTasksFailed();
		final int downloadsFailed = mDownloads.getTasksFailed();
		if (uploadsFailed + downloadsFailed > 0) {
			Alarms.maybeRegisterRetryFailedAlarm();
			Log.i(TAG, String.format("Shutting down. Failed up %d, down %d.",
					uploadsFailed, downloadsFailed));
		} else {
			Alarms.unregisterRetryFailedAlarm();
			Log.i(TAG, "Shutting down. All transfers completed.");
		}
		
		sIsUploadingPhotoMedia = false;
		sIsUploadingVideoMedia = false;
		sIsUploadingAudioMedia = false;
		sIsWorking = false;
		super.onDestroy();
	}

	@Override
	public IBinder onBind(Intent intent) {
		// Nothing to do. (yet)
		return null;
	}
	
	public static void cancelAll(Context context) {
		final Intent intent = new Intent(ACTION_CANCEL_ALL);
		context.stopService(intent);
	}

	public static boolean isWorking() {
		return sIsWorking;
	}

	public static void setIsUploadingPhotoMedia(boolean isUploading) {
		sIsUploadingPhotoMedia = isUploading;
	}

	public static boolean isUploadingPhotoMedia() {
		return sIsUploadingPhotoMedia;
	}
	
	public static void setIsUploadingVideoMedia(boolean isUploading) {
		sIsUploadingVideoMedia = isUploading;
	}

	public static boolean isUploadingVideoMedia() {
		return sIsUploadingVideoMedia;
	}
	
	public static void setIsUploadingAudioMedia(boolean isUploading) {
		sIsUploadingAudioMedia = isUploading;
	}

	public static boolean isUploadingAudioMedia() {
		return sIsUploadingAudioMedia;
	}

	public void cancelUploads() {
		final int canceledUploads = mUploads.cancelAll();
		Log.i(TAG, "Cancelled uploads count: " + canceledUploads);
		mUploads = new TransferExecutor(this, HttpPost.METHOD_NAME,
				mCompletionCallback);
	}

	public void cancelDownloads() {
		final int canceledDownloads = mDownloads.cancelAll();
		Log.i(TAG, "Cancelled downloads count: " + canceledDownloads);
		mDownloads = new TransferExecutor(this, HttpGet.METHOD_NAME,
				mCompletionCallback);
	}
	
	public void cancelAndInvalidateToken() {
		Log.w(TAG, "Cancelling all transfers and invalidating token.");
		Context context = UbuntuOneFiles.getInstance().getApplicationContext();
		Preferences.invalidateToken(context);
		cancelUploads();
		cancelDownloads();
		stopSelf();
	}

	public abstract class RunnableTransferTask implements Runnable {
		protected final ResultReceiver mReceiver;
		
		protected final Bundle mExtras;
		protected final long mSize;
		protected final String mResourcePath;
		protected final String mContentPath;
		protected Callback mProgress;
		protected Exception mException;
		
		public RunnableTransferTask(Intent intent) {
			this.mReceiver = (ResultReceiver) intent
					.getParcelableExtra(EXTRA_CALLBACK);
			this.mExtras = intent.getExtras();
			this.mSize = mExtras.getLong(EXTRA_SIZE);
			this.mResourcePath = intent.getStringExtra(EXTRA_RESOURCE_PATH);
			this.mContentPath = intent.getStringExtra(EXTRA_CONTENT_PATH);
		}
		
		public long getSize() {
			return mSize;
		}
		
		public void setProgressCallback(Callback progress) {
			this.mProgress = progress;
		}
		
		protected abstract void onPreTransfer();
		
		protected abstract void onTransfer() throws Exception;
		
		protected abstract void onPostTransfer() throws Exception;
		
		protected abstract void onFailedTransfer(Exception e);
		
		public final void run() {
			boolean restart = true;
			for (int i = 3; i > 0 && restart; i--) {
				try {
					onPreTransfer();
					onTransfer();
					onPostTransfer();
					if (mReceiver != null) {
						mReceiver.send(Status.FINISHED, mExtras);
					}
					// Task completed, clear any exception, don't restart.
					mException = null;
					restart = false;
				} catch (Exception e) {
					// Task failed, save exception, assume not restarting.
					mException = e;
					restart = false;
					
					// Allow retries on given exception classes.
					if (e instanceof SocketTimeoutException) {
						restart = true;
					}
				}
			}
			
			final Exception e = mException;
			if (e != null) {
				onFailedTransfer(e);
				if (mReceiver != null) {
					// Control what exceptions should appear on UI.
					if (mException instanceof SocketException ||
							mException instanceof UnknownHostException) {
						mExtras.putString(EXTRA_ERROR, "Lost network connection.");
						mReceiver.send(Status.ERROR, mExtras);
					} else if (mException instanceof NotFoundException) {
						mExtras.putString(EXTRA_ERROR, "File not found.");
						mReceiver.send(Status.ERROR, mExtras);
					} else if (mException instanceof InternalServerErrorException) {
						// L8R karni: add the Oops-Id header to extras for submition?
						mExtras.putString(EXTRA_ERROR, "Oops, something went wrong.");
						mReceiver.send(Status.ERROR, mExtras);
						
						// Don't retry transfers that returned HTTP 500
						if (this instanceof UploadTask) {
							MetaUtilities.deleteByResourcePath(mResourcePath);
						} else if (this instanceof DownloadTask) {
							MetaUtilities.setStateAndData(mResourcePath, null, null);
						}
					} else if (mException instanceof UnauthorizedException) {
						cancelAndInvalidateToken();
						mExtras.putString(EXTRA_ERROR, mException.getMessage());
						mReceiver.send(Status.ERROR, mExtras);
					} else if (mException.getMessage() != null) {
						mExtras.putString(EXTRA_ERROR, mException.getMessage());
						mReceiver.send(Status.ERROR, mExtras);
					}
				}
				if (!NetworkReceiver.isConnected()) {
					stopSelf();
				}
			}
			
		}
	}

	public class DownloadTask extends RunnableTransferTask {
		private final String mDataUri;
		private final String mPartUri;
		private File mPartialFile;
		private File mCompleteFile;
		
		public DownloadTask(Intent intent) {
			super(intent);
			mDataUri = intent.getStringExtra(EXTRA_DATA);
			mPartUri = mDataUri.concat(PARTIAL_SUFFIX);
		}
		
		protected void onPreTransfer() {
			Log.d(TAG, "Starting download task: " + mResourcePath);
		
			// Update resource state in content provider.
			MetaUtilities.setState(mResourcePath, ResourceState.STATE_GETTING);
			MetaUtilities.notifyChange(Nodes.CONTENT_URI);
			
			// Setup partial and complete File objects.
			final URI partURI = URI.create(Uri.encode(mPartUri, ":/"));
			mPartialFile = new File(partURI);
			final URI fullURI = URI.create(Uri.encode(mDataUri, ":/"));
			mCompleteFile = new File(fullURI);
		}
		
		protected void onTransfer() throws OAuthMessageSignerException,
				OAuthExpectationFailedException, OAuthCommunicationException,
				ClientProtocolException, IOException, JSONException,
				ParseException, BadRequestException, UnauthorizedException,
				ForbiddenException, NotFoundException,
				MethodNotAllowedException, RequestTimeOutException,
				ExpectationFailedException, InternalServerErrorException,
				NotImplementedException, ServiceUnavailableException, Exception {
			
			final File parentDirectory = mCompleteFile.getParentFile();
			if (!parentDirectory.exists()) {
				Log.d(TAG, "Creating directory: " + parentDirectory);
				final boolean created = parentDirectory.mkdirs();
				if (!created) {
					throw new Exception("Can't create target directory: "
							+ parentDirectory);
				}
			}
			
			// Download the file.
			RestApi.getFileContent(mHttpClient, mConsumer,
					mProgress, mContentPath, mPartUri, null);
		}
		
		protected void onPostTransfer() throws IOException {
			// Commit the partial file.
			if (mCompleteFile.exists()) {
				final boolean deleted = mCompleteFile.delete();
				if (!deleted) {
					throw new IOException("Can't delete old file: "
							+ mCompleteFile);
				}
			}
			final boolean renamed = mPartialFile.renameTo(mCompleteFile);
			if (!renamed) {
				throw new IOException("Can't rename partial file: "
						+ mPartialFile);
			}
			
			// Update content provider to reflect state.
			MetaUtilities.setStateAndData(mResourcePath, null, mDataUri);
			MetaUtilities.setIsCached(mResourcePath, true);
			MetaUtilities.notifyChange(Nodes.CONTENT_URI);
			
			MediaScannerConnection.scanFile(UbuntuOneFiles.getInstance(),
					new String[] { mCompleteFile.getPath() }, null, null);
			
			Log.d(TAG, "Finished download task: " + mResourcePath);
		}
		
		protected void onFailedTransfer(Exception e) {
			mException = e;
			
			// Remove partial file.
			if (mPartialFile != null && mPartialFile.exists()) {
				final boolean deleted = mPartialFile.delete();
				if (!deleted) {
					Log.e(TAG, "Can't delete partial file: " + mPartialFile);
				}
			}
			
			// Update content provider to reflect failed state.
			MetaUtilities.setIsCached(mResourcePath, false);
			MetaUtilities.setState(mResourcePath,
					ResourceState.STATE_GETTING_FAILED);
			MetaUtilities.notifyChange(Nodes.CONTENT_URI);
			
			Log.e(TAG, "Download task failed: " + mResourcePath, e);
		}
	}
	
	private class UploadTask extends RunnableTransferTask {
		private final String mDataUri;
		private final String mMime;
		
		public UploadTask(Intent intent) {
			super(intent);
			mDataUri = intent.getStringExtra(EXTRA_DATA);
			mMime = mExtras.getString(EXTRA_MIME);
		}

		@Override
		protected void onPreTransfer() {
			Log.d(TAG, "Starting upload task: " + mResourcePath);
			
			// Update resource state in content provider.
			MetaUtilities.setState(mResourcePath, ResourceState.STATE_POSTING);
			MetaUtilities.notifyChange(Nodes.CONTENT_URI);
		}

		@Override
		protected void onTransfer() throws Exception {
			if (mDataUri == null) {
				throw new FileNotFoundException("Unknown upload source:" + mDataUri);
			}
			
			boolean isFile =
					mDataUri.startsWith(ContentResolver.SCHEME_FILE);
			boolean isContent =
					mDataUri.startsWith(ContentResolver.SCHEME_CONTENT);
			if (!isFile && !isContent) {
				throw new FileNotFoundException(
						"Requested upload of unknown uri type: " + mDataUri);
			}
			Log.d(TAG, "Uploading from uri: " + mDataUri);
			
			// TODO karni: We NEED to support content:// URIs !
			// Fix ubuntuone-files-java-client to accept InputStream.
			// InputStream in = getContentResolver().openInputStream(uri);

			final NodeInfo nodeInfo = RestApi.putFileContent(
					mHttpClient, mConsumer, mProgress, mDataUri,
					mMime, mSize, mContentPath);
			
			if (nodeInfo != null) {
				// L8R Handle *this* nodeInfo instead of calling refresh. Then, make receiver private.
			}
			MetaServiceHelper.refreshNode(getApplicationContext(),
					mReceiver, mResourcePath);
		}

		@Override
		protected void onPostTransfer() throws Exception {
			// Update content provider to reflect state.
			MetaUtilities.setState(mResourcePath, null);
			MetaUtilities.setIsCached(mResourcePath, true);
			MetaUtilities.notifyChange(Nodes.CONTENT_URI);
			
			Log.d(TAG, "Finished upload task: " + mResourcePath);
		}

		@Override
		protected void onFailedTransfer(Exception e) {
			this.mException = e;
			
			// Update content provider to reflect failed state.
			MetaUtilities.setIsCached(mResourcePath, false);
			MetaUtilities.setState(mResourcePath,
					ResourceState.STATE_POSTING_FAILED);
			MetaUtilities.notifyChange(Nodes.CONTENT_URI);
			
			Log.e(TAG, "Upload task failed: " + mResourcePath, e);
		}
	}
}
